/* SPDX-FileCopyrightText: 2013 Blender Authors
 *
 * SPDX-License-Identifier: GPL-2.0-or-later */

/** \file
 * \ingroup depsgraph
 *
 * Public API for Querying Depsgraph.
 */

#pragma once

#include "BLI_function_ref.hh"
#include "BLI_iterator.h"
#include "BLI_set.hh"
#include "BLI_utildefines.h"

#include "DEG_depsgraph.hh"
#include "DEG_depsgraph_build.hh"

/* Needed for the instance iterator. */
#include "DNA_object_types.h"

#include "BKE_duplilist.hh"
#include "BKE_object_types.hh"

struct BLI_Iterator;
struct CustomData_MeshMasks;
struct Depsgraph;
struct DupliObject;
struct ID;
struct ListBase;
struct PointerRNA;
struct Scene;
struct ViewLayer;
struct ViewerPath;

/* -------------------------------------------------------------------- */
/** \name DEG input data
 * \{ */

/** Get scene that depsgraph was built for. */
Scene *DEG_get_input_scene(const Depsgraph *graph);

/** Get view layer that depsgraph was built for. */
ViewLayer *DEG_get_input_view_layer(const Depsgraph *graph);

/** Get bmain that depsgraph was built for. */
Main *DEG_get_bmain(const Depsgraph *graph);

/** Get evaluation mode that depsgraph was built for. */
eEvaluationMode DEG_get_mode(const Depsgraph *graph);

/** Get time that depsgraph is being evaluated or was last evaluated at. */
float DEG_get_ctime(const Depsgraph *graph);

/** \} */

/* -------------------------------------------------------------------- */
/** \name DEG evaluated data
 * \{ */

/** Check if given ID type was tagged for update. */
bool DEG_id_type_updated(const Depsgraph *depsgraph, short id_type);
bool DEG_id_type_any_updated(const Depsgraph *depsgraph);

/** Check if given ID type is present in the depsgraph */
bool DEG_id_type_any_exists(const Depsgraph *depsgraph, short id_type);

/** Get additional evaluation flags for the given ID. */
uint32_t DEG_get_eval_flags_for_id(const Depsgraph *graph, const ID *id);

/** Get additional mesh CustomData_MeshMasks flags for the given object. */
void DEG_get_customdata_mask_for_object(const Depsgraph *graph,
                                        Object *object,
                                        CustomData_MeshMasks *r_mask);

/**
 * Get scene at its evaluated state.
 *
 * Technically, this is a copied-on-written and fully evaluated version of the input scene.
 * This function will check that the data-block has been expanded (and copied) from the original
 * one. Assert will happen if it's not.
 */
Scene *DEG_get_evaluated_scene(const Depsgraph *graph);

/**
 * Get view layer at its evaluated state.
 * This is a shortcut for accessing active view layer from evaluated scene.
 */
ViewLayer *DEG_get_evaluated_view_layer(const Depsgraph *graph);

/** Get evaluated version of given ID data-block. */
ID *DEG_get_evaluated_id(const Depsgraph *depsgraph, ID *id);
const ID *DEG_get_evaluated_id(const Depsgraph *depsgraph, const ID *id);

template<typename T> T *DEG_get_evaluated(const Depsgraph *depsgraph, T *id)
{
  static_assert(blender::dna::is_ID_v<T>);
  return reinterpret_cast<T *>(DEG_get_evaluated_id(depsgraph, reinterpret_cast<ID *>(id)));
}

template<typename T> const T *DEG_get_evaluated(const Depsgraph *depsgraph, const T *id)
{
  static_assert(blender::dna::is_ID_v<T>);
  return reinterpret_cast<const T *>(
      DEG_get_evaluated_id(depsgraph, reinterpret_cast<const ID *>(id)));
}

/** Get evaluated version of data pointed to by RNA pointer */
void DEG_get_evaluated_rna_pointer(const Depsgraph *depsgraph,
                                   PointerRNA *ptr,
                                   PointerRNA *r_ptr_eval);

/** Get original version of given evaluated ID data-block. */
ID *DEG_get_original_id(ID *id);
const ID *DEG_get_original_id(const ID *id);

template<typename T> T *DEG_get_original(T *id)
{
  static_assert(blender::dna::is_ID_v<T>);
  return reinterpret_cast<T *>(DEG_get_original_id(reinterpret_cast<ID *>(id)));
}

template<typename T> const T *DEG_get_original(const T *id)
{
  static_assert(blender::dna::is_ID_v<T>);
  return reinterpret_cast<const T *>(DEG_get_original_id(reinterpret_cast<const ID *>(id)));
}

/**
 * Get the depsgraph that owns the given ID. This is efficient because the depsgraph is cached on
 * the ID.
 *
 * Only IDs that use the copy-on-eval mechanism of depsgraph evaluation have a corresponding
 * depsgraph. Original IDs as well as temporary IDs e.g. generated by geometry nodes do not have a
 * corresponding depsgraph, so null is returned here for these IDs.
 */
Depsgraph *DEG_get_depsgraph_by_id(const ID &id);

/**
 * Check whether given ID is an original.
 *
 * Original IDs are considered all the IDs which are not covered by copy-on-evaluation system and
 * are not out-of-main localized data-blocks.
 */
bool DEG_is_original_id(const ID *id);

template<typename T> bool DEG_is_original(const T *id)
{
  static_assert(blender::dna::is_ID_v<T>);
  return DEG_is_original_id(reinterpret_cast<const ID *>(id));
}

/* Opposite of the above (`DEG_is_original_*`).
 *
 * If the data-block is not original it must be evaluated, and vice versa. */

bool DEG_is_evaluated_id(const ID *id);

template<typename T> bool DEG_is_evaluated(const T *id)
{
  static_assert(blender::dna::is_ID_v<T>);
  return DEG_is_evaluated_id(reinterpret_cast<const ID *>(id));
}

/**
 * Check whether depsgraph is fully evaluated. This includes the following checks:
 * - Relations are up-to-date.
 * - Nothing is tagged for update.
 */
bool DEG_is_fully_evaluated(const Depsgraph *depsgraph);

/**
 * Check every component of the data-block is evaluated. For example, an object disabled in the
 * viewport is not fully evaluated, even though the copy-on-eval data-block is created.
 */
bool DEG_id_is_fully_evaluated(const Depsgraph *depsgraph, const ID *id_eval);

/**
 * Returns false when the objects geometry is not fully evaluated in its depsgraph yet. In this
 * case, the geometry must not be accessed. Otherwise returns true when geometry is fully evaluated
 * or the object does not belong to any specific depsgraph.
 *
 * The result of this function is non deterministic when multi-threading is used because the
 * depsgraph nodes are not totally ordered. When the depsgraph contains all correct relations and
 * there are no cycles, the result should always be `true` here though. So it does not break
 * determinism when there are no dependency graph cycles.
 */
bool DEG_object_geometry_is_evaluated(const Object &object);

/**
 *  Same as #DEG_object_geometry_is_evaluated
 *  but for the transformation of the object.
 */
bool DEG_object_transform_is_evaluated(const Object &object);

/**
 * Same as #DEG_object_geometry_is_evaluated
 * but for the geometry of all objects in the collection.
 */
bool DEG_collection_geometry_is_evaluated(const Collection &collection);

/** \} */

/* -------------------------------------------------------------------- */
/** \name DEG object iterators
 * \{ */

enum DegIterFlag {
  DEG_ITER_OBJECT_FLAG_LINKED_DIRECTLY = (1 << 0),
  DEG_ITER_OBJECT_FLAG_LINKED_INDIRECTLY = (1 << 1),
  DEG_ITER_OBJECT_FLAG_LINKED_VIA_SET = (1 << 2),
  DEG_ITER_OBJECT_FLAG_VISIBLE = (1 << 3),
  DEG_ITER_OBJECT_FLAG_DUPLI = (1 << 4),
};
ENUM_OPERATORS(DegIterFlag, DEG_ITER_OBJECT_FLAG_DUPLI)

struct DEGObjectIterSettings {
  Depsgraph *depsgraph;
  /**
   * Bit-field of the #DegIterFlag.
   *
   * NOTE: Be careful with #DEG_ITER_OBJECT_FLAG_LINKED_INDIRECTLY objects.
   * Although they are available they have no overrides (collection_properties)
   * and will crash if you try to access it.
   */
  uint32_t flags;

  /**
   * When set, the final evaluated geometry of the corresponding object is omitted. Instead the
   * geometry for the viewer path included in the iterator.
   */
  const ViewerPath *viewer_path;

  /**
   * If not empty, the iterator should only return objects that are in this list (or their
   * instances are in it). Pointers in this span should be the original data-block.
   */
  blender::Set<const Object *> *included_objects;
};

/**
 * Flags to get objects for draw manager and final render.
 */
#define DEG_OBJECT_ITER_FOR_RENDER_ENGINE_FLAGS \
  DEG_ITER_OBJECT_FLAG_LINKED_DIRECTLY | DEG_ITER_OBJECT_FLAG_LINKED_VIA_SET | \
      DEG_ITER_OBJECT_FLAG_VISIBLE | DEG_ITER_OBJECT_FLAG_DUPLI

struct DEGObjectIterData {
  DEGObjectIterSettings *settings;
  Depsgraph *graph;
  int flag;

  Scene *scene;

  eEvaluationMode eval_mode;

  /** Object whose preview instead of evaluated geometry should be part of the iterator. */
  Object *object_orig_with_preview;

  Object *next_object;

  /* **** Iteration over dupli-list. *** */

  /* Object which created the dupli-list. */
  Object *dupli_parent;
  /* List of duplicated objects. */
  DupliList dupli_list;
  /* Next duplicated object to step into. */
  DupliObject *dupli_object_next;
  /* The dupli_list index of dupli_object_next. */
  int dupli_object_next_index;
  /* Corresponds to current object: current iterator object is evaluated from
   * this duplicated object. */
  DupliObject *dupli_object_current;
  /* Temporary storage to report fully populated DNA to the render engine or
   * other users of the iterator. */
  Object temp_dupli_object;
  blender::bke::ObjectRuntime temp_dupli_object_runtime;

  /* **** Iteration over ID nodes **** */
  size_t id_node_index;
  size_t num_id_nodes;

  /* Copy the current/next data and move the DupliList. */
  void transfer_from(DEGObjectIterData &other);
};

void DEG_iterator_objects_begin(BLI_Iterator *iter, DEGObjectIterData *data);
void DEG_iterator_objects_next(BLI_Iterator *iter);
void DEG_iterator_objects_end(BLI_Iterator *iter);

#define DEG_OBJECT_ITER_BEGIN(settings_, instance_) \
  { \
    DEGObjectIterData data_ = { \
        (settings_), \
        (settings_)->depsgraph, \
        (int)(settings_)->flags, \
    }; \
\
    ITER_BEGIN (DEG_iterator_objects_begin, \
                DEG_iterator_objects_next, \
                DEG_iterator_objects_end, \
                &data_, \
                Object *, \
                instance_)

#define DEG_OBJECT_ITER_END \
  ITER_END; \
  } \
  ((void)0)

/** \} */

/* -------------------------------------------------------------------- */
/** \name DEG object iterators - Manual dupli iteration
 * Helper functions for handling dupli instances iteration manually (ie. without passing
 * DEG_ITER_OBJECT_FLAG_DUPLI to DEGObjectIterSettings::flags)
 * \{ */

/**
 * Returns true if the object should be visible on the given context.
 */
bool DEG_iterator_object_is_visible(eEvaluationMode eval_mode, const Object *ob);

/**
 * Returns true if the dupli instance should be visible on the given context.
 */
bool DEG_iterator_dupli_is_visible(const DupliObject *dupli, eEvaluationMode eval_mode);

namespace evil {
/**
 * WARNING: DON'T USE!!!
 *
 * These functions are exposed publicly as a temporary measure while we figure out how to fully get
 * rid of temporary objects in the Draw module (See #144811).
 *
 * DON'T ADD NEW USE CASES FOR THESE FUNCTIONS.
 */

/**
 * WARNING: DON'T USE!!!
 * Generates a temporary object for a given dupli instance.
 *
 * Returns true if the resulting object should be visible, otherwise the temp object should be
 * considered invalid.
 * NOTE: DEG_iterator_temp_object_free_properties should be called regardless.
 *
 * \param do_matrix_setup: If false, the temp_object won't have valid
 * object_to_world/world_to_object matrices, and the OB_NEG_SCALE flag will never be set.
 */
[[nodiscard]] bool DEG_iterator_temp_object_from_dupli(const Object *dupli_parent,
                                                       const DupliObject *dupli,
                                                       eEvaluationMode eval_mode,
                                                       bool do_matrix_setup,
                                                       Object *r_temp_object,
                                                       ObjectRuntimeHandle *r_temp_runtime);

/**
 * WARNING: DON'T USE!!!
 * Frees any property allocated when calling DEG_iterator_temp_object_from_dupli.
 */
void DEG_iterator_temp_object_free_properties(const DupliObject *dupli, Object *temp_object);
}  // namespace evil

/** \} */

/* -------------------------------------------------------------------- */
/** \name DEG ID iterators
 * \{ */

struct DEGIDIterData {
  Depsgraph *graph;
  bool only_updated;

  size_t id_node_index;
  size_t num_id_nodes;
};

void DEG_iterator_ids_begin(BLI_Iterator *iter, DEGIDIterData *data);
void DEG_iterator_ids_next(BLI_Iterator *iter);
void DEG_iterator_ids_end(BLI_Iterator *iter);

/** \} */

/* -------------------------------------------------------------------- */
/** \name DEG traversal
 * \{ */

using DEGForeachIDCallback = blender::FunctionRef<void(ID *id)>;
using DEGForeachIDComponentCallback =
    blender::FunctionRef<void(ID *id, eDepsObjectComponentType component)>;

/**
 * \note Modifies runtime flags in depsgraph nodes,
 * so can not be used in parallel. Keep an eye on that!
 */
void DEG_foreach_ancestor_ID(const Depsgraph *depsgraph,
                             const ID *id,
                             DEGForeachIDCallback callback);
void DEG_foreach_dependent_ID(const Depsgraph *depsgraph,
                              const ID *id,
                              DEGForeachIDCallback callback);

/**
 * Starts traversal from given component of the given ID, invokes callback for every other
 * component which is directly on indirectly dependent on the source one.
 */
enum {
  /* Ignore transform solvers which depends on multiple inputs and affects final transform.
   * Is used for cases like snapping objects which are part of a rigid body simulation:
   * without this there will be "false-positive" dependencies between transform components of
   * objects:
   *
   *     object 1 transform before solver ---> solver ------> object 1 final transform
   *     object 2 transform before solver -----^     \------> object 2 final transform
   */
  DEG_FOREACH_COMPONENT_IGNORE_TRANSFORM_SOLVERS = (1 << 0),
};
void DEG_foreach_dependent_ID_component(const Depsgraph *depsgraph,
                                        const ID *id,
                                        eDepsObjectComponentType source_component_type,
                                        int flags,
                                        DEGForeachIDComponentCallback callback);

void DEG_foreach_ID(const Depsgraph *depsgraph, DEGForeachIDCallback callback);

/** \} */
