/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef nsIWidgetListener_h__
#define nsIWidgetListener_h__

#include <stdint.h>

#include "mozilla/EventForwards.h"
#include "mozilla/layers/LayersTypes.h"
#include "mozilla/TimeStamp.h"

#include "nsRegionFwd.h"
#include "Units.h"

class nsView;
class nsIWidget;
class nsIAppWindow;
class nsMenuPopupFrame;

namespace mozilla {
class PresShell;
class PresShellWidgetListener;
}  // namespace mozilla

/**
 * sizemode is an adjunct to widget size
 */
enum nsSizeMode {
  nsSizeMode_Normal = 0,
  nsSizeMode_Minimized,
  nsSizeMode_Maximized,
  nsSizeMode_Fullscreen,
  nsSizeMode_Invalid
};

class nsIWidgetListener {
 public:
  /**
   * If this listener is for an nsIAppWindow, return it. If this is null, then
   * this is likely a listener for a popup or a pres shell.
   */
  virtual nsIAppWindow* GetAppWindow() { return nullptr; }

  /** If this listener is for a pres shell, return it. */
  virtual mozilla::PresShellWidgetListener* GetAsPresShellWidgetListener() {
    return nullptr;
  }

  /** If this listener is for an nsMenuPopupFrame, return it. */
  virtual nsMenuPopupFrame* GetAsMenuPopupFrame() { return nullptr; }

  /** Return the presshell for this widget listener. */
  virtual mozilla::PresShell* GetPresShell() { return nullptr; }

  /**
   * Called when a window is moved to location (x, y). Returns true if the
   * notification was handled. Coordinates are outer window screen coordinates.
   */
  enum class ByMoveToRect : bool { No, Yes };
  virtual bool WindowMoved(nsIWidget* aWidget, int32_t aX, int32_t aY,
                           ByMoveToRect) {
    return false;
  }

  /**
   * Called when a window is resized to (width, height). Returns true if the
   * notification was handled. Coordinates are outer window screen coordinates.
   */
  virtual bool WindowResized(nsIWidget* aWidget, int32_t aWidth,
                             int32_t aHeight) {
    return false;
  }

  /**
   * Called when the size mode (minimized, maximized, fullscreen) is changed.
   */
  virtual void SizeModeChanged(nsSizeMode aSizeMode) {}

  virtual void DynamicToolbarMaxHeightChanged(mozilla::ScreenIntCoord aHeight) {
  }
  virtual void DynamicToolbarOffsetChanged(mozilla::ScreenIntCoord aOffset) {}
  /** Called when the software keyboard appears/disappears. */
  virtual void KeyboardHeightChanged(mozilla::ScreenIntCoord aHeight) {}
  virtual void AndroidPipModeChanged(bool) {}

  /** Called when the macOS titlebar is shown while in fullscreen. */
  virtual void MacFullscreenMenubarOverlapChanged(
      mozilla::DesktopCoord aOverlapAmount) {}

  /**
   * Called when the occlusion state is changed.
   */
  virtual void OcclusionStateChanged(bool aIsFullyOccluded) {}

  /** Called when the window is activated and focused. */
  virtual void WindowActivated() {}

  /** Called when the window is deactivated and no longer focused. */
  virtual void WindowDeactivated() {}

  /**
   * Called when the show/hide toolbar button on the Mac titlebar is pressed.
   */
  virtual void OSToolbarButtonPressed() {}

  /**
   * Called when a request is made to close the window. Returns true if the
   * notification was handled. Returns true if the notification was handled.
   */
  virtual bool RequestWindowClose(nsIWidget* aWidget) { return false; }

  /** Paint the window if needed. */
  MOZ_CAN_RUN_SCRIPT_BOUNDARY
  virtual void PaintWindow(nsIWidget* aWidget) {}

  virtual void DidCompositeWindow(mozilla::layers::TransactionId aTransactionId,
                                  const mozilla::TimeStamp& aCompositeStart,
                                  const mozilla::TimeStamp& aCompositeEnd) {}

  /**
   * Returns true if this is a popup that should not be visible. If this
   * is a popup that is visible, not a popup or this state is unknown,
   * returns false.
   */
  virtual bool ShouldNotBeVisible() { return false; }

  /** Returns true if painting should be suppressed for this listener */
  virtual bool IsPaintSuppressed() const { return false; }

  /** Handle an event. */
  virtual nsEventStatus HandleEvent(mozilla::WidgetGUIEvent* aEvent) {
    return nsEventStatus_eIgnore;
  }

  /** Called when safe area insets are changed. */
  virtual void SafeAreaInsetsChanged(
      const mozilla::LayoutDeviceIntMargin& aSafeAreaInsets) {}
};

#endif
