{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Allows drawing with OpenGL.
-- 
-- \<picture>
--   \<source srcset=\"glarea-dark.png\" media=\"(prefers-color-scheme: dark)\">
--   \<img alt=\"An example GtkGLArea\" src=\"glarea.png\">
-- \<\/picture>
-- 
-- @GtkGLArea@ sets up its own t'GI.Gdk.Objects.GLContext.GLContext', and creates a custom
-- GL framebuffer that the widget will do GL rendering onto. It also ensures
-- that this framebuffer is the default GL rendering target when rendering.
-- The completed rendering is integrated into the larger GTK scene graph as
-- a texture.
-- 
-- In order to draw, you have to connect to the [GLArea::render]("GI.Gtk.Objects.GLArea#g:signal:render")
-- signal, or subclass @GtkGLArea@ and override the GtkGLAreaClass.render
-- virtual function.
-- 
-- The @GtkGLArea@ widget ensures that the @GdkGLContext@ is associated with
-- the widget\'s drawing area, and it is kept updated when the size and
-- position of the drawing area changes.
-- 
-- == Drawing with GtkGLArea
-- 
-- The simplest way to draw using OpenGL commands in a @GtkGLArea@ is to
-- create a widget instance and connect to the [GLArea::render]("GI.Gtk.Objects.GLArea#g:signal:render") signal:
-- 
-- The @render()@ function will be called when the @GtkGLArea@ is ready
-- for you to draw its content:
-- 
-- The initial contents of the framebuffer are transparent.
-- 
-- 
-- === /c code/
-- >static gboolean
-- >render (GtkGLArea *area, GdkGLContext *context)
-- >{
-- >  // inside this function it's safe to use GL; the given
-- >  // GdkGLContext has been made current to the drawable
-- >  // surface used by the `GtkGLArea` and the viewport has
-- >  // already been set to be the size of the allocation
-- >
-- >  // we can start by clearing the buffer
-- >  glClearColor (0, 0, 0, 0);
-- >  glClear (GL_COLOR_BUFFER_BIT);
-- >
-- >  // record the active framebuffer ID, so we can return to it
-- >  // with `glBindFramebuffer (GL_FRAMEBUFFER, screen_fb)` should
-- >  // we, for instance, intend on utilizing the results of an
-- >  // intermediate render texture pass
-- >  GLuint screen_fb = 0;
-- >  glGetIntegerv (GL_FRAMEBUFFER_BINDING, &screen_fb);
-- >
-- >  // draw your object
-- >  // draw_an_object ();
-- >
-- >  // we completed our drawing; the draw commands will be
-- >  // flushed at the end of the signal emission chain, and
-- >  // the buffers will be drawn on the window
-- >  return TRUE;
-- >}
-- >
-- >void setup_glarea (void)
-- >{
-- >  // create a GtkGLArea instance
-- >  GtkWidget *gl_area = gtk_gl_area_new ();
-- >
-- >  // connect to the "render" signal
-- >  g_signal_connect (gl_area, "render", G_CALLBACK (render), NULL);
-- >}
-- 
-- 
-- If you need to initialize OpenGL state, e.g. buffer objects or
-- shaders, you should use the [Widget::realize]("GI.Gtk.Objects.Widget#g:signal:realize") signal;
-- you can use the [Widget::unrealize]("GI.Gtk.Objects.Widget#g:signal:unrealize") signal to clean up.
-- Since the @GdkGLContext@ creation and initialization may fail, you
-- will need to check for errors, using 'GI.Gtk.Objects.GLArea.gLAreaGetError'.
-- 
-- An example of how to safely initialize the GL state is:
-- 
-- 
-- === /c code/
-- >static void
-- >on_realize (GtkGLArea *area)
-- >{
-- >  // We need to make the context current if we want to
-- >  // call GL API
-- >  gtk_gl_area_make_current (area);
-- >
-- >  // If there were errors during the initialization or
-- >  // when trying to make the context current, this
-- >  // function will return a GError for you to catch
-- >  if (gtk_gl_area_get_error (area) != NULL)
-- >    return;
-- >
-- >  // You can also use gtk_gl_area_set_error() in order
-- >  // to show eventual initialization errors on the
-- >  // GtkGLArea widget itself
-- >  GError *internal_error = NULL;
-- >  init_buffer_objects (&error);
-- >  if (error != NULL)
-- >    {
-- >      gtk_gl_area_set_error (area, error);
-- >      g_error_free (error);
-- >      return;
-- >    }
-- >
-- >  init_shaders (&error);
-- >  if (error != NULL)
-- >    {
-- >      gtk_gl_area_set_error (area, error);
-- >      g_error_free (error);
-- >      return;
-- >    }
-- >}
-- 
-- 
-- If you need to change the options for creating the @GdkGLContext@
-- you should use the [GLArea::createContext]("GI.Gtk.Objects.GLArea#g:signal:createContext") signal.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.GLArea
    ( 

-- * Exported types
    GLArea(..)                              ,
    IsGLArea                                ,
    toGLArea                                ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [actionSetEnabled]("GI.Gtk.Objects.Widget#g:method:actionSetEnabled"), [activate]("GI.Gtk.Objects.Widget#g:method:activate"), [activateAction]("GI.Gtk.Objects.Widget#g:method:activateAction"), [activateDefault]("GI.Gtk.Objects.Widget#g:method:activateDefault"), [addController]("GI.Gtk.Objects.Widget#g:method:addController"), [addCssClass]("GI.Gtk.Objects.Widget#g:method:addCssClass"), [addMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:addMnemonicLabel"), [addTickCallback]("GI.Gtk.Objects.Widget#g:method:addTickCallback"), [allocate]("GI.Gtk.Objects.Widget#g:method:allocate"), [announce]("GI.Gtk.Interfaces.Accessible#g:method:announce"), [attachBuffers]("GI.Gtk.Objects.GLArea#g:method:attachBuffers"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [childFocus]("GI.Gtk.Objects.Widget#g:method:childFocus"), [computeBounds]("GI.Gtk.Objects.Widget#g:method:computeBounds"), [computeExpand]("GI.Gtk.Objects.Widget#g:method:computeExpand"), [computePoint]("GI.Gtk.Objects.Widget#g:method:computePoint"), [computeTransform]("GI.Gtk.Objects.Widget#g:method:computeTransform"), [contains]("GI.Gtk.Objects.Widget#g:method:contains"), [createPangoContext]("GI.Gtk.Objects.Widget#g:method:createPangoContext"), [createPangoLayout]("GI.Gtk.Objects.Widget#g:method:createPangoLayout"), [disposeTemplate]("GI.Gtk.Objects.Widget#g:method:disposeTemplate"), [dragCheckThreshold]("GI.Gtk.Objects.Widget#g:method:dragCheckThreshold"), [errorBell]("GI.Gtk.Objects.Widget#g:method:errorBell"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [grabFocus]("GI.Gtk.Objects.Widget#g:method:grabFocus"), [hasCssClass]("GI.Gtk.Objects.Widget#g:method:hasCssClass"), [hasDefault]("GI.Gtk.Objects.Widget#g:method:hasDefault"), [hasFocus]("GI.Gtk.Objects.Widget#g:method:hasFocus"), [hasVisibleFocus]("GI.Gtk.Objects.Widget#g:method:hasVisibleFocus"), [hide]("GI.Gtk.Objects.Widget#g:method:hide"), [inDestruction]("GI.Gtk.Objects.Widget#g:method:inDestruction"), [initTemplate]("GI.Gtk.Objects.Widget#g:method:initTemplate"), [insertActionGroup]("GI.Gtk.Objects.Widget#g:method:insertActionGroup"), [insertAfter]("GI.Gtk.Objects.Widget#g:method:insertAfter"), [insertBefore]("GI.Gtk.Objects.Widget#g:method:insertBefore"), [isAncestor]("GI.Gtk.Objects.Widget#g:method:isAncestor"), [isDrawable]("GI.Gtk.Objects.Widget#g:method:isDrawable"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isFocus]("GI.Gtk.Objects.Widget#g:method:isFocus"), [isSensitive]("GI.Gtk.Objects.Widget#g:method:isSensitive"), [isVisible]("GI.Gtk.Objects.Widget#g:method:isVisible"), [keynavFailed]("GI.Gtk.Objects.Widget#g:method:keynavFailed"), [listMnemonicLabels]("GI.Gtk.Objects.Widget#g:method:listMnemonicLabels"), [makeCurrent]("GI.Gtk.Objects.GLArea#g:method:makeCurrent"), [map]("GI.Gtk.Objects.Widget#g:method:map"), [measure]("GI.Gtk.Objects.Widget#g:method:measure"), [mnemonicActivate]("GI.Gtk.Objects.Widget#g:method:mnemonicActivate"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [observeChildren]("GI.Gtk.Objects.Widget#g:method:observeChildren"), [observeControllers]("GI.Gtk.Objects.Widget#g:method:observeControllers"), [pick]("GI.Gtk.Objects.Widget#g:method:pick"), [queueAllocate]("GI.Gtk.Objects.Widget#g:method:queueAllocate"), [queueDraw]("GI.Gtk.Objects.Widget#g:method:queueDraw"), [queueRender]("GI.Gtk.Objects.GLArea#g:method:queueRender"), [queueResize]("GI.Gtk.Objects.Widget#g:method:queueResize"), [realize]("GI.Gtk.Objects.Widget#g:method:realize"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeController]("GI.Gtk.Objects.Widget#g:method:removeController"), [removeCssClass]("GI.Gtk.Objects.Widget#g:method:removeCssClass"), [removeMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:removeMnemonicLabel"), [removeTickCallback]("GI.Gtk.Objects.Widget#g:method:removeTickCallback"), [resetProperty]("GI.Gtk.Interfaces.Accessible#g:method:resetProperty"), [resetRelation]("GI.Gtk.Interfaces.Accessible#g:method:resetRelation"), [resetState]("GI.Gtk.Interfaces.Accessible#g:method:resetState"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [shouldLayout]("GI.Gtk.Objects.Widget#g:method:shouldLayout"), [show]("GI.Gtk.Objects.Widget#g:method:show"), [sizeAllocate]("GI.Gtk.Objects.Widget#g:method:sizeAllocate"), [snapshotChild]("GI.Gtk.Objects.Widget#g:method:snapshotChild"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [translateCoordinates]("GI.Gtk.Objects.Widget#g:method:translateCoordinates"), [triggerTooltipQuery]("GI.Gtk.Objects.Widget#g:method:triggerTooltipQuery"), [unmap]("GI.Gtk.Objects.Widget#g:method:unmap"), [unparent]("GI.Gtk.Objects.Widget#g:method:unparent"), [unrealize]("GI.Gtk.Objects.Widget#g:method:unrealize"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [unsetStateFlags]("GI.Gtk.Objects.Widget#g:method:unsetStateFlags"), [updateNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:updateNextAccessibleSibling"), [updatePlatformState]("GI.Gtk.Interfaces.Accessible#g:method:updatePlatformState"), [updateProperty]("GI.Gtk.Interfaces.Accessible#g:method:updateProperty"), [updateRelation]("GI.Gtk.Interfaces.Accessible#g:method:updateRelation"), [updateState]("GI.Gtk.Interfaces.Accessible#g:method:updateState"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleParent"), [getAccessibleRole]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleRole"), [getAllocatedBaseline]("GI.Gtk.Objects.Widget#g:method:getAllocatedBaseline"), [getAllocatedHeight]("GI.Gtk.Objects.Widget#g:method:getAllocatedHeight"), [getAllocatedWidth]("GI.Gtk.Objects.Widget#g:method:getAllocatedWidth"), [getAllocation]("GI.Gtk.Objects.Widget#g:method:getAllocation"), [getAllowedApis]("GI.Gtk.Objects.GLArea#g:method:getAllowedApis"), [getAncestor]("GI.Gtk.Objects.Widget#g:method:getAncestor"), [getApi]("GI.Gtk.Objects.GLArea#g:method:getApi"), [getAtContext]("GI.Gtk.Interfaces.Accessible#g:method:getAtContext"), [getAutoRender]("GI.Gtk.Objects.GLArea#g:method:getAutoRender"), [getBaseline]("GI.Gtk.Objects.Widget#g:method:getBaseline"), [getBounds]("GI.Gtk.Interfaces.Accessible#g:method:getBounds"), [getBuildableId]("GI.Gtk.Interfaces.Buildable#g:method:getBuildableId"), [getCanFocus]("GI.Gtk.Objects.Widget#g:method:getCanFocus"), [getCanTarget]("GI.Gtk.Objects.Widget#g:method:getCanTarget"), [getChildVisible]("GI.Gtk.Objects.Widget#g:method:getChildVisible"), [getClipboard]("GI.Gtk.Objects.Widget#g:method:getClipboard"), [getColor]("GI.Gtk.Objects.Widget#g:method:getColor"), [getContext]("GI.Gtk.Objects.GLArea#g:method:getContext"), [getCssClasses]("GI.Gtk.Objects.Widget#g:method:getCssClasses"), [getCssName]("GI.Gtk.Objects.Widget#g:method:getCssName"), [getCursor]("GI.Gtk.Objects.Widget#g:method:getCursor"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDirection]("GI.Gtk.Objects.Widget#g:method:getDirection"), [getDisplay]("GI.Gtk.Objects.Widget#g:method:getDisplay"), [getError]("GI.Gtk.Objects.GLArea#g:method:getError"), [getFirstAccessibleChild]("GI.Gtk.Interfaces.Accessible#g:method:getFirstAccessibleChild"), [getFirstChild]("GI.Gtk.Objects.Widget#g:method:getFirstChild"), [getFocusChild]("GI.Gtk.Objects.Widget#g:method:getFocusChild"), [getFocusOnClick]("GI.Gtk.Objects.Widget#g:method:getFocusOnClick"), [getFocusable]("GI.Gtk.Objects.Widget#g:method:getFocusable"), [getFontMap]("GI.Gtk.Objects.Widget#g:method:getFontMap"), [getFontOptions]("GI.Gtk.Objects.Widget#g:method:getFontOptions"), [getFrameClock]("GI.Gtk.Objects.Widget#g:method:getFrameClock"), [getHalign]("GI.Gtk.Objects.Widget#g:method:getHalign"), [getHasDepthBuffer]("GI.Gtk.Objects.GLArea#g:method:getHasDepthBuffer"), [getHasStencilBuffer]("GI.Gtk.Objects.GLArea#g:method:getHasStencilBuffer"), [getHasTooltip]("GI.Gtk.Objects.Widget#g:method:getHasTooltip"), [getHeight]("GI.Gtk.Objects.Widget#g:method:getHeight"), [getHexpand]("GI.Gtk.Objects.Widget#g:method:getHexpand"), [getHexpandSet]("GI.Gtk.Objects.Widget#g:method:getHexpandSet"), [getLastChild]("GI.Gtk.Objects.Widget#g:method:getLastChild"), [getLayoutManager]("GI.Gtk.Objects.Widget#g:method:getLayoutManager"), [getLimitEvents]("GI.Gtk.Objects.Widget#g:method:getLimitEvents"), [getMapped]("GI.Gtk.Objects.Widget#g:method:getMapped"), [getMarginBottom]("GI.Gtk.Objects.Widget#g:method:getMarginBottom"), [getMarginEnd]("GI.Gtk.Objects.Widget#g:method:getMarginEnd"), [getMarginStart]("GI.Gtk.Objects.Widget#g:method:getMarginStart"), [getMarginTop]("GI.Gtk.Objects.Widget#g:method:getMarginTop"), [getName]("GI.Gtk.Objects.Widget#g:method:getName"), [getNative]("GI.Gtk.Objects.Widget#g:method:getNative"), [getNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:getNextAccessibleSibling"), [getNextSibling]("GI.Gtk.Objects.Widget#g:method:getNextSibling"), [getOpacity]("GI.Gtk.Objects.Widget#g:method:getOpacity"), [getOverflow]("GI.Gtk.Objects.Widget#g:method:getOverflow"), [getPangoContext]("GI.Gtk.Objects.Widget#g:method:getPangoContext"), [getParent]("GI.Gtk.Objects.Widget#g:method:getParent"), [getPlatformState]("GI.Gtk.Interfaces.Accessible#g:method:getPlatformState"), [getPreferredSize]("GI.Gtk.Objects.Widget#g:method:getPreferredSize"), [getPrevSibling]("GI.Gtk.Objects.Widget#g:method:getPrevSibling"), [getPrimaryClipboard]("GI.Gtk.Objects.Widget#g:method:getPrimaryClipboard"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRealized]("GI.Gtk.Objects.Widget#g:method:getRealized"), [getReceivesDefault]("GI.Gtk.Objects.Widget#g:method:getReceivesDefault"), [getRequestMode]("GI.Gtk.Objects.Widget#g:method:getRequestMode"), [getRequiredVersion]("GI.Gtk.Objects.GLArea#g:method:getRequiredVersion"), [getRoot]("GI.Gtk.Objects.Widget#g:method:getRoot"), [getScaleFactor]("GI.Gtk.Objects.Widget#g:method:getScaleFactor"), [getSensitive]("GI.Gtk.Objects.Widget#g:method:getSensitive"), [getSettings]("GI.Gtk.Objects.Widget#g:method:getSettings"), [getSize]("GI.Gtk.Objects.Widget#g:method:getSize"), [getSizeRequest]("GI.Gtk.Objects.Widget#g:method:getSizeRequest"), [getStateFlags]("GI.Gtk.Objects.Widget#g:method:getStateFlags"), [getStyleContext]("GI.Gtk.Objects.Widget#g:method:getStyleContext"), [getTemplateChild]("GI.Gtk.Objects.Widget#g:method:getTemplateChild"), [getTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:getTooltipMarkup"), [getTooltipText]("GI.Gtk.Objects.Widget#g:method:getTooltipText"), [getUseEs]("GI.Gtk.Objects.GLArea#g:method:getUseEs"), [getValign]("GI.Gtk.Objects.Widget#g:method:getValign"), [getVexpand]("GI.Gtk.Objects.Widget#g:method:getVexpand"), [getVexpandSet]("GI.Gtk.Objects.Widget#g:method:getVexpandSet"), [getVisible]("GI.Gtk.Objects.Widget#g:method:getVisible"), [getWidth]("GI.Gtk.Objects.Widget#g:method:getWidth").
-- 
-- ==== Setters
-- [setAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:setAccessibleParent"), [setAllowedApis]("GI.Gtk.Objects.GLArea#g:method:setAllowedApis"), [setAutoRender]("GI.Gtk.Objects.GLArea#g:method:setAutoRender"), [setCanFocus]("GI.Gtk.Objects.Widget#g:method:setCanFocus"), [setCanTarget]("GI.Gtk.Objects.Widget#g:method:setCanTarget"), [setChildVisible]("GI.Gtk.Objects.Widget#g:method:setChildVisible"), [setCssClasses]("GI.Gtk.Objects.Widget#g:method:setCssClasses"), [setCursor]("GI.Gtk.Objects.Widget#g:method:setCursor"), [setCursorFromName]("GI.Gtk.Objects.Widget#g:method:setCursorFromName"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDirection]("GI.Gtk.Objects.Widget#g:method:setDirection"), [setError]("GI.Gtk.Objects.GLArea#g:method:setError"), [setFocusChild]("GI.Gtk.Objects.Widget#g:method:setFocusChild"), [setFocusOnClick]("GI.Gtk.Objects.Widget#g:method:setFocusOnClick"), [setFocusable]("GI.Gtk.Objects.Widget#g:method:setFocusable"), [setFontMap]("GI.Gtk.Objects.Widget#g:method:setFontMap"), [setFontOptions]("GI.Gtk.Objects.Widget#g:method:setFontOptions"), [setHalign]("GI.Gtk.Objects.Widget#g:method:setHalign"), [setHasDepthBuffer]("GI.Gtk.Objects.GLArea#g:method:setHasDepthBuffer"), [setHasStencilBuffer]("GI.Gtk.Objects.GLArea#g:method:setHasStencilBuffer"), [setHasTooltip]("GI.Gtk.Objects.Widget#g:method:setHasTooltip"), [setHexpand]("GI.Gtk.Objects.Widget#g:method:setHexpand"), [setHexpandSet]("GI.Gtk.Objects.Widget#g:method:setHexpandSet"), [setLayoutManager]("GI.Gtk.Objects.Widget#g:method:setLayoutManager"), [setLimitEvents]("GI.Gtk.Objects.Widget#g:method:setLimitEvents"), [setMarginBottom]("GI.Gtk.Objects.Widget#g:method:setMarginBottom"), [setMarginEnd]("GI.Gtk.Objects.Widget#g:method:setMarginEnd"), [setMarginStart]("GI.Gtk.Objects.Widget#g:method:setMarginStart"), [setMarginTop]("GI.Gtk.Objects.Widget#g:method:setMarginTop"), [setName]("GI.Gtk.Objects.Widget#g:method:setName"), [setOpacity]("GI.Gtk.Objects.Widget#g:method:setOpacity"), [setOverflow]("GI.Gtk.Objects.Widget#g:method:setOverflow"), [setParent]("GI.Gtk.Objects.Widget#g:method:setParent"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setReceivesDefault]("GI.Gtk.Objects.Widget#g:method:setReceivesDefault"), [setRequiredVersion]("GI.Gtk.Objects.GLArea#g:method:setRequiredVersion"), [setSensitive]("GI.Gtk.Objects.Widget#g:method:setSensitive"), [setSizeRequest]("GI.Gtk.Objects.Widget#g:method:setSizeRequest"), [setStateFlags]("GI.Gtk.Objects.Widget#g:method:setStateFlags"), [setTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:setTooltipMarkup"), [setTooltipText]("GI.Gtk.Objects.Widget#g:method:setTooltipText"), [setUseEs]("GI.Gtk.Objects.GLArea#g:method:setUseEs"), [setValign]("GI.Gtk.Objects.Widget#g:method:setValign"), [setVexpand]("GI.Gtk.Objects.Widget#g:method:setVexpand"), [setVexpandSet]("GI.Gtk.Objects.Widget#g:method:setVexpandSet"), [setVisible]("GI.Gtk.Objects.Widget#g:method:setVisible").

#if defined(ENABLE_OVERLOADING)
    ResolveGLAreaMethod                     ,
#endif

-- ** attachBuffers #method:attachBuffers#

#if defined(ENABLE_OVERLOADING)
    GLAreaAttachBuffersMethodInfo           ,
#endif
    gLAreaAttachBuffers                     ,


-- ** getAllowedApis #method:getAllowedApis#

#if defined(ENABLE_OVERLOADING)
    GLAreaGetAllowedApisMethodInfo          ,
#endif
    gLAreaGetAllowedApis                    ,


-- ** getApi #method:getApi#

#if defined(ENABLE_OVERLOADING)
    GLAreaGetApiMethodInfo                  ,
#endif
    gLAreaGetApi                            ,


-- ** getAutoRender #method:getAutoRender#

#if defined(ENABLE_OVERLOADING)
    GLAreaGetAutoRenderMethodInfo           ,
#endif
    gLAreaGetAutoRender                     ,


-- ** getContext #method:getContext#

#if defined(ENABLE_OVERLOADING)
    GLAreaGetContextMethodInfo              ,
#endif
    gLAreaGetContext                        ,


-- ** getError #method:getError#

#if defined(ENABLE_OVERLOADING)
    GLAreaGetErrorMethodInfo                ,
#endif
    gLAreaGetError                          ,


-- ** getHasDepthBuffer #method:getHasDepthBuffer#

#if defined(ENABLE_OVERLOADING)
    GLAreaGetHasDepthBufferMethodInfo       ,
#endif
    gLAreaGetHasDepthBuffer                 ,


-- ** getHasStencilBuffer #method:getHasStencilBuffer#

#if defined(ENABLE_OVERLOADING)
    GLAreaGetHasStencilBufferMethodInfo     ,
#endif
    gLAreaGetHasStencilBuffer               ,


-- ** getRequiredVersion #method:getRequiredVersion#

#if defined(ENABLE_OVERLOADING)
    GLAreaGetRequiredVersionMethodInfo      ,
#endif
    gLAreaGetRequiredVersion                ,


-- ** getUseEs #method:getUseEs#

#if defined(ENABLE_OVERLOADING)
    GLAreaGetUseEsMethodInfo                ,
#endif
    gLAreaGetUseEs                          ,


-- ** makeCurrent #method:makeCurrent#

#if defined(ENABLE_OVERLOADING)
    GLAreaMakeCurrentMethodInfo             ,
#endif
    gLAreaMakeCurrent                       ,


-- ** new #method:new#

    gLAreaNew                               ,


-- ** queueRender #method:queueRender#

#if defined(ENABLE_OVERLOADING)
    GLAreaQueueRenderMethodInfo             ,
#endif
    gLAreaQueueRender                       ,


-- ** setAllowedApis #method:setAllowedApis#

#if defined(ENABLE_OVERLOADING)
    GLAreaSetAllowedApisMethodInfo          ,
#endif
    gLAreaSetAllowedApis                    ,


-- ** setAutoRender #method:setAutoRender#

#if defined(ENABLE_OVERLOADING)
    GLAreaSetAutoRenderMethodInfo           ,
#endif
    gLAreaSetAutoRender                     ,


-- ** setError #method:setError#

#if defined(ENABLE_OVERLOADING)
    GLAreaSetErrorMethodInfo                ,
#endif
    gLAreaSetError                          ,


-- ** setHasDepthBuffer #method:setHasDepthBuffer#

#if defined(ENABLE_OVERLOADING)
    GLAreaSetHasDepthBufferMethodInfo       ,
#endif
    gLAreaSetHasDepthBuffer                 ,


-- ** setHasStencilBuffer #method:setHasStencilBuffer#

#if defined(ENABLE_OVERLOADING)
    GLAreaSetHasStencilBufferMethodInfo     ,
#endif
    gLAreaSetHasStencilBuffer               ,


-- ** setRequiredVersion #method:setRequiredVersion#

#if defined(ENABLE_OVERLOADING)
    GLAreaSetRequiredVersionMethodInfo      ,
#endif
    gLAreaSetRequiredVersion                ,


-- ** setUseEs #method:setUseEs#

#if defined(ENABLE_OVERLOADING)
    GLAreaSetUseEsMethodInfo                ,
#endif
    gLAreaSetUseEs                          ,




 -- * Properties


-- ** allowedApis #attr:allowedApis#
-- | The allowed APIs.
-- 
-- /Since: 4.12/

#if defined(ENABLE_OVERLOADING)
    GLAreaAllowedApisPropertyInfo           ,
#endif
    constructGLAreaAllowedApis              ,
#if defined(ENABLE_OVERLOADING)
    gLAreaAllowedApis                       ,
#endif
    getGLAreaAllowedApis                    ,
    setGLAreaAllowedApis                    ,


-- ** api #attr:api#
-- | The API currently in use.
-- 
-- /Since: 4.12/

#if defined(ENABLE_OVERLOADING)
    GLAreaApiPropertyInfo                   ,
#endif
#if defined(ENABLE_OVERLOADING)
    gLAreaApi                               ,
#endif
    getGLAreaApi                            ,


-- ** autoRender #attr:autoRender#
-- | If set to 'P.True' the [render](#g:signal:render) signal will be emitted every time
-- the widget draws.
-- 
-- This is the default and is useful if drawing the widget is faster.
-- 
-- If set to 'P.False' the data from previous rendering is kept around and will
-- be used for drawing the widget the next time, unless the window is resized.
-- In order to force a rendering 'GI.Gtk.Objects.GLArea.gLAreaQueueRender' must be called.
-- This mode is useful when the scene changes seldom, but takes a long time
-- to redraw.

#if defined(ENABLE_OVERLOADING)
    GLAreaAutoRenderPropertyInfo            ,
#endif
    constructGLAreaAutoRender               ,
#if defined(ENABLE_OVERLOADING)
    gLAreaAutoRender                        ,
#endif
    getGLAreaAutoRender                     ,
    setGLAreaAutoRender                     ,


-- ** context #attr:context#
-- | The @GdkGLContext@ used by the @GtkGLArea@ widget.
-- 
-- The @GtkGLArea@ widget is responsible for creating the @GdkGLContext@
-- instance. If you need to render with other kinds of buffers (stencil,
-- depth, etc), use render buffers.

#if defined(ENABLE_OVERLOADING)
    GLAreaContextPropertyInfo               ,
#endif
#if defined(ENABLE_OVERLOADING)
    gLAreaContext                           ,
#endif
    getGLAreaContext                        ,


-- ** hasDepthBuffer #attr:hasDepthBuffer#
-- | If set to 'P.True' the widget will allocate and enable a depth buffer for the
-- target framebuffer.
-- 
-- Setting this property will enable GL\'s depth testing as a side effect. If
-- you don\'t need depth testing, you should call @glDisable(GL_DEPTH_TEST)@
-- in your @GtkGLArea::render@ handler.

#if defined(ENABLE_OVERLOADING)
    GLAreaHasDepthBufferPropertyInfo        ,
#endif
    constructGLAreaHasDepthBuffer           ,
#if defined(ENABLE_OVERLOADING)
    gLAreaHasDepthBuffer                    ,
#endif
    getGLAreaHasDepthBuffer                 ,
    setGLAreaHasDepthBuffer                 ,


-- ** hasStencilBuffer #attr:hasStencilBuffer#
-- | If set to 'P.True' the widget will allocate and enable a stencil buffer for the
-- target framebuffer.

#if defined(ENABLE_OVERLOADING)
    GLAreaHasStencilBufferPropertyInfo      ,
#endif
    constructGLAreaHasStencilBuffer         ,
#if defined(ENABLE_OVERLOADING)
    gLAreaHasStencilBuffer                  ,
#endif
    getGLAreaHasStencilBuffer               ,
    setGLAreaHasStencilBuffer               ,


-- ** useEs #attr:useEs#
-- | If set to 'P.True' the widget will try to create a @GdkGLContext@ using
-- OpenGL ES instead of OpenGL.

#if defined(ENABLE_OVERLOADING)
    GLAreaUseEsPropertyInfo                 ,
#endif
    constructGLAreaUseEs                    ,
#if defined(ENABLE_OVERLOADING)
    gLAreaUseEs                             ,
#endif
    getGLAreaUseEs                          ,
    setGLAreaUseEs                          ,




 -- * Signals


-- ** createContext #signal:createContext#

    GLAreaCreateContextCallback             ,
#if defined(ENABLE_OVERLOADING)
    GLAreaCreateContextSignalInfo           ,
#endif
    afterGLAreaCreateContext                ,
    onGLAreaCreateContext                   ,


-- ** render #signal:render#

    GLAreaRenderCallback                    ,
#if defined(ENABLE_OVERLOADING)
    GLAreaRenderSignalInfo                  ,
#endif
    afterGLAreaRender                       ,
    onGLAreaRender                          ,


-- ** resize #signal:resize#

    GLAreaResizeCallback                    ,
#if defined(ENABLE_OVERLOADING)
    GLAreaResizeSignalInfo                  ,
#endif
    afterGLAreaResize                       ,
    onGLAreaResize                          ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.GLContext as Gdk.GLContext
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.ComponentTransfer as Gsk.ComponentTransfer
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Objects.GLContext as Gdk.GLContext
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget

#endif

-- | Memory-managed wrapper type.
newtype GLArea = GLArea (SP.ManagedPtr GLArea)
    deriving (Eq)

instance SP.ManagedPtrNewtype GLArea where
    toManagedPtr (GLArea p) = p

foreign import ccall "gtk_gl_area_get_type"
    c_gtk_gl_area_get_type :: IO B.Types.GType

instance B.Types.TypedObject GLArea where
    glibType = c_gtk_gl_area_get_type

instance B.Types.GObject GLArea

-- | Type class for types which can be safely cast to t'GLArea', for instance with `toGLArea`.
class (SP.GObject o, O.IsDescendantOf GLArea o) => IsGLArea o
instance (SP.GObject o, O.IsDescendantOf GLArea o) => IsGLArea o

instance O.HasParentTypes GLArea
type instance O.ParentTypes GLArea = '[Gtk.Widget.Widget, GObject.Object.Object, Gtk.Accessible.Accessible, Gtk.Buildable.Buildable, Gtk.ConstraintTarget.ConstraintTarget]

-- | Cast to t'GLArea', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toGLArea :: (MIO.MonadIO m, IsGLArea o) => o -> m GLArea
toGLArea = MIO.liftIO . B.ManagedPtr.unsafeCastTo GLArea

-- | Convert t'GLArea' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe GLArea) where
    gvalueGType_ = c_gtk_gl_area_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr GLArea)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr GLArea)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject GLArea ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveGLAreaMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveGLAreaMethod "actionSetEnabled" o = Gtk.Widget.WidgetActionSetEnabledMethodInfo
    ResolveGLAreaMethod "activate" o = Gtk.Widget.WidgetActivateMethodInfo
    ResolveGLAreaMethod "activateAction" o = Gtk.Widget.WidgetActivateActionMethodInfo
    ResolveGLAreaMethod "activateDefault" o = Gtk.Widget.WidgetActivateDefaultMethodInfo
    ResolveGLAreaMethod "addController" o = Gtk.Widget.WidgetAddControllerMethodInfo
    ResolveGLAreaMethod "addCssClass" o = Gtk.Widget.WidgetAddCssClassMethodInfo
    ResolveGLAreaMethod "addMnemonicLabel" o = Gtk.Widget.WidgetAddMnemonicLabelMethodInfo
    ResolveGLAreaMethod "addTickCallback" o = Gtk.Widget.WidgetAddTickCallbackMethodInfo
    ResolveGLAreaMethod "allocate" o = Gtk.Widget.WidgetAllocateMethodInfo
    ResolveGLAreaMethod "announce" o = Gtk.Accessible.AccessibleAnnounceMethodInfo
    ResolveGLAreaMethod "attachBuffers" o = GLAreaAttachBuffersMethodInfo
    ResolveGLAreaMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveGLAreaMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveGLAreaMethod "childFocus" o = Gtk.Widget.WidgetChildFocusMethodInfo
    ResolveGLAreaMethod "computeBounds" o = Gtk.Widget.WidgetComputeBoundsMethodInfo
    ResolveGLAreaMethod "computeExpand" o = Gtk.Widget.WidgetComputeExpandMethodInfo
    ResolveGLAreaMethod "computePoint" o = Gtk.Widget.WidgetComputePointMethodInfo
    ResolveGLAreaMethod "computeTransform" o = Gtk.Widget.WidgetComputeTransformMethodInfo
    ResolveGLAreaMethod "contains" o = Gtk.Widget.WidgetContainsMethodInfo
    ResolveGLAreaMethod "createPangoContext" o = Gtk.Widget.WidgetCreatePangoContextMethodInfo
    ResolveGLAreaMethod "createPangoLayout" o = Gtk.Widget.WidgetCreatePangoLayoutMethodInfo
    ResolveGLAreaMethod "disposeTemplate" o = Gtk.Widget.WidgetDisposeTemplateMethodInfo
    ResolveGLAreaMethod "dragCheckThreshold" o = Gtk.Widget.WidgetDragCheckThresholdMethodInfo
    ResolveGLAreaMethod "errorBell" o = Gtk.Widget.WidgetErrorBellMethodInfo
    ResolveGLAreaMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveGLAreaMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveGLAreaMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveGLAreaMethod "grabFocus" o = Gtk.Widget.WidgetGrabFocusMethodInfo
    ResolveGLAreaMethod "hasCssClass" o = Gtk.Widget.WidgetHasCssClassMethodInfo
    ResolveGLAreaMethod "hasDefault" o = Gtk.Widget.WidgetHasDefaultMethodInfo
    ResolveGLAreaMethod "hasFocus" o = Gtk.Widget.WidgetHasFocusMethodInfo
    ResolveGLAreaMethod "hasVisibleFocus" o = Gtk.Widget.WidgetHasVisibleFocusMethodInfo
    ResolveGLAreaMethod "hide" o = Gtk.Widget.WidgetHideMethodInfo
    ResolveGLAreaMethod "inDestruction" o = Gtk.Widget.WidgetInDestructionMethodInfo
    ResolveGLAreaMethod "initTemplate" o = Gtk.Widget.WidgetInitTemplateMethodInfo
    ResolveGLAreaMethod "insertActionGroup" o = Gtk.Widget.WidgetInsertActionGroupMethodInfo
    ResolveGLAreaMethod "insertAfter" o = Gtk.Widget.WidgetInsertAfterMethodInfo
    ResolveGLAreaMethod "insertBefore" o = Gtk.Widget.WidgetInsertBeforeMethodInfo
    ResolveGLAreaMethod "isAncestor" o = Gtk.Widget.WidgetIsAncestorMethodInfo
    ResolveGLAreaMethod "isDrawable" o = Gtk.Widget.WidgetIsDrawableMethodInfo
    ResolveGLAreaMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveGLAreaMethod "isFocus" o = Gtk.Widget.WidgetIsFocusMethodInfo
    ResolveGLAreaMethod "isSensitive" o = Gtk.Widget.WidgetIsSensitiveMethodInfo
    ResolveGLAreaMethod "isVisible" o = Gtk.Widget.WidgetIsVisibleMethodInfo
    ResolveGLAreaMethod "keynavFailed" o = Gtk.Widget.WidgetKeynavFailedMethodInfo
    ResolveGLAreaMethod "listMnemonicLabels" o = Gtk.Widget.WidgetListMnemonicLabelsMethodInfo
    ResolveGLAreaMethod "makeCurrent" o = GLAreaMakeCurrentMethodInfo
    ResolveGLAreaMethod "map" o = Gtk.Widget.WidgetMapMethodInfo
    ResolveGLAreaMethod "measure" o = Gtk.Widget.WidgetMeasureMethodInfo
    ResolveGLAreaMethod "mnemonicActivate" o = Gtk.Widget.WidgetMnemonicActivateMethodInfo
    ResolveGLAreaMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveGLAreaMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveGLAreaMethod "observeChildren" o = Gtk.Widget.WidgetObserveChildrenMethodInfo
    ResolveGLAreaMethod "observeControllers" o = Gtk.Widget.WidgetObserveControllersMethodInfo
    ResolveGLAreaMethod "pick" o = Gtk.Widget.WidgetPickMethodInfo
    ResolveGLAreaMethod "queueAllocate" o = Gtk.Widget.WidgetQueueAllocateMethodInfo
    ResolveGLAreaMethod "queueDraw" o = Gtk.Widget.WidgetQueueDrawMethodInfo
    ResolveGLAreaMethod "queueRender" o = GLAreaQueueRenderMethodInfo
    ResolveGLAreaMethod "queueResize" o = Gtk.Widget.WidgetQueueResizeMethodInfo
    ResolveGLAreaMethod "realize" o = Gtk.Widget.WidgetRealizeMethodInfo
    ResolveGLAreaMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveGLAreaMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveGLAreaMethod "removeController" o = Gtk.Widget.WidgetRemoveControllerMethodInfo
    ResolveGLAreaMethod "removeCssClass" o = Gtk.Widget.WidgetRemoveCssClassMethodInfo
    ResolveGLAreaMethod "removeMnemonicLabel" o = Gtk.Widget.WidgetRemoveMnemonicLabelMethodInfo
    ResolveGLAreaMethod "removeTickCallback" o = Gtk.Widget.WidgetRemoveTickCallbackMethodInfo
    ResolveGLAreaMethod "resetProperty" o = Gtk.Accessible.AccessibleResetPropertyMethodInfo
    ResolveGLAreaMethod "resetRelation" o = Gtk.Accessible.AccessibleResetRelationMethodInfo
    ResolveGLAreaMethod "resetState" o = Gtk.Accessible.AccessibleResetStateMethodInfo
    ResolveGLAreaMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveGLAreaMethod "shouldLayout" o = Gtk.Widget.WidgetShouldLayoutMethodInfo
    ResolveGLAreaMethod "show" o = Gtk.Widget.WidgetShowMethodInfo
    ResolveGLAreaMethod "sizeAllocate" o = Gtk.Widget.WidgetSizeAllocateMethodInfo
    ResolveGLAreaMethod "snapshotChild" o = Gtk.Widget.WidgetSnapshotChildMethodInfo
    ResolveGLAreaMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveGLAreaMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveGLAreaMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveGLAreaMethod "translateCoordinates" o = Gtk.Widget.WidgetTranslateCoordinatesMethodInfo
    ResolveGLAreaMethod "triggerTooltipQuery" o = Gtk.Widget.WidgetTriggerTooltipQueryMethodInfo
    ResolveGLAreaMethod "unmap" o = Gtk.Widget.WidgetUnmapMethodInfo
    ResolveGLAreaMethod "unparent" o = Gtk.Widget.WidgetUnparentMethodInfo
    ResolveGLAreaMethod "unrealize" o = Gtk.Widget.WidgetUnrealizeMethodInfo
    ResolveGLAreaMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveGLAreaMethod "unsetStateFlags" o = Gtk.Widget.WidgetUnsetStateFlagsMethodInfo
    ResolveGLAreaMethod "updateNextAccessibleSibling" o = Gtk.Accessible.AccessibleUpdateNextAccessibleSiblingMethodInfo
    ResolveGLAreaMethod "updatePlatformState" o = Gtk.Accessible.AccessibleUpdatePlatformStateMethodInfo
    ResolveGLAreaMethod "updateProperty" o = Gtk.Accessible.AccessibleUpdatePropertyMethodInfo
    ResolveGLAreaMethod "updateRelation" o = Gtk.Accessible.AccessibleUpdateRelationMethodInfo
    ResolveGLAreaMethod "updateState" o = Gtk.Accessible.AccessibleUpdateStateMethodInfo
    ResolveGLAreaMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveGLAreaMethod "getAccessibleParent" o = Gtk.Accessible.AccessibleGetAccessibleParentMethodInfo
    ResolveGLAreaMethod "getAccessibleRole" o = Gtk.Accessible.AccessibleGetAccessibleRoleMethodInfo
    ResolveGLAreaMethod "getAllocatedBaseline" o = Gtk.Widget.WidgetGetAllocatedBaselineMethodInfo
    ResolveGLAreaMethod "getAllocatedHeight" o = Gtk.Widget.WidgetGetAllocatedHeightMethodInfo
    ResolveGLAreaMethod "getAllocatedWidth" o = Gtk.Widget.WidgetGetAllocatedWidthMethodInfo
    ResolveGLAreaMethod "getAllocation" o = Gtk.Widget.WidgetGetAllocationMethodInfo
    ResolveGLAreaMethod "getAllowedApis" o = GLAreaGetAllowedApisMethodInfo
    ResolveGLAreaMethod "getAncestor" o = Gtk.Widget.WidgetGetAncestorMethodInfo
    ResolveGLAreaMethod "getApi" o = GLAreaGetApiMethodInfo
    ResolveGLAreaMethod "getAtContext" o = Gtk.Accessible.AccessibleGetAtContextMethodInfo
    ResolveGLAreaMethod "getAutoRender" o = GLAreaGetAutoRenderMethodInfo
    ResolveGLAreaMethod "getBaseline" o = Gtk.Widget.WidgetGetBaselineMethodInfo
    ResolveGLAreaMethod "getBounds" o = Gtk.Accessible.AccessibleGetBoundsMethodInfo
    ResolveGLAreaMethod "getBuildableId" o = Gtk.Buildable.BuildableGetBuildableIdMethodInfo
    ResolveGLAreaMethod "getCanFocus" o = Gtk.Widget.WidgetGetCanFocusMethodInfo
    ResolveGLAreaMethod "getCanTarget" o = Gtk.Widget.WidgetGetCanTargetMethodInfo
    ResolveGLAreaMethod "getChildVisible" o = Gtk.Widget.WidgetGetChildVisibleMethodInfo
    ResolveGLAreaMethod "getClipboard" o = Gtk.Widget.WidgetGetClipboardMethodInfo
    ResolveGLAreaMethod "getColor" o = Gtk.Widget.WidgetGetColorMethodInfo
    ResolveGLAreaMethod "getContext" o = GLAreaGetContextMethodInfo
    ResolveGLAreaMethod "getCssClasses" o = Gtk.Widget.WidgetGetCssClassesMethodInfo
    ResolveGLAreaMethod "getCssName" o = Gtk.Widget.WidgetGetCssNameMethodInfo
    ResolveGLAreaMethod "getCursor" o = Gtk.Widget.WidgetGetCursorMethodInfo
    ResolveGLAreaMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveGLAreaMethod "getDirection" o = Gtk.Widget.WidgetGetDirectionMethodInfo
    ResolveGLAreaMethod "getDisplay" o = Gtk.Widget.WidgetGetDisplayMethodInfo
    ResolveGLAreaMethod "getError" o = GLAreaGetErrorMethodInfo
    ResolveGLAreaMethod "getFirstAccessibleChild" o = Gtk.Accessible.AccessibleGetFirstAccessibleChildMethodInfo
    ResolveGLAreaMethod "getFirstChild" o = Gtk.Widget.WidgetGetFirstChildMethodInfo
    ResolveGLAreaMethod "getFocusChild" o = Gtk.Widget.WidgetGetFocusChildMethodInfo
    ResolveGLAreaMethod "getFocusOnClick" o = Gtk.Widget.WidgetGetFocusOnClickMethodInfo
    ResolveGLAreaMethod "getFocusable" o = Gtk.Widget.WidgetGetFocusableMethodInfo
    ResolveGLAreaMethod "getFontMap" o = Gtk.Widget.WidgetGetFontMapMethodInfo
    ResolveGLAreaMethod "getFontOptions" o = Gtk.Widget.WidgetGetFontOptionsMethodInfo
    ResolveGLAreaMethod "getFrameClock" o = Gtk.Widget.WidgetGetFrameClockMethodInfo
    ResolveGLAreaMethod "getHalign" o = Gtk.Widget.WidgetGetHalignMethodInfo
    ResolveGLAreaMethod "getHasDepthBuffer" o = GLAreaGetHasDepthBufferMethodInfo
    ResolveGLAreaMethod "getHasStencilBuffer" o = GLAreaGetHasStencilBufferMethodInfo
    ResolveGLAreaMethod "getHasTooltip" o = Gtk.Widget.WidgetGetHasTooltipMethodInfo
    ResolveGLAreaMethod "getHeight" o = Gtk.Widget.WidgetGetHeightMethodInfo
    ResolveGLAreaMethod "getHexpand" o = Gtk.Widget.WidgetGetHexpandMethodInfo
    ResolveGLAreaMethod "getHexpandSet" o = Gtk.Widget.WidgetGetHexpandSetMethodInfo
    ResolveGLAreaMethod "getLastChild" o = Gtk.Widget.WidgetGetLastChildMethodInfo
    ResolveGLAreaMethod "getLayoutManager" o = Gtk.Widget.WidgetGetLayoutManagerMethodInfo
    ResolveGLAreaMethod "getLimitEvents" o = Gtk.Widget.WidgetGetLimitEventsMethodInfo
    ResolveGLAreaMethod "getMapped" o = Gtk.Widget.WidgetGetMappedMethodInfo
    ResolveGLAreaMethod "getMarginBottom" o = Gtk.Widget.WidgetGetMarginBottomMethodInfo
    ResolveGLAreaMethod "getMarginEnd" o = Gtk.Widget.WidgetGetMarginEndMethodInfo
    ResolveGLAreaMethod "getMarginStart" o = Gtk.Widget.WidgetGetMarginStartMethodInfo
    ResolveGLAreaMethod "getMarginTop" o = Gtk.Widget.WidgetGetMarginTopMethodInfo
    ResolveGLAreaMethod "getName" o = Gtk.Widget.WidgetGetNameMethodInfo
    ResolveGLAreaMethod "getNative" o = Gtk.Widget.WidgetGetNativeMethodInfo
    ResolveGLAreaMethod "getNextAccessibleSibling" o = Gtk.Accessible.AccessibleGetNextAccessibleSiblingMethodInfo
    ResolveGLAreaMethod "getNextSibling" o = Gtk.Widget.WidgetGetNextSiblingMethodInfo
    ResolveGLAreaMethod "getOpacity" o = Gtk.Widget.WidgetGetOpacityMethodInfo
    ResolveGLAreaMethod "getOverflow" o = Gtk.Widget.WidgetGetOverflowMethodInfo
    ResolveGLAreaMethod "getPangoContext" o = Gtk.Widget.WidgetGetPangoContextMethodInfo
    ResolveGLAreaMethod "getParent" o = Gtk.Widget.WidgetGetParentMethodInfo
    ResolveGLAreaMethod "getPlatformState" o = Gtk.Accessible.AccessibleGetPlatformStateMethodInfo
    ResolveGLAreaMethod "getPreferredSize" o = Gtk.Widget.WidgetGetPreferredSizeMethodInfo
    ResolveGLAreaMethod "getPrevSibling" o = Gtk.Widget.WidgetGetPrevSiblingMethodInfo
    ResolveGLAreaMethod "getPrimaryClipboard" o = Gtk.Widget.WidgetGetPrimaryClipboardMethodInfo
    ResolveGLAreaMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveGLAreaMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveGLAreaMethod "getRealized" o = Gtk.Widget.WidgetGetRealizedMethodInfo
    ResolveGLAreaMethod "getReceivesDefault" o = Gtk.Widget.WidgetGetReceivesDefaultMethodInfo
    ResolveGLAreaMethod "getRequestMode" o = Gtk.Widget.WidgetGetRequestModeMethodInfo
    ResolveGLAreaMethod "getRequiredVersion" o = GLAreaGetRequiredVersionMethodInfo
    ResolveGLAreaMethod "getRoot" o = Gtk.Widget.WidgetGetRootMethodInfo
    ResolveGLAreaMethod "getScaleFactor" o = Gtk.Widget.WidgetGetScaleFactorMethodInfo
    ResolveGLAreaMethod "getSensitive" o = Gtk.Widget.WidgetGetSensitiveMethodInfo
    ResolveGLAreaMethod "getSettings" o = Gtk.Widget.WidgetGetSettingsMethodInfo
    ResolveGLAreaMethod "getSize" o = Gtk.Widget.WidgetGetSizeMethodInfo
    ResolveGLAreaMethod "getSizeRequest" o = Gtk.Widget.WidgetGetSizeRequestMethodInfo
    ResolveGLAreaMethod "getStateFlags" o = Gtk.Widget.WidgetGetStateFlagsMethodInfo
    ResolveGLAreaMethod "getStyleContext" o = Gtk.Widget.WidgetGetStyleContextMethodInfo
    ResolveGLAreaMethod "getTemplateChild" o = Gtk.Widget.WidgetGetTemplateChildMethodInfo
    ResolveGLAreaMethod "getTooltipMarkup" o = Gtk.Widget.WidgetGetTooltipMarkupMethodInfo
    ResolveGLAreaMethod "getTooltipText" o = Gtk.Widget.WidgetGetTooltipTextMethodInfo
    ResolveGLAreaMethod "getUseEs" o = GLAreaGetUseEsMethodInfo
    ResolveGLAreaMethod "getValign" o = Gtk.Widget.WidgetGetValignMethodInfo
    ResolveGLAreaMethod "getVexpand" o = Gtk.Widget.WidgetGetVexpandMethodInfo
    ResolveGLAreaMethod "getVexpandSet" o = Gtk.Widget.WidgetGetVexpandSetMethodInfo
    ResolveGLAreaMethod "getVisible" o = Gtk.Widget.WidgetGetVisibleMethodInfo
    ResolveGLAreaMethod "getWidth" o = Gtk.Widget.WidgetGetWidthMethodInfo
    ResolveGLAreaMethod "setAccessibleParent" o = Gtk.Accessible.AccessibleSetAccessibleParentMethodInfo
    ResolveGLAreaMethod "setAllowedApis" o = GLAreaSetAllowedApisMethodInfo
    ResolveGLAreaMethod "setAutoRender" o = GLAreaSetAutoRenderMethodInfo
    ResolveGLAreaMethod "setCanFocus" o = Gtk.Widget.WidgetSetCanFocusMethodInfo
    ResolveGLAreaMethod "setCanTarget" o = Gtk.Widget.WidgetSetCanTargetMethodInfo
    ResolveGLAreaMethod "setChildVisible" o = Gtk.Widget.WidgetSetChildVisibleMethodInfo
    ResolveGLAreaMethod "setCssClasses" o = Gtk.Widget.WidgetSetCssClassesMethodInfo
    ResolveGLAreaMethod "setCursor" o = Gtk.Widget.WidgetSetCursorMethodInfo
    ResolveGLAreaMethod "setCursorFromName" o = Gtk.Widget.WidgetSetCursorFromNameMethodInfo
    ResolveGLAreaMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveGLAreaMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveGLAreaMethod "setDirection" o = Gtk.Widget.WidgetSetDirectionMethodInfo
    ResolveGLAreaMethod "setError" o = GLAreaSetErrorMethodInfo
    ResolveGLAreaMethod "setFocusChild" o = Gtk.Widget.WidgetSetFocusChildMethodInfo
    ResolveGLAreaMethod "setFocusOnClick" o = Gtk.Widget.WidgetSetFocusOnClickMethodInfo
    ResolveGLAreaMethod "setFocusable" o = Gtk.Widget.WidgetSetFocusableMethodInfo
    ResolveGLAreaMethod "setFontMap" o = Gtk.Widget.WidgetSetFontMapMethodInfo
    ResolveGLAreaMethod "setFontOptions" o = Gtk.Widget.WidgetSetFontOptionsMethodInfo
    ResolveGLAreaMethod "setHalign" o = Gtk.Widget.WidgetSetHalignMethodInfo
    ResolveGLAreaMethod "setHasDepthBuffer" o = GLAreaSetHasDepthBufferMethodInfo
    ResolveGLAreaMethod "setHasStencilBuffer" o = GLAreaSetHasStencilBufferMethodInfo
    ResolveGLAreaMethod "setHasTooltip" o = Gtk.Widget.WidgetSetHasTooltipMethodInfo
    ResolveGLAreaMethod "setHexpand" o = Gtk.Widget.WidgetSetHexpandMethodInfo
    ResolveGLAreaMethod "setHexpandSet" o = Gtk.Widget.WidgetSetHexpandSetMethodInfo
    ResolveGLAreaMethod "setLayoutManager" o = Gtk.Widget.WidgetSetLayoutManagerMethodInfo
    ResolveGLAreaMethod "setLimitEvents" o = Gtk.Widget.WidgetSetLimitEventsMethodInfo
    ResolveGLAreaMethod "setMarginBottom" o = Gtk.Widget.WidgetSetMarginBottomMethodInfo
    ResolveGLAreaMethod "setMarginEnd" o = Gtk.Widget.WidgetSetMarginEndMethodInfo
    ResolveGLAreaMethod "setMarginStart" o = Gtk.Widget.WidgetSetMarginStartMethodInfo
    ResolveGLAreaMethod "setMarginTop" o = Gtk.Widget.WidgetSetMarginTopMethodInfo
    ResolveGLAreaMethod "setName" o = Gtk.Widget.WidgetSetNameMethodInfo
    ResolveGLAreaMethod "setOpacity" o = Gtk.Widget.WidgetSetOpacityMethodInfo
    ResolveGLAreaMethod "setOverflow" o = Gtk.Widget.WidgetSetOverflowMethodInfo
    ResolveGLAreaMethod "setParent" o = Gtk.Widget.WidgetSetParentMethodInfo
    ResolveGLAreaMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveGLAreaMethod "setReceivesDefault" o = Gtk.Widget.WidgetSetReceivesDefaultMethodInfo
    ResolveGLAreaMethod "setRequiredVersion" o = GLAreaSetRequiredVersionMethodInfo
    ResolveGLAreaMethod "setSensitive" o = Gtk.Widget.WidgetSetSensitiveMethodInfo
    ResolveGLAreaMethod "setSizeRequest" o = Gtk.Widget.WidgetSetSizeRequestMethodInfo
    ResolveGLAreaMethod "setStateFlags" o = Gtk.Widget.WidgetSetStateFlagsMethodInfo
    ResolveGLAreaMethod "setTooltipMarkup" o = Gtk.Widget.WidgetSetTooltipMarkupMethodInfo
    ResolveGLAreaMethod "setTooltipText" o = Gtk.Widget.WidgetSetTooltipTextMethodInfo
    ResolveGLAreaMethod "setUseEs" o = GLAreaSetUseEsMethodInfo
    ResolveGLAreaMethod "setValign" o = Gtk.Widget.WidgetSetValignMethodInfo
    ResolveGLAreaMethod "setVexpand" o = Gtk.Widget.WidgetSetVexpandMethodInfo
    ResolveGLAreaMethod "setVexpandSet" o = Gtk.Widget.WidgetSetVexpandSetMethodInfo
    ResolveGLAreaMethod "setVisible" o = Gtk.Widget.WidgetSetVisibleMethodInfo
    ResolveGLAreaMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveGLAreaMethod t GLArea, O.OverloadedMethod info GLArea p) => OL.IsLabel t (GLArea -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveGLAreaMethod t GLArea, O.OverloadedMethod info GLArea p, R.HasField t GLArea p) => R.HasField t GLArea p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveGLAreaMethod t GLArea, O.OverloadedMethodInfo info GLArea) => OL.IsLabel t (O.MethodProxy info GLArea) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal GLArea::create-context
-- | Emitted when the widget is being realized.
-- 
-- This allows you to override how the GL context is created.
-- This is useful when you want to reuse an existing GL context,
-- or if you want to try creating different kinds of GL options.
-- 
-- If context creation fails then the signal handler can use
-- 'GI.Gtk.Objects.GLArea.gLAreaSetError' to register a more detailed error
-- of how the construction failed.
type GLAreaCreateContextCallback =
    IO Gdk.GLContext.GLContext
    -- ^ __Returns:__ a newly created @GdkGLContext@;
    --     the @GtkGLArea@ widget will take ownership of the returned value.

type C_GLAreaCreateContextCallback =
    Ptr GLArea ->                           -- object
    Ptr () ->                               -- user_data
    IO (Ptr Gdk.GLContext.GLContext)

-- | Generate a function pointer callable from C code, from a `C_GLAreaCreateContextCallback`.
foreign import ccall "wrapper"
    mk_GLAreaCreateContextCallback :: C_GLAreaCreateContextCallback -> IO (FunPtr C_GLAreaCreateContextCallback)

wrap_GLAreaCreateContextCallback :: 
    GObject a => (a -> GLAreaCreateContextCallback) ->
    C_GLAreaCreateContextCallback
wrap_GLAreaCreateContextCallback gi'cb gi'selfPtr _ = do
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 
    result' <- B.ManagedPtr.disownObject result
    return result'


-- | Connect a signal handler for the [createContext](#signal:createContext) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' gLArea #createContext callback
-- @
-- 
-- 
onGLAreaCreateContext :: (IsGLArea a, MonadIO m) => a -> ((?self :: a) => GLAreaCreateContextCallback) -> m SignalHandlerId
onGLAreaCreateContext obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GLAreaCreateContextCallback wrapped
    wrapped'' <- mk_GLAreaCreateContextCallback wrapped'
    connectSignalFunPtr obj "create-context" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [createContext](#signal:createContext) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' gLArea #createContext callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterGLAreaCreateContext :: (IsGLArea a, MonadIO m) => a -> ((?self :: a) => GLAreaCreateContextCallback) -> m SignalHandlerId
afterGLAreaCreateContext obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GLAreaCreateContextCallback wrapped
    wrapped'' <- mk_GLAreaCreateContextCallback wrapped'
    connectSignalFunPtr obj "create-context" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data GLAreaCreateContextSignalInfo
instance SignalInfo GLAreaCreateContextSignalInfo where
    type HaskellCallbackType GLAreaCreateContextSignalInfo = GLAreaCreateContextCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_GLAreaCreateContextCallback cb
        cb'' <- mk_GLAreaCreateContextCallback cb'
        connectSignalFunPtr obj "create-context" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea::create-context"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:signal:createContext"})

#endif

-- signal GLArea::render
-- | Emitted every time the contents of the @GtkGLArea@ should be redrawn.
-- 
-- The /@context@/ is bound to the /@area@/ prior to emitting this function,
-- and the buffers are painted to the window once the emission terminates.
type GLAreaRenderCallback =
    Gdk.GLContext.GLContext
    -- ^ /@context@/: the @GdkGLContext@ used by /@area@/
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_GLAreaRenderCallback =
    Ptr GLArea ->                           -- object
    Ptr Gdk.GLContext.GLContext ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_GLAreaRenderCallback`.
foreign import ccall "wrapper"
    mk_GLAreaRenderCallback :: C_GLAreaRenderCallback -> IO (FunPtr C_GLAreaRenderCallback)

wrap_GLAreaRenderCallback :: 
    GObject a => (a -> GLAreaRenderCallback) ->
    C_GLAreaRenderCallback
wrap_GLAreaRenderCallback gi'cb gi'selfPtr context _ = do
    context' <- (newObject Gdk.GLContext.GLContext) context
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [render](#signal:render) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' gLArea #render callback
-- @
-- 
-- 
onGLAreaRender :: (IsGLArea a, MonadIO m) => a -> ((?self :: a) => GLAreaRenderCallback) -> m SignalHandlerId
onGLAreaRender obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GLAreaRenderCallback wrapped
    wrapped'' <- mk_GLAreaRenderCallback wrapped'
    connectSignalFunPtr obj "render" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [render](#signal:render) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' gLArea #render callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterGLAreaRender :: (IsGLArea a, MonadIO m) => a -> ((?self :: a) => GLAreaRenderCallback) -> m SignalHandlerId
afterGLAreaRender obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GLAreaRenderCallback wrapped
    wrapped'' <- mk_GLAreaRenderCallback wrapped'
    connectSignalFunPtr obj "render" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data GLAreaRenderSignalInfo
instance SignalInfo GLAreaRenderSignalInfo where
    type HaskellCallbackType GLAreaRenderSignalInfo = GLAreaRenderCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_GLAreaRenderCallback cb
        cb'' <- mk_GLAreaRenderCallback cb'
        connectSignalFunPtr obj "render" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea::render"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:signal:render"})

#endif

-- signal GLArea::resize
-- | Emitted once when the widget is realized, and then each time the widget
-- is changed while realized.
-- 
-- This is useful in order to keep GL state up to date with the widget size,
-- like for instance camera properties which may depend on the width\/height
-- ratio.
-- 
-- The GL context for the area is guaranteed to be current when this signal
-- is emitted.
-- 
-- The default handler sets up the GL viewport.
type GLAreaResizeCallback =
    Int32
    -- ^ /@width@/: the width of the viewport
    -> Int32
    -- ^ /@height@/: the height of the viewport
    -> IO ()

type C_GLAreaResizeCallback =
    Ptr GLArea ->                           -- object
    Int32 ->
    Int32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_GLAreaResizeCallback`.
foreign import ccall "wrapper"
    mk_GLAreaResizeCallback :: C_GLAreaResizeCallback -> IO (FunPtr C_GLAreaResizeCallback)

wrap_GLAreaResizeCallback :: 
    GObject a => (a -> GLAreaResizeCallback) ->
    C_GLAreaResizeCallback
wrap_GLAreaResizeCallback gi'cb gi'selfPtr width height _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  width height


-- | Connect a signal handler for the [resize](#signal:resize) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' gLArea #resize callback
-- @
-- 
-- 
onGLAreaResize :: (IsGLArea a, MonadIO m) => a -> ((?self :: a) => GLAreaResizeCallback) -> m SignalHandlerId
onGLAreaResize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GLAreaResizeCallback wrapped
    wrapped'' <- mk_GLAreaResizeCallback wrapped'
    connectSignalFunPtr obj "resize" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [resize](#signal:resize) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' gLArea #resize callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterGLAreaResize :: (IsGLArea a, MonadIO m) => a -> ((?self :: a) => GLAreaResizeCallback) -> m SignalHandlerId
afterGLAreaResize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GLAreaResizeCallback wrapped
    wrapped'' <- mk_GLAreaResizeCallback wrapped'
    connectSignalFunPtr obj "resize" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data GLAreaResizeSignalInfo
instance SignalInfo GLAreaResizeSignalInfo where
    type HaskellCallbackType GLAreaResizeSignalInfo = GLAreaResizeCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_GLAreaResizeCallback cb
        cb'' <- mk_GLAreaResizeCallback cb'
        connectSignalFunPtr obj "resize" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea::resize"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:signal:resize"})

#endif

-- VVV Prop "allowed-apis"
   -- Type: TInterface (Name {namespace = "Gdk", name = "GLAPI"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@allowed-apis@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gLArea #allowedApis
-- @
getGLAreaAllowedApis :: (MonadIO m, IsGLArea o) => o -> m [Gdk.Flags.GLAPI]
getGLAreaAllowedApis obj = MIO.liftIO $ B.Properties.getObjectPropertyFlags obj "allowed-apis"

-- | Set the value of the “@allowed-apis@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' gLArea [ #allowedApis 'Data.GI.Base.Attributes.:=' value ]
-- @
setGLAreaAllowedApis :: (MonadIO m, IsGLArea o) => o -> [Gdk.Flags.GLAPI] -> m ()
setGLAreaAllowedApis obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyFlags obj "allowed-apis" val

-- | Construct a t'GValueConstruct' with valid value for the “@allowed-apis@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructGLAreaAllowedApis :: (IsGLArea o, MIO.MonadIO m) => [Gdk.Flags.GLAPI] -> m (GValueConstruct o)
constructGLAreaAllowedApis val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFlags "allowed-apis" val

#if defined(ENABLE_OVERLOADING)
data GLAreaAllowedApisPropertyInfo
instance AttrInfo GLAreaAllowedApisPropertyInfo where
    type AttrAllowedOps GLAreaAllowedApisPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint GLAreaAllowedApisPropertyInfo = IsGLArea
    type AttrSetTypeConstraint GLAreaAllowedApisPropertyInfo = (~) [Gdk.Flags.GLAPI]
    type AttrTransferTypeConstraint GLAreaAllowedApisPropertyInfo = (~) [Gdk.Flags.GLAPI]
    type AttrTransferType GLAreaAllowedApisPropertyInfo = [Gdk.Flags.GLAPI]
    type AttrGetType GLAreaAllowedApisPropertyInfo = [Gdk.Flags.GLAPI]
    type AttrLabel GLAreaAllowedApisPropertyInfo = "allowed-apis"
    type AttrOrigin GLAreaAllowedApisPropertyInfo = GLArea
    attrGet = getGLAreaAllowedApis
    attrSet = setGLAreaAllowedApis
    attrPut = setGLAreaAllowedApis
    attrTransfer _ v = do
        return v
    attrConstruct = constructGLAreaAllowedApis
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.allowedApis"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:attr:allowedApis"
        })
#endif

-- VVV Prop "api"
   -- Type: TInterface (Name {namespace = "Gdk", name = "GLAPI"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@api@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gLArea #api
-- @
getGLAreaApi :: (MonadIO m, IsGLArea o) => o -> m [Gdk.Flags.GLAPI]
getGLAreaApi obj = MIO.liftIO $ B.Properties.getObjectPropertyFlags obj "api"

#if defined(ENABLE_OVERLOADING)
data GLAreaApiPropertyInfo
instance AttrInfo GLAreaApiPropertyInfo where
    type AttrAllowedOps GLAreaApiPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint GLAreaApiPropertyInfo = IsGLArea
    type AttrSetTypeConstraint GLAreaApiPropertyInfo = (~) ()
    type AttrTransferTypeConstraint GLAreaApiPropertyInfo = (~) ()
    type AttrTransferType GLAreaApiPropertyInfo = ()
    type AttrGetType GLAreaApiPropertyInfo = [Gdk.Flags.GLAPI]
    type AttrLabel GLAreaApiPropertyInfo = "api"
    type AttrOrigin GLAreaApiPropertyInfo = GLArea
    attrGet = getGLAreaApi
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.api"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:attr:api"
        })
#endif

-- VVV Prop "auto-render"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@auto-render@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gLArea #autoRender
-- @
getGLAreaAutoRender :: (MonadIO m, IsGLArea o) => o -> m Bool
getGLAreaAutoRender obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "auto-render"

-- | Set the value of the “@auto-render@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' gLArea [ #autoRender 'Data.GI.Base.Attributes.:=' value ]
-- @
setGLAreaAutoRender :: (MonadIO m, IsGLArea o) => o -> Bool -> m ()
setGLAreaAutoRender obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "auto-render" val

-- | Construct a t'GValueConstruct' with valid value for the “@auto-render@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructGLAreaAutoRender :: (IsGLArea o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructGLAreaAutoRender val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "auto-render" val

#if defined(ENABLE_OVERLOADING)
data GLAreaAutoRenderPropertyInfo
instance AttrInfo GLAreaAutoRenderPropertyInfo where
    type AttrAllowedOps GLAreaAutoRenderPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint GLAreaAutoRenderPropertyInfo = IsGLArea
    type AttrSetTypeConstraint GLAreaAutoRenderPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint GLAreaAutoRenderPropertyInfo = (~) Bool
    type AttrTransferType GLAreaAutoRenderPropertyInfo = Bool
    type AttrGetType GLAreaAutoRenderPropertyInfo = Bool
    type AttrLabel GLAreaAutoRenderPropertyInfo = "auto-render"
    type AttrOrigin GLAreaAutoRenderPropertyInfo = GLArea
    attrGet = getGLAreaAutoRender
    attrSet = setGLAreaAutoRender
    attrPut = setGLAreaAutoRender
    attrTransfer _ v = do
        return v
    attrConstruct = constructGLAreaAutoRender
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.autoRender"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:attr:autoRender"
        })
#endif

-- VVV Prop "context"
   -- Type: TInterface (Name {namespace = "Gdk", name = "GLContext"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@context@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gLArea #context
-- @
getGLAreaContext :: (MonadIO m, IsGLArea o) => o -> m (Maybe Gdk.GLContext.GLContext)
getGLAreaContext obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "context" Gdk.GLContext.GLContext

#if defined(ENABLE_OVERLOADING)
data GLAreaContextPropertyInfo
instance AttrInfo GLAreaContextPropertyInfo where
    type AttrAllowedOps GLAreaContextPropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint GLAreaContextPropertyInfo = IsGLArea
    type AttrSetTypeConstraint GLAreaContextPropertyInfo = (~) ()
    type AttrTransferTypeConstraint GLAreaContextPropertyInfo = (~) ()
    type AttrTransferType GLAreaContextPropertyInfo = ()
    type AttrGetType GLAreaContextPropertyInfo = (Maybe Gdk.GLContext.GLContext)
    type AttrLabel GLAreaContextPropertyInfo = "context"
    type AttrOrigin GLAreaContextPropertyInfo = GLArea
    attrGet = getGLAreaContext
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.context"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:attr:context"
        })
#endif

-- VVV Prop "has-depth-buffer"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@has-depth-buffer@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gLArea #hasDepthBuffer
-- @
getGLAreaHasDepthBuffer :: (MonadIO m, IsGLArea o) => o -> m Bool
getGLAreaHasDepthBuffer obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "has-depth-buffer"

-- | Set the value of the “@has-depth-buffer@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' gLArea [ #hasDepthBuffer 'Data.GI.Base.Attributes.:=' value ]
-- @
setGLAreaHasDepthBuffer :: (MonadIO m, IsGLArea o) => o -> Bool -> m ()
setGLAreaHasDepthBuffer obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "has-depth-buffer" val

-- | Construct a t'GValueConstruct' with valid value for the “@has-depth-buffer@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructGLAreaHasDepthBuffer :: (IsGLArea o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructGLAreaHasDepthBuffer val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "has-depth-buffer" val

#if defined(ENABLE_OVERLOADING)
data GLAreaHasDepthBufferPropertyInfo
instance AttrInfo GLAreaHasDepthBufferPropertyInfo where
    type AttrAllowedOps GLAreaHasDepthBufferPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint GLAreaHasDepthBufferPropertyInfo = IsGLArea
    type AttrSetTypeConstraint GLAreaHasDepthBufferPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint GLAreaHasDepthBufferPropertyInfo = (~) Bool
    type AttrTransferType GLAreaHasDepthBufferPropertyInfo = Bool
    type AttrGetType GLAreaHasDepthBufferPropertyInfo = Bool
    type AttrLabel GLAreaHasDepthBufferPropertyInfo = "has-depth-buffer"
    type AttrOrigin GLAreaHasDepthBufferPropertyInfo = GLArea
    attrGet = getGLAreaHasDepthBuffer
    attrSet = setGLAreaHasDepthBuffer
    attrPut = setGLAreaHasDepthBuffer
    attrTransfer _ v = do
        return v
    attrConstruct = constructGLAreaHasDepthBuffer
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.hasDepthBuffer"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:attr:hasDepthBuffer"
        })
#endif

-- VVV Prop "has-stencil-buffer"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@has-stencil-buffer@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gLArea #hasStencilBuffer
-- @
getGLAreaHasStencilBuffer :: (MonadIO m, IsGLArea o) => o -> m Bool
getGLAreaHasStencilBuffer obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "has-stencil-buffer"

-- | Set the value of the “@has-stencil-buffer@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' gLArea [ #hasStencilBuffer 'Data.GI.Base.Attributes.:=' value ]
-- @
setGLAreaHasStencilBuffer :: (MonadIO m, IsGLArea o) => o -> Bool -> m ()
setGLAreaHasStencilBuffer obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "has-stencil-buffer" val

-- | Construct a t'GValueConstruct' with valid value for the “@has-stencil-buffer@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructGLAreaHasStencilBuffer :: (IsGLArea o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructGLAreaHasStencilBuffer val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "has-stencil-buffer" val

#if defined(ENABLE_OVERLOADING)
data GLAreaHasStencilBufferPropertyInfo
instance AttrInfo GLAreaHasStencilBufferPropertyInfo where
    type AttrAllowedOps GLAreaHasStencilBufferPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint GLAreaHasStencilBufferPropertyInfo = IsGLArea
    type AttrSetTypeConstraint GLAreaHasStencilBufferPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint GLAreaHasStencilBufferPropertyInfo = (~) Bool
    type AttrTransferType GLAreaHasStencilBufferPropertyInfo = Bool
    type AttrGetType GLAreaHasStencilBufferPropertyInfo = Bool
    type AttrLabel GLAreaHasStencilBufferPropertyInfo = "has-stencil-buffer"
    type AttrOrigin GLAreaHasStencilBufferPropertyInfo = GLArea
    attrGet = getGLAreaHasStencilBuffer
    attrSet = setGLAreaHasStencilBuffer
    attrPut = setGLAreaHasStencilBuffer
    attrTransfer _ v = do
        return v
    attrConstruct = constructGLAreaHasStencilBuffer
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.hasStencilBuffer"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:attr:hasStencilBuffer"
        })
#endif

-- VVV Prop "use-es"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@use-es@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gLArea #useEs
-- @
getGLAreaUseEs :: (MonadIO m, IsGLArea o) => o -> m Bool
getGLAreaUseEs obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "use-es"

-- | Set the value of the “@use-es@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' gLArea [ #useEs 'Data.GI.Base.Attributes.:=' value ]
-- @
setGLAreaUseEs :: (MonadIO m, IsGLArea o) => o -> Bool -> m ()
setGLAreaUseEs obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "use-es" val

-- | Construct a t'GValueConstruct' with valid value for the “@use-es@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructGLAreaUseEs :: (IsGLArea o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructGLAreaUseEs val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "use-es" val

#if defined(ENABLE_OVERLOADING)
data GLAreaUseEsPropertyInfo
instance AttrInfo GLAreaUseEsPropertyInfo where
    type AttrAllowedOps GLAreaUseEsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint GLAreaUseEsPropertyInfo = IsGLArea
    type AttrSetTypeConstraint GLAreaUseEsPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint GLAreaUseEsPropertyInfo = (~) Bool
    type AttrTransferType GLAreaUseEsPropertyInfo = Bool
    type AttrGetType GLAreaUseEsPropertyInfo = Bool
    type AttrLabel GLAreaUseEsPropertyInfo = "use-es"
    type AttrOrigin GLAreaUseEsPropertyInfo = GLArea
    attrGet = getGLAreaUseEs
    attrSet = setGLAreaUseEs
    attrPut = setGLAreaUseEs
    attrTransfer _ v = do
        return v
    attrConstruct = constructGLAreaUseEs
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.useEs"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#g:attr:useEs"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList GLArea
type instance O.AttributeList GLArea = GLAreaAttributeList
type GLAreaAttributeList = ('[ '("accessibleRole", Gtk.Accessible.AccessibleAccessibleRolePropertyInfo), '("allowedApis", GLAreaAllowedApisPropertyInfo), '("api", GLAreaApiPropertyInfo), '("autoRender", GLAreaAutoRenderPropertyInfo), '("canFocus", Gtk.Widget.WidgetCanFocusPropertyInfo), '("canTarget", Gtk.Widget.WidgetCanTargetPropertyInfo), '("context", GLAreaContextPropertyInfo), '("cssClasses", Gtk.Widget.WidgetCssClassesPropertyInfo), '("cssName", Gtk.Widget.WidgetCssNamePropertyInfo), '("cursor", Gtk.Widget.WidgetCursorPropertyInfo), '("focusOnClick", Gtk.Widget.WidgetFocusOnClickPropertyInfo), '("focusable", Gtk.Widget.WidgetFocusablePropertyInfo), '("halign", Gtk.Widget.WidgetHalignPropertyInfo), '("hasDefault", Gtk.Widget.WidgetHasDefaultPropertyInfo), '("hasDepthBuffer", GLAreaHasDepthBufferPropertyInfo), '("hasFocus", Gtk.Widget.WidgetHasFocusPropertyInfo), '("hasStencilBuffer", GLAreaHasStencilBufferPropertyInfo), '("hasTooltip", Gtk.Widget.WidgetHasTooltipPropertyInfo), '("heightRequest", Gtk.Widget.WidgetHeightRequestPropertyInfo), '("hexpand", Gtk.Widget.WidgetHexpandPropertyInfo), '("hexpandSet", Gtk.Widget.WidgetHexpandSetPropertyInfo), '("layoutManager", Gtk.Widget.WidgetLayoutManagerPropertyInfo), '("limitEvents", Gtk.Widget.WidgetLimitEventsPropertyInfo), '("marginBottom", Gtk.Widget.WidgetMarginBottomPropertyInfo), '("marginEnd", Gtk.Widget.WidgetMarginEndPropertyInfo), '("marginStart", Gtk.Widget.WidgetMarginStartPropertyInfo), '("marginTop", Gtk.Widget.WidgetMarginTopPropertyInfo), '("name", Gtk.Widget.WidgetNamePropertyInfo), '("opacity", Gtk.Widget.WidgetOpacityPropertyInfo), '("overflow", Gtk.Widget.WidgetOverflowPropertyInfo), '("parent", Gtk.Widget.WidgetParentPropertyInfo), '("receivesDefault", Gtk.Widget.WidgetReceivesDefaultPropertyInfo), '("root", Gtk.Widget.WidgetRootPropertyInfo), '("scaleFactor", Gtk.Widget.WidgetScaleFactorPropertyInfo), '("sensitive", Gtk.Widget.WidgetSensitivePropertyInfo), '("tooltipMarkup", Gtk.Widget.WidgetTooltipMarkupPropertyInfo), '("tooltipText", Gtk.Widget.WidgetTooltipTextPropertyInfo), '("useEs", GLAreaUseEsPropertyInfo), '("valign", Gtk.Widget.WidgetValignPropertyInfo), '("vexpand", Gtk.Widget.WidgetVexpandPropertyInfo), '("vexpandSet", Gtk.Widget.WidgetVexpandSetPropertyInfo), '("visible", Gtk.Widget.WidgetVisiblePropertyInfo), '("widthRequest", Gtk.Widget.WidgetWidthRequestPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
gLAreaAllowedApis :: AttrLabelProxy "allowedApis"
gLAreaAllowedApis = AttrLabelProxy

gLAreaApi :: AttrLabelProxy "api"
gLAreaApi = AttrLabelProxy

gLAreaAutoRender :: AttrLabelProxy "autoRender"
gLAreaAutoRender = AttrLabelProxy

gLAreaContext :: AttrLabelProxy "context"
gLAreaContext = AttrLabelProxy

gLAreaHasDepthBuffer :: AttrLabelProxy "hasDepthBuffer"
gLAreaHasDepthBuffer = AttrLabelProxy

gLAreaHasStencilBuffer :: AttrLabelProxy "hasStencilBuffer"
gLAreaHasStencilBuffer = AttrLabelProxy

gLAreaUseEs :: AttrLabelProxy "useEs"
gLAreaUseEs = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList GLArea = GLAreaSignalList
type GLAreaSignalList = ('[ '("createContext", GLAreaCreateContextSignalInfo), '("destroy", Gtk.Widget.WidgetDestroySignalInfo), '("directionChanged", Gtk.Widget.WidgetDirectionChangedSignalInfo), '("hide", Gtk.Widget.WidgetHideSignalInfo), '("keynavFailed", Gtk.Widget.WidgetKeynavFailedSignalInfo), '("map", Gtk.Widget.WidgetMapSignalInfo), '("mnemonicActivate", Gtk.Widget.WidgetMnemonicActivateSignalInfo), '("moveFocus", Gtk.Widget.WidgetMoveFocusSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("queryTooltip", Gtk.Widget.WidgetQueryTooltipSignalInfo), '("realize", Gtk.Widget.WidgetRealizeSignalInfo), '("render", GLAreaRenderSignalInfo), '("resize", GLAreaResizeSignalInfo), '("show", Gtk.Widget.WidgetShowSignalInfo), '("stateFlagsChanged", Gtk.Widget.WidgetStateFlagsChangedSignalInfo), '("unmap", Gtk.Widget.WidgetUnmapSignalInfo), '("unrealize", Gtk.Widget.WidgetUnrealizeSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method GLArea::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "GLArea" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_new" gtk_gl_area_new :: 
    IO (Ptr GLArea)

-- | Creates a new @GtkGLArea@ widget.
gLAreaNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m GLArea
    -- ^ __Returns:__ a new @GtkGLArea@
gLAreaNew  = liftIO $ do
    result <- gtk_gl_area_new
    checkUnexpectedReturnNULL "gLAreaNew" result
    result' <- (newObject GLArea) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method GLArea::attach_buffers
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_attach_buffers" gtk_gl_area_attach_buffers :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO ()

-- | Binds buffers to the framebuffer.
-- 
-- Ensures that the /@area@/ framebuffer object is made the current draw
-- and read target, and that all the required buffers for the /@area@/
-- are created and bound to the framebuffer.
-- 
-- This function is automatically called before emitting the
-- [GLArea::render]("GI.Gtk.Objects.GLArea#g:signal:render") signal, and doesn\'t normally need to be
-- called by application code.
gLAreaAttachBuffers ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m ()
gLAreaAttachBuffers area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    gtk_gl_area_attach_buffers area'
    touchManagedPtr area
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaAttachBuffersMethodInfo
instance (signature ~ (m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaAttachBuffersMethodInfo a signature where
    overloadedMethod = gLAreaAttachBuffers

instance O.OverloadedMethodInfo GLAreaAttachBuffersMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaAttachBuffers",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaAttachBuffers"
        })


#endif

-- method GLArea::get_allowed_apis
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "GLAPI" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_get_allowed_apis" gtk_gl_area_get_allowed_apis :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO CUInt

-- | Gets the allowed APIs.
-- 
-- See 'GI.Gtk.Objects.GLArea.gLAreaSetAllowedApis'.
-- 
-- /Since: 4.12/
gLAreaGetAllowedApis ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m [Gdk.Flags.GLAPI]
    -- ^ __Returns:__ the allowed APIs
gLAreaGetAllowedApis area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    result <- gtk_gl_area_get_allowed_apis area'
    let result' = wordToGFlags result
    touchManagedPtr area
    return result'

#if defined(ENABLE_OVERLOADING)
data GLAreaGetAllowedApisMethodInfo
instance (signature ~ (m [Gdk.Flags.GLAPI]), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaGetAllowedApisMethodInfo a signature where
    overloadedMethod = gLAreaGetAllowedApis

instance O.OverloadedMethodInfo GLAreaGetAllowedApisMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaGetAllowedApis",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaGetAllowedApis"
        })


#endif

-- method GLArea::get_api
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "GLAPI" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_get_api" gtk_gl_area_get_api :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO CUInt

-- | Gets the API that is currently in use.
-- 
-- If the GL area has not been realized yet, 0 is returned.
-- 
-- /Since: 4.12/
gLAreaGetApi ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m [Gdk.Flags.GLAPI]
    -- ^ __Returns:__ the currently used API
gLAreaGetApi area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    result <- gtk_gl_area_get_api area'
    let result' = wordToGFlags result
    touchManagedPtr area
    return result'

#if defined(ENABLE_OVERLOADING)
data GLAreaGetApiMethodInfo
instance (signature ~ (m [Gdk.Flags.GLAPI]), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaGetApiMethodInfo a signature where
    overloadedMethod = gLAreaGetApi

instance O.OverloadedMethodInfo GLAreaGetApiMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaGetApi",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaGetApi"
        })


#endif

-- method GLArea::get_auto_render
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_get_auto_render" gtk_gl_area_get_auto_render :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO CInt

-- | Returns whether the area is in auto render mode or not.
gLAreaGetAutoRender ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@area@/ is auto rendering, 'P.False' otherwise
gLAreaGetAutoRender area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    result <- gtk_gl_area_get_auto_render area'
    let result' = (/= 0) result
    touchManagedPtr area
    return result'

#if defined(ENABLE_OVERLOADING)
data GLAreaGetAutoRenderMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaGetAutoRenderMethodInfo a signature where
    overloadedMethod = gLAreaGetAutoRender

instance O.OverloadedMethodInfo GLAreaGetAutoRenderMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaGetAutoRender",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaGetAutoRender"
        })


#endif

-- method GLArea::get_context
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "GLContext" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_get_context" gtk_gl_area_get_context :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO (Ptr Gdk.GLContext.GLContext)

-- | Retrieves the @GdkGLContext@ used by /@area@/.
gLAreaGetContext ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m (Maybe Gdk.GLContext.GLContext)
    -- ^ __Returns:__ the @GdkGLContext@
gLAreaGetContext area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    result <- gtk_gl_area_get_context area'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gdk.GLContext.GLContext) result'
        return result''
    touchManagedPtr area
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data GLAreaGetContextMethodInfo
instance (signature ~ (m (Maybe Gdk.GLContext.GLContext)), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaGetContextMethodInfo a signature where
    overloadedMethod = gLAreaGetContext

instance O.OverloadedMethodInfo GLAreaGetContextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaGetContext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaGetContext"
        })


#endif

-- method GLArea::get_error
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TError
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_get_error" gtk_gl_area_get_error :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO (Ptr GError)

-- | Gets the current error set on the /@area@/.
gLAreaGetError ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m (Maybe GError)
    -- ^ __Returns:__ the @GError@
gLAreaGetError area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    result <- gtk_gl_area_get_error area'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed GError) result'
        return result''
    touchManagedPtr area
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data GLAreaGetErrorMethodInfo
instance (signature ~ (m (Maybe GError)), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaGetErrorMethodInfo a signature where
    overloadedMethod = gLAreaGetError

instance O.OverloadedMethodInfo GLAreaGetErrorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaGetError",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaGetError"
        })


#endif

-- method GLArea::get_has_depth_buffer
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_get_has_depth_buffer" gtk_gl_area_get_has_depth_buffer :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO CInt

-- | Returns whether the area has a depth buffer.
gLAreaGetHasDepthBuffer ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@area@/ has a depth buffer, 'P.False' otherwise
gLAreaGetHasDepthBuffer area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    result <- gtk_gl_area_get_has_depth_buffer area'
    let result' = (/= 0) result
    touchManagedPtr area
    return result'

#if defined(ENABLE_OVERLOADING)
data GLAreaGetHasDepthBufferMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaGetHasDepthBufferMethodInfo a signature where
    overloadedMethod = gLAreaGetHasDepthBuffer

instance O.OverloadedMethodInfo GLAreaGetHasDepthBufferMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaGetHasDepthBuffer",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaGetHasDepthBuffer"
        })


#endif

-- method GLArea::get_has_stencil_buffer
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_get_has_stencil_buffer" gtk_gl_area_get_has_stencil_buffer :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO CInt

-- | Returns whether the area has a stencil buffer.
gLAreaGetHasStencilBuffer ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the /@area@/ has a stencil buffer, 'P.False' otherwise
gLAreaGetHasStencilBuffer area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    result <- gtk_gl_area_get_has_stencil_buffer area'
    let result' = (/= 0) result
    touchManagedPtr area
    return result'

#if defined(ENABLE_OVERLOADING)
data GLAreaGetHasStencilBufferMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaGetHasStencilBufferMethodInfo a signature where
    overloadedMethod = gLAreaGetHasStencilBuffer

instance O.OverloadedMethodInfo GLAreaGetHasStencilBufferMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaGetHasStencilBuffer",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaGetHasStencilBuffer"
        })


#endif

-- method GLArea::get_required_version
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "major"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the required major version"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "minor"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the required minor version"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_get_required_version" gtk_gl_area_get_required_version :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    Ptr Int32 ->                            -- major : TBasicType TInt
    Ptr Int32 ->                            -- minor : TBasicType TInt
    IO ()

-- | Retrieves the required version of OpenGL.
-- 
-- See 'GI.Gtk.Objects.GLArea.gLAreaSetRequiredVersion'.
gLAreaGetRequiredVersion ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m ((Int32, Int32))
gLAreaGetRequiredVersion area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    major <- allocMem :: IO (Ptr Int32)
    minor <- allocMem :: IO (Ptr Int32)
    gtk_gl_area_get_required_version area' major minor
    major' <- peek major
    minor' <- peek minor
    touchManagedPtr area
    freeMem major
    freeMem minor
    return (major', minor')

#if defined(ENABLE_OVERLOADING)
data GLAreaGetRequiredVersionMethodInfo
instance (signature ~ (m ((Int32, Int32))), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaGetRequiredVersionMethodInfo a signature where
    overloadedMethod = gLAreaGetRequiredVersion

instance O.OverloadedMethodInfo GLAreaGetRequiredVersionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaGetRequiredVersion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaGetRequiredVersion"
        })


#endif

-- method GLArea::get_use_es
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_get_use_es" gtk_gl_area_get_use_es :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO CInt

{-# DEPRECATED gLAreaGetUseEs ["(Since version 4.12)","Use 'GI.Gtk.Objects.GLArea.gLAreaGetApi'"] #-}
-- | Returns whether the @GtkGLArea@ should use OpenGL ES.
-- 
-- See 'GI.Gtk.Objects.GLArea.gLAreaSetUseEs'.
gLAreaGetUseEs ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the @GtkGLArea@ should create an OpenGL ES context
    --   and 'P.False' otherwise
gLAreaGetUseEs area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    result <- gtk_gl_area_get_use_es area'
    let result' = (/= 0) result
    touchManagedPtr area
    return result'

#if defined(ENABLE_OVERLOADING)
data GLAreaGetUseEsMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaGetUseEsMethodInfo a signature where
    overloadedMethod = gLAreaGetUseEs

instance O.OverloadedMethodInfo GLAreaGetUseEsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaGetUseEs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaGetUseEs"
        })


#endif

-- method GLArea::make_current
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_make_current" gtk_gl_area_make_current :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO ()

-- | Ensures that the @GdkGLContext@ used by /@area@/ is associated with
-- the @GtkGLArea@.
-- 
-- This function is automatically called before emitting the
-- [GLArea::render]("GI.Gtk.Objects.GLArea#g:signal:render") signal, and doesn\'t normally need
-- to be called by application code.
gLAreaMakeCurrent ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m ()
gLAreaMakeCurrent area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    gtk_gl_area_make_current area'
    touchManagedPtr area
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaMakeCurrentMethodInfo
instance (signature ~ (m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaMakeCurrentMethodInfo a signature where
    overloadedMethod = gLAreaMakeCurrent

instance O.OverloadedMethodInfo GLAreaMakeCurrentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaMakeCurrent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaMakeCurrent"
        })


#endif

-- method GLArea::queue_render
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_queue_render" gtk_gl_area_queue_render :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    IO ()

-- | Marks the currently rendered data (if any) as invalid, and queues
-- a redraw of the widget.
-- 
-- This ensures that the [GLArea::render]("GI.Gtk.Objects.GLArea#g:signal:render") signal
-- is emitted during the draw.
-- 
-- This is only needed when 'GI.Gtk.Objects.GLArea.gLAreaSetAutoRender' has
-- been called with a 'P.False' value. The default behaviour is to
-- emit [GLArea::render]("GI.Gtk.Objects.GLArea#g:signal:render") on each draw.
gLAreaQueueRender ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> m ()
gLAreaQueueRender area = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    gtk_gl_area_queue_render area'
    touchManagedPtr area
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaQueueRenderMethodInfo
instance (signature ~ (m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaQueueRenderMethodInfo a signature where
    overloadedMethod = gLAreaQueueRender

instance O.OverloadedMethodInfo GLAreaQueueRenderMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaQueueRender",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaQueueRender"
        })


#endif

-- method GLArea::set_allowed_apis
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "apis"
--           , argType = TInterface Name { namespace = "Gdk" , name = "GLAPI" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the allowed APIs" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_set_allowed_apis" gtk_gl_area_set_allowed_apis :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    CUInt ->                                -- apis : TInterface (Name {namespace = "Gdk", name = "GLAPI"})
    IO ()

-- | Sets the allowed APIs to create a context with.
-- 
-- You should check [GLArea:api]("GI.Gtk.Objects.GLArea#g:attr:api") before drawing
-- with either API.
-- 
-- By default, all APIs are allowed.
-- 
-- /Since: 4.12/
gLAreaSetAllowedApis ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> [Gdk.Flags.GLAPI]
    -- ^ /@apis@/: the allowed APIs
    -> m ()
gLAreaSetAllowedApis area apis = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    let apis' = gflagsToWord apis
    gtk_gl_area_set_allowed_apis area' apis'
    touchManagedPtr area
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaSetAllowedApisMethodInfo
instance (signature ~ ([Gdk.Flags.GLAPI] -> m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaSetAllowedApisMethodInfo a signature where
    overloadedMethod = gLAreaSetAllowedApis

instance O.OverloadedMethodInfo GLAreaSetAllowedApisMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaSetAllowedApis",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaSetAllowedApis"
        })


#endif

-- method GLArea::set_auto_render
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "auto_render"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a boolean" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_set_auto_render" gtk_gl_area_set_auto_render :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    CInt ->                                 -- auto_render : TBasicType TBoolean
    IO ()

-- | Sets whether the @GtkGLArea@ is in auto render mode.
-- 
-- If /@autoRender@/ is 'P.True' the [GLArea::render]("GI.Gtk.Objects.GLArea#g:signal:render") signal will
-- be emitted every time the widget draws. This is the default and is
-- useful if drawing the widget is faster.
-- 
-- If /@autoRender@/ is 'P.False' the data from previous rendering is kept
-- around and will be used for drawing the widget the next time,
-- unless the window is resized. In order to force a rendering
-- 'GI.Gtk.Objects.GLArea.gLAreaQueueRender' must be called. This mode is
-- useful when the scene changes seldom, but takes a long time to redraw.
gLAreaSetAutoRender ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> Bool
    -- ^ /@autoRender@/: a boolean
    -> m ()
gLAreaSetAutoRender area autoRender = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    let autoRender' = (P.fromIntegral . P.fromEnum) autoRender
    gtk_gl_area_set_auto_render area' autoRender'
    touchManagedPtr area
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaSetAutoRenderMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaSetAutoRenderMethodInfo a signature where
    overloadedMethod = gLAreaSetAutoRender

instance O.OverloadedMethodInfo GLAreaSetAutoRenderMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaSetAutoRender",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaSetAutoRender"
        })


#endif

-- method GLArea::set_error
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "error"
--           , argType = TError
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a new `GError`, or %NULL to unset the error"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_set_error" gtk_gl_area_set_error :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    Ptr GError ->                           -- error : TError
    IO ()

-- | Sets an error on the area which will be shown instead of the
-- GL rendering.
-- 
-- This is useful in the [GLArea::createContext]("GI.Gtk.Objects.GLArea#g:signal:createContext")
-- signal if GL context creation fails.
gLAreaSetError ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> Maybe (GError)
    -- ^ /@error@/: a new @GError@, or 'P.Nothing' to unset the error
    -> m ()
gLAreaSetError area error_ = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    maybeError_ <- case error_ of
        Nothing -> return FP.nullPtr
        Just jError_ -> do
            jError_' <- unsafeManagedPtrGetPtr jError_
            return jError_'
    gtk_gl_area_set_error area' maybeError_
    touchManagedPtr area
    whenJust error_ touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaSetErrorMethodInfo
instance (signature ~ (Maybe (GError) -> m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaSetErrorMethodInfo a signature where
    overloadedMethod = gLAreaSetError

instance O.OverloadedMethodInfo GLAreaSetErrorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaSetError",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaSetError"
        })


#endif

-- method GLArea::set_has_depth_buffer
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "has_depth_buffer"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE to add a depth buffer"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_set_has_depth_buffer" gtk_gl_area_set_has_depth_buffer :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    CInt ->                                 -- has_depth_buffer : TBasicType TBoolean
    IO ()

-- | Sets whether the @GtkGLArea@ should use a depth buffer.
-- 
-- If /@hasDepthBuffer@/ is 'P.True' the widget will allocate and
-- enable a depth buffer for the target framebuffer. Otherwise
-- there will be none.
gLAreaSetHasDepthBuffer ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> Bool
    -- ^ /@hasDepthBuffer@/: 'P.True' to add a depth buffer
    -> m ()
gLAreaSetHasDepthBuffer area hasDepthBuffer = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    let hasDepthBuffer' = (P.fromIntegral . P.fromEnum) hasDepthBuffer
    gtk_gl_area_set_has_depth_buffer area' hasDepthBuffer'
    touchManagedPtr area
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaSetHasDepthBufferMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaSetHasDepthBufferMethodInfo a signature where
    overloadedMethod = gLAreaSetHasDepthBuffer

instance O.OverloadedMethodInfo GLAreaSetHasDepthBufferMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaSetHasDepthBuffer",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaSetHasDepthBuffer"
        })


#endif

-- method GLArea::set_has_stencil_buffer
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "has_stencil_buffer"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE to add a stencil buffer"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_set_has_stencil_buffer" gtk_gl_area_set_has_stencil_buffer :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    CInt ->                                 -- has_stencil_buffer : TBasicType TBoolean
    IO ()

-- | Sets whether the @GtkGLArea@ should use a stencil buffer.
-- 
-- If /@hasStencilBuffer@/ is 'P.True' the widget will allocate and
-- enable a stencil buffer for the target framebuffer. Otherwise
-- there will be none.
gLAreaSetHasStencilBuffer ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> Bool
    -- ^ /@hasStencilBuffer@/: 'P.True' to add a stencil buffer
    -> m ()
gLAreaSetHasStencilBuffer area hasStencilBuffer = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    let hasStencilBuffer' = (P.fromIntegral . P.fromEnum) hasStencilBuffer
    gtk_gl_area_set_has_stencil_buffer area' hasStencilBuffer'
    touchManagedPtr area
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaSetHasStencilBufferMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaSetHasStencilBufferMethodInfo a signature where
    overloadedMethod = gLAreaSetHasStencilBuffer

instance O.OverloadedMethodInfo GLAreaSetHasStencilBufferMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaSetHasStencilBuffer",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaSetHasStencilBuffer"
        })


#endif

-- method GLArea::set_required_version
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "major"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the major version" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minor"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the minor version" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_set_required_version" gtk_gl_area_set_required_version :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    Int32 ->                                -- major : TBasicType TInt
    Int32 ->                                -- minor : TBasicType TInt
    IO ()

-- | Sets the required version of OpenGL to be used when creating
-- the context for the widget.
-- 
-- This function must be called before the area has been realized.
gLAreaSetRequiredVersion ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> Int32
    -- ^ /@major@/: the major version
    -> Int32
    -- ^ /@minor@/: the minor version
    -> m ()
gLAreaSetRequiredVersion area major minor = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    gtk_gl_area_set_required_version area' major minor
    touchManagedPtr area
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaSetRequiredVersionMethodInfo
instance (signature ~ (Int32 -> Int32 -> m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaSetRequiredVersionMethodInfo a signature where
    overloadedMethod = gLAreaSetRequiredVersion

instance O.OverloadedMethodInfo GLAreaSetRequiredVersionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaSetRequiredVersion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaSetRequiredVersion"
        })


#endif

-- method GLArea::set_use_es
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "area"
--           , argType = TInterface Name { namespace = "Gtk" , name = "GLArea" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGLArea`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "use_es"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether to use OpenGL or OpenGL ES"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gl_area_set_use_es" gtk_gl_area_set_use_es :: 
    Ptr GLArea ->                           -- area : TInterface (Name {namespace = "Gtk", name = "GLArea"})
    CInt ->                                 -- use_es : TBasicType TBoolean
    IO ()

{-# DEPRECATED gLAreaSetUseEs ["(Since version 4.12)","Use 'GI.Gtk.Objects.GLArea.gLAreaSetAllowedApis'"] #-}
-- | Sets whether the /@area@/ should create an OpenGL or an OpenGL ES context.
-- 
-- You should check the capabilities of the @GdkGLContext@ before drawing
-- with either API.
gLAreaSetUseEs ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLArea a) =>
    a
    -- ^ /@area@/: a @GtkGLArea@
    -> Bool
    -- ^ /@useEs@/: whether to use OpenGL or OpenGL ES
    -> m ()
gLAreaSetUseEs area useEs = liftIO $ do
    area' <- unsafeManagedPtrCastPtr area
    let useEs' = (P.fromIntegral . P.fromEnum) useEs
    gtk_gl_area_set_use_es area' useEs'
    touchManagedPtr area
    return ()

#if defined(ENABLE_OVERLOADING)
data GLAreaSetUseEsMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsGLArea a) => O.OverloadedMethod GLAreaSetUseEsMethodInfo a signature where
    overloadedMethod = gLAreaSetUseEs

instance O.OverloadedMethodInfo GLAreaSetUseEsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.GLArea.gLAreaSetUseEs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-GLArea.html#v:gLAreaSetUseEs"
        })


#endif


