// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A0__F328

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_192_3_F_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x7311f620u, 0x213f6d8fu, 0, 86 }, // 7311f620213f6d8f = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__16_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x22ba27dau, 0x8f9d8de1u, 0, 46 }, // 22ba27da8f9d8de1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xc6a7c77du, 0xb049b25du, 0, 103 }, // c6a7c77db049b25d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x32358779u, 0x46769019u, 40, 6 }, // 3235877946769019 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x1c1d56a0u, 0x5c6bca3au, 40, 86 }, // 1c1d56a05c6bca3a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x36261d2cu, 0xd7202d8bu, 80, 126 }, // 36261d2cd7202d8b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x334d981au, 0x6e1efe79u, 80, 155 }, // 334d981a6e1efe79 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__32_32__CO__wave1_warp4_stg1--Arch_gfx950
    { 0xe4aee706u, 0xd893d7e9u, 120, 155 }, // e4aee706d893d7e9 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__64_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x3a8cf9b8u, 0x1475d381u, 120, 103 }, // 3a8cf9b81475d381 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__64_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x11762425u, 0xb7f79ef5u, 143, 126 }, // 11762425b7f79ef5 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x02d74980u, 0xc37ada07u, 149, 155 }, // 02d74980c37ada07 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_F_F_0___gfx950__P__64_64__CO__wave1_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{ 4, 2, 2, 1, 2, 1, 3, 7, 2, 3},
 { 2, 0, 2, 2, 0, 2, 0, 2, 2, 0},
 { 0, 0, 0, 6, 6, 2, 6, 2, 2, 2},
 { 2, 0, 0, 5, 5, 5, 5, 5, 5, 5},
 { 1, 1, 1, 3, 9, 9, 9, 9, 9, 9},
 { 1, 1, 8, 8, 9, 9, 9, 9, 9, 9},
 { 1, 1, 8, 9, 9, 9, 9, 9, 9,10},
 { 1, 1, 8, 9, 9, 9, 9, 9,10,10},
 { 1, 1, 8, 8, 9, 9, 9,10,10,10},
 { 1, 8, 8, 8, 9, 8, 9, 9, 9, 9}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A0__F328 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 4;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 4;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = "i32"
// BLOCK_DMODEL = 192
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 0

// vim: set fileencoding=utf-8

