﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/connect/ConnectRequest.h>
#include <aws/connect/Connect_EXPORTS.h>
#include <aws/connect/model/MonitorCapability.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Connect {
namespace Model {

/**
 */
class MonitorContactRequest : public ConnectRequest {
 public:
  AWS_CONNECT_API MonitorContactRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "MonitorContact"; }

  AWS_CONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the Amazon Connect instance. You can find the instanceId in
   * the ARN of the instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  MonitorContactRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the contact.</p>
   */
  inline const Aws::String& GetContactId() const { return m_contactId; }
  inline bool ContactIdHasBeenSet() const { return m_contactIdHasBeenSet; }
  template <typename ContactIdT = Aws::String>
  void SetContactId(ContactIdT&& value) {
    m_contactIdHasBeenSet = true;
    m_contactId = std::forward<ContactIdT>(value);
  }
  template <typename ContactIdT = Aws::String>
  MonitorContactRequest& WithContactId(ContactIdT&& value) {
    SetContactId(std::forward<ContactIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the user account.</p>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  MonitorContactRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify which monitoring actions the user is allowed to take. For example,
   * whether the user is allowed to escalate from silent monitoring to barge.
   * AllowedMonitorCapabilities is required if barge is enabled.</p>
   */
  inline const Aws::Vector<MonitorCapability>& GetAllowedMonitorCapabilities() const { return m_allowedMonitorCapabilities; }
  inline bool AllowedMonitorCapabilitiesHasBeenSet() const { return m_allowedMonitorCapabilitiesHasBeenSet; }
  template <typename AllowedMonitorCapabilitiesT = Aws::Vector<MonitorCapability>>
  void SetAllowedMonitorCapabilities(AllowedMonitorCapabilitiesT&& value) {
    m_allowedMonitorCapabilitiesHasBeenSet = true;
    m_allowedMonitorCapabilities = std::forward<AllowedMonitorCapabilitiesT>(value);
  }
  template <typename AllowedMonitorCapabilitiesT = Aws::Vector<MonitorCapability>>
  MonitorContactRequest& WithAllowedMonitorCapabilities(AllowedMonitorCapabilitiesT&& value) {
    SetAllowedMonitorCapabilities(std::forward<AllowedMonitorCapabilitiesT>(value));
    return *this;
  }
  inline MonitorContactRequest& AddAllowedMonitorCapabilities(MonitorCapability value) {
    m_allowedMonitorCapabilitiesHasBeenSet = true;
    m_allowedMonitorCapabilities.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If not provided, the Amazon Web Services SDK
   * populates this field. For more information about idempotency, see <a
   * href="https://aws.amazon.com/builders-library/making-retries-safe-with-idempotent-APIs/">Making
   * retries safe with idempotent APIs</a>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  MonitorContactRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceId;
  bool m_instanceIdHasBeenSet = false;

  Aws::String m_contactId;
  bool m_contactIdHasBeenSet = false;

  Aws::String m_userId;
  bool m_userIdHasBeenSet = false;

  Aws::Vector<MonitorCapability> m_allowedMonitorCapabilities;
  bool m_allowedMonitorCapabilitiesHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace Connect
}  // namespace Aws
