﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Array.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/stream/ResponseStream.h>
#include <aws/location/LocationService_EXPORTS.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace LocationService {
namespace Model {
class GetMapTileResult {
 public:
  AWS_LOCATIONSERVICE_API GetMapTileResult() = default;
  AWS_LOCATIONSERVICE_API GetMapTileResult(GetMapTileResult&&) = default;
  AWS_LOCATIONSERVICE_API GetMapTileResult& operator=(GetMapTileResult&&) = default;
  // we delete these because Microsoft doesn't handle move generation correctly
  // and we therefore don't trust them to get it right here either.
  GetMapTileResult(const GetMapTileResult&) = delete;
  GetMapTileResult& operator=(const GetMapTileResult&) = delete;

  AWS_LOCATIONSERVICE_API GetMapTileResult(Aws::AmazonWebServiceResult<Aws::Utils::Stream::ResponseStream>&& result);
  AWS_LOCATIONSERVICE_API GetMapTileResult& operator=(Aws::AmazonWebServiceResult<Aws::Utils::Stream::ResponseStream>&& result);

  ///@{
  /**
   * <p>Contains Mapbox Vector Tile (MVT) data.</p>
   */
  inline Aws::IOStream& GetBlob() const { return m_blob.GetUnderlyingStream(); }
  inline void ReplaceBody(Aws::IOStream* body) { m_blob = Aws::Utils::Stream::ResponseStream(body); }

  ///@}

  ///@{
  /**
   * <p>The map tile's content type. For example,
   * <code>application/vnd.mapbox-vector-tile</code>.</p>
   */
  inline const Aws::String& GetContentType() const { return m_contentType; }
  template <typename ContentTypeT = Aws::String>
  void SetContentType(ContentTypeT&& value) {
    m_contentTypeHasBeenSet = true;
    m_contentType = std::forward<ContentTypeT>(value);
  }
  template <typename ContentTypeT = Aws::String>
  GetMapTileResult& WithContentType(ContentTypeT&& value) {
    SetContentType(std::forward<ContentTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The HTTP Cache-Control directive for the value.</p>
   */
  inline const Aws::String& GetCacheControl() const { return m_cacheControl; }
  template <typename CacheControlT = Aws::String>
  void SetCacheControl(CacheControlT&& value) {
    m_cacheControlHasBeenSet = true;
    m_cacheControl = std::forward<CacheControlT>(value);
  }
  template <typename CacheControlT = Aws::String>
  GetMapTileResult& WithCacheControl(CacheControlT&& value) {
    SetCacheControl(std::forward<CacheControlT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  GetMapTileResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Utils::Stream::ResponseStream m_blob{};

  Aws::String m_contentType;

  Aws::String m_cacheControl;

  Aws::String m_requestId;
  bool m_blobHasBeenSet = false;
  bool m_contentTypeHasBeenSet = false;
  bool m_cacheControlHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace LocationService
}  // namespace Aws
