/*
 * nUFRaw - Unidentified Flying Raw converter for digital camera images
 *
 * nufraw_message.c - Error message handling functions
 * Copyright 2004-2016 by Udi Fuchs
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "nufraw.h"
#include <string.h>

/*
 * Every nufraw internal function that might fail should return a status
 * with one of these values:
 * NUFRAW_SUCCESS
 * NUFRAW_WARNING
 * NUFRAW_ERROR
 * The relevant message can be retrived using nufraw_get_message().
 * Even when NUFRAW_SUCCESS is returned there could be an information message.
 */
char *nufraw_get_message(nufraw_data *uf)
{
    return uf->message;
}

static void message_append(nufraw_data *uf, char *message)
{
    if (message == NULL) return;
    if (uf->message == NULL) {
        uf->message = g_strdup(message);
        return;
    }
    if (uf->message[strlen(uf->message) - 1] == '\n')
        uf->message = g_strconcat(uf->message, message, NULL);
    else
        uf->message = g_strconcat(uf->message, "\n", message, NULL);
}


/* The following function should be used by nufraw's internal functions. */
void nufraw_message_init(nufraw_data *uf)
{
    uf->status = NUFRAW_SUCCESS;
    uf->message = NULL;
}

void nufraw_message_reset(nufraw_data *uf)
{
    uf->status = NUFRAW_SUCCESS;
    g_free(uf->message);
    uf->message = NULL;
}

void nufraw_set_error(nufraw_data *uf, const char *format, ...)
{
    uf->status = NUFRAW_ERROR;
    if (format != NULL) {
        va_list ap;
        va_start(ap, format);
        char *message = g_strdup_vprintf(format, ap);
        va_end(ap);
        message_append(uf, message);
        g_free(message);
    }
}

void nufraw_set_warning(nufraw_data *uf, const char *format, ...)
{
    // Set warning only if no error was set before
    if (uf->status != NUFRAW_ERROR) uf->status = NUFRAW_WARNING;
    if (format != NULL) {
        va_list ap;
        va_start(ap, format);
        char *message = g_strdup_vprintf(format, ap);
        va_end(ap);
        message_append(uf, message);
        g_free(message);
    }
}

void nufraw_set_info(nufraw_data *uf, const char *format, ...)
{
    if (format != NULL) {
        va_list ap;
        va_start(ap, format);
        char *message = g_strdup_vprintf(format, ap);
        va_end(ap);
        message_append(uf, message);
        g_free(message);
    }
}

int nufraw_get_status(nufraw_data *uf)
{
    return uf->status;
}

int nufraw_is_error(nufraw_data *uf)
{
    return uf->status == NUFRAW_ERROR;
}

// Old error handling, should be removed after being fully implemented.

static char *nufraw_message_buffer(char *buffer, char *message)
{
#ifdef NUFRAW_DEBUG
    nufraw_batch_messenger(message);
#endif
    char *buf;
    if (buffer == NULL) return g_strdup(message);
    buf = g_strconcat(buffer, message, NULL);
    g_free(buffer);
    return buf;
}

void nufraw_batch_messenger(char *message)
{
    /* Print the 'nufraw:' header only if there are no newlines in the message
     * (not including possibly one at the end).
     * Otherwise, the header will be printed only for the first line. */
    if (g_strstr_len(message, strlen(message) - 1, "\n") == NULL)
        g_printerr("%s: ", nufraw_binary);
    g_printerr("%s%c", message, message[strlen(message) - 1] != '\n' ? '\n' : 0);
}

char *nufraw_message(int code, const char *format, ...)
{
    // TODO: The following static variables are not thread-safe
    static char *logBuffer = NULL;
    static char *errorBuffer = NULL;
    static gboolean errorFlag = FALSE;
    static void *parentWindow = NULL;
    char *message = NULL;
    void *saveParentWindow;

    if (code == NUFRAW_SET_PARENT) {
        saveParentWindow = parentWindow;
        parentWindow = (void *)format;
        return saveParentWindow;
    }
    if (format != NULL) {
        va_list ap;
        va_start(ap, format);
        message = g_strdup_vprintf(format, ap);
        va_end(ap);
    }
    switch (code) {
        case NUFRAW_SET_ERROR:
            errorFlag = TRUE;
        case NUFRAW_SET_WARNING:
            errorBuffer = nufraw_message_buffer(errorBuffer, message);
        case NUFRAW_SET_LOG:
        case NUFRAW_DCRAW_SET_LOG:
            logBuffer = nufraw_message_buffer(logBuffer, message);
            g_free(message);
            return NULL;
        case NUFRAW_GET_ERROR:
            if (!errorFlag) return NULL;
        case NUFRAW_GET_WARNING:
            return errorBuffer;
        case NUFRAW_GET_LOG:
            return logBuffer;
        case NUFRAW_CLEAN:
            g_free(logBuffer);
            logBuffer = NULL;
        case NUFRAW_RESET:
            g_free(errorBuffer);
            errorBuffer = NULL;
            errorFlag = FALSE;
            return NULL;
        case NUFRAW_BATCH_MESSAGE:
            if (parentWindow == NULL)
                nufraw_messenger(message, parentWindow);
            g_free(message);
            return NULL;
        case NUFRAW_INTERACTIVE_MESSAGE:
            if (parentWindow != NULL)
                nufraw_messenger(message, parentWindow);
            g_free(message);
            return NULL;
        case NUFRAW_REPORT:
            nufraw_messenger(errorBuffer, parentWindow);
            return NULL;
        default:
            nufraw_messenger(message, parentWindow);
            g_free(message);
            return NULL;
    }
}
