#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <exception>
#include <initializer_list>
#include <iterator>
#include <memory>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

class Opaque;

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) &noexcept;
  String &operator=(String &&) &noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_STR
#define CXXBRIDGE1_RUST_STR
class Str final {
public:
  Str() noexcept;
  Str(const String &) noexcept;
  Str(const std::string &);
  Str(const char *);
  Str(const char *, std::size_t);

  Str &operator=(const Str &) &noexcept = default;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  Str(const Str &) noexcept = default;
  ~Str() noexcept = default;

  using iterator = const char *;
  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const Str &) const noexcept;
  bool operator!=(const Str &) const noexcept;
  bool operator<(const Str &) const noexcept;
  bool operator<=(const Str &) const noexcept;
  bool operator>(const Str &) const noexcept;
  bool operator>=(const Str &) const noexcept;

  void swap(Str &) noexcept;

private:
  class uninit;
  Str(uninit) noexcept;
  friend impl<Str>;

  std::array<std::uintptr_t, 2> repr;
};
#endif // CXXBRIDGE1_RUST_STR

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) &noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) &noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C& c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) &noexcept = default;
  Slice &operator=(Slice<T> &&) &noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

template <typename T>
class Slice<T>::iterator final {
public:
  using iterator_category = std::random_access_iterator_tag;
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) &noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) &noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_RUST_ERROR
#define CXXBRIDGE1_RUST_ERROR
class Error final : public std::exception {
public:
  Error(const Error &);
  Error(Error &&) noexcept;
  ~Error() noexcept override;

  Error &operator=(const Error &) &;
  Error &operator=(Error &&) &noexcept;

  const char *what() const noexcept override;

private:
  Error() noexcept = default;
  friend impl<Error>;
  const char *msg;
  std::size_t len;
};
#endif // CXXBRIDGE1_RUST_ERROR

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT

namespace repr {
struct PtrLen final {
  void *ptr;
  ::std::size_t len;
};
} // namespace repr

namespace detail {
template <typename T, typename = void *>
struct operator_new {
  void *operator()(::std::size_t sz) { return ::operator new(sz); }
};

template <typename T>
struct operator_new<T, decltype(T::operator new(sizeof(T)))> {
  void *operator()(::std::size_t sz) { return T::operator new(sz); }
};
} // namespace detail

template <typename T>
union ManuallyDrop {
  T value;
  ManuallyDrop(T &&value) : value(::std::move(value)) {}
  ~ManuallyDrop() {}
};

template <typename T>
union MaybeUninit {
  T value;
  void *operator new(::std::size_t sz) { return detail::operator_new<T>{}(sz); }
  MaybeUninit() {}
  ~MaybeUninit() {}
};

namespace {
template <>
class impl<Error> final {
public:
  static Error error(repr::PtrLen repr) noexcept {
    Error error;
    error.msg = static_cast<char const *>(repr.ptr);
    error.len = repr.len;
    return error;
  }
};
} // namespace
} // namespace cxxbridge1
} // namespace rust

namespace pdns {
  namespace rust {
    namespace settings {
      namespace rec {
        struct ForwardZone;
        struct AuthZone;
        struct TrustAnchor;
        struct NegativeTrustAnchor;
        struct ProtobufServer;
        struct DNSTapFrameStreamServer;
        struct DNSTapNODFrameStreamServer;
        struct TSIGTriplet;
        struct RPZ;
        struct ZoneToCache;
        struct SubnetOrder;
        struct SortList;
        struct AllowedAdditionalQType;
        struct ProxyMapping;
        struct ApiZones;
        struct XFR;
        struct FCZDefault;
        struct ForwardingCatalogZone;
        struct IncomingTLS;
        struct IncomingWSConfig;
        struct Value;
        struct OldStyle;
        struct Dnssec;
        struct Incoming;
        struct Recursor;
        struct Webservice;
        struct Carbon;
        struct Outgoing;
        struct Packetcache;
        struct Logging;
        struct Ecs;
        struct Recordcache;
        struct Nod;
        struct Snmp;
        struct Recursorsettings;
      }
    }
  }
}

namespace pdns {
namespace rust {
namespace settings {
namespace rec {
#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardZone
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardZone
struct ForwardZone final {
  ::rust::String zone;
  ::rust::Vec<::rust::String> forwarders;
  bool recurse;
  bool notify_allowed;

  void validate(::rust::Str field) const;
  bool operator==(ForwardZone const &) const noexcept;
  bool operator!=(ForwardZone const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardZone

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AuthZone
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AuthZone
struct AuthZone final {
  ::rust::String zone;
  ::rust::String file;

  void validate(::rust::Str field) const;
  bool operator==(AuthZone const &) const noexcept;
  bool operator!=(AuthZone const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AuthZone

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TrustAnchor
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TrustAnchor
struct TrustAnchor final {
  ::rust::String name;
  ::rust::Vec<::rust::String> dsrecords;

  void validate(::rust::Str field) const;
  bool operator==(TrustAnchor const &) const noexcept;
  bool operator!=(TrustAnchor const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TrustAnchor

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$NegativeTrustAnchor
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$NegativeTrustAnchor
struct NegativeTrustAnchor final {
  ::rust::String name;
  ::rust::String reason;

  void validate(::rust::Str field) const;
  bool operator==(NegativeTrustAnchor const &) const noexcept;
  bool operator!=(NegativeTrustAnchor const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$NegativeTrustAnchor

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProtobufServer
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProtobufServer
struct ProtobufServer final {
  ::rust::Vec<::rust::String> servers;
  ::std::uint64_t timeout;
  ::std::uint64_t maxQueuedEntries;
  ::std::uint64_t reconnectWaitTime;
  bool taggedOnly;
  bool asyncConnect;
  bool logQueries;
  bool logResponses;
  ::rust::Vec<::rust::String> exportTypes;
  bool logMappedFrom;

  bool operator==(ProtobufServer const &) const noexcept;
  bool operator!=(ProtobufServer const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProtobufServer

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapFrameStreamServer
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapFrameStreamServer
struct DNSTapFrameStreamServer final {
  ::rust::Vec<::rust::String> servers;
  bool logQueries;
  bool logResponses;
  ::std::uint64_t bufferHint;
  ::std::uint64_t flushTimeout;
  ::std::uint64_t inputQueueSize;
  ::std::uint64_t outputQueueSize;
  ::std::uint64_t queueNotifyThreshold;
  ::std::uint64_t reopenInterval;

  bool operator==(DNSTapFrameStreamServer const &) const noexcept;
  bool operator!=(DNSTapFrameStreamServer const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapFrameStreamServer

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapNODFrameStreamServer
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapNODFrameStreamServer
struct DNSTapNODFrameStreamServer final {
  ::rust::Vec<::rust::String> servers;
  bool logNODs;
  bool logUDRs;
  ::std::uint64_t bufferHint;
  ::std::uint64_t flushTimeout;
  ::std::uint64_t inputQueueSize;
  ::std::uint64_t outputQueueSize;
  ::std::uint64_t queueNotifyThreshold;
  ::std::uint64_t reopenInterval;

  bool operator==(DNSTapNODFrameStreamServer const &) const noexcept;
  bool operator!=(DNSTapNODFrameStreamServer const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$DNSTapNODFrameStreamServer

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TSIGTriplet
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TSIGTriplet
struct TSIGTriplet final {
  ::rust::String name;
  ::rust::String algo;
  ::rust::String secret;

  bool operator==(TSIGTriplet const &) const noexcept;
  bool operator!=(TSIGTriplet const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$TSIGTriplet

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$RPZ
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$RPZ
struct RPZ final {
  ::rust::String name;
  ::rust::Vec<::rust::String> addresses;
  ::rust::String defcontent;
  ::rust::String defpol;
  bool defpolOverrideLocalData;
  ::std::uint32_t defttl;
  ::std::uint32_t extendedErrorCode;
  ::rust::String extendedErrorExtra;
  bool includeSOA;
  bool ignoreDuplicates;
  ::std::uint32_t maxTTL;
  ::rust::String policyName;
  ::rust::Vec<::rust::String> tags;
  bool overridesGettag;
  ::std::uint32_t zoneSizeHint;
  ::pdns::rust::settings::rec::TSIGTriplet tsig;
  ::std::uint32_t refresh;
  ::std::uint32_t maxReceivedMBytes;
  ::rust::String localAddress;
  ::std::uint32_t axfrTimeout;
  ::rust::String dumpFile;
  ::rust::String seedFile;

  bool operator==(RPZ const &) const noexcept;
  bool operator!=(RPZ const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$RPZ

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ZoneToCache
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ZoneToCache
struct ZoneToCache final {
  ::rust::String zone;
  ::rust::String method;
  ::rust::Vec<::rust::String> sources;
  ::std::uint64_t timeout;
  ::pdns::rust::settings::rec::TSIGTriplet tsig;
  ::std::uint64_t refreshPeriod;
  ::std::uint64_t retryOnErrorPeriod;
  ::std::uint64_t maxReceivedMBytes;
  ::rust::String localAddress;
  ::rust::String zonemd;
  ::rust::String dnssec;

  bool operator==(ZoneToCache const &) const noexcept;
  bool operator!=(ZoneToCache const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ZoneToCache

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SubnetOrder
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SubnetOrder
struct SubnetOrder final {
  ::rust::String subnet;
  ::std::uint32_t order;

  bool operator==(SubnetOrder const &) const noexcept;
  bool operator!=(SubnetOrder const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SubnetOrder

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SortList
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SortList
struct SortList final {
  ::rust::String key;
  ::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> subnets;

  bool operator==(SortList const &) const noexcept;
  bool operator!=(SortList const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$SortList

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AllowedAdditionalQType
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AllowedAdditionalQType
struct AllowedAdditionalQType final {
  ::rust::String qtype;
  ::rust::Vec<::rust::String> targets;
  ::rust::String mode;

  bool operator==(AllowedAdditionalQType const &) const noexcept;
  bool operator!=(AllowedAdditionalQType const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$AllowedAdditionalQType

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProxyMapping
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProxyMapping
struct ProxyMapping final {
  ::rust::String subnet;
  ::rust::String address;
  ::rust::Vec<::rust::String> domains;

  bool operator==(ProxyMapping const &) const noexcept;
  bool operator!=(ProxyMapping const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ProxyMapping

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ApiZones
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ApiZones
struct ApiZones final {
  ::rust::Vec<::pdns::rust::settings::rec::AuthZone> auth_zones;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> forward_zones;

  void validate(::rust::Str field) const;
  bool operator==(ApiZones const &) const noexcept;
  bool operator!=(ApiZones const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ApiZones

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$XFR
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$XFR
struct XFR final {
  ::rust::Vec<::rust::String> addresses;
  ::std::uint32_t zoneSizeHint;
  ::pdns::rust::settings::rec::TSIGTriplet tsig;
  ::std::uint32_t refresh;
  ::std::uint32_t maxReceivedMBytes;
  ::rust::String localAddress;
  ::std::uint32_t axfrTimeout;

  bool operator==(XFR const &) const noexcept;
  bool operator!=(XFR const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$XFR

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$FCZDefault
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$FCZDefault
struct FCZDefault final {
  ::rust::String name;
  ::rust::Vec<::rust::String> forwarders;
  bool recurse;
  bool notify_allowed;

  bool operator==(FCZDefault const &) const noexcept;
  bool operator!=(FCZDefault const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$FCZDefault

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardingCatalogZone
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardingCatalogZone
struct ForwardingCatalogZone final {
  ::rust::String zone;
  bool notify_allowed;
  ::pdns::rust::settings::rec::XFR xfr;
  ::rust::Vec<::pdns::rust::settings::rec::FCZDefault> groups;

  bool operator==(ForwardingCatalogZone const &) const noexcept;
  bool operator!=(ForwardingCatalogZone const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$ForwardingCatalogZone

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingTLS
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingTLS
struct IncomingTLS final {
  ::rust::String certificate;
  ::rust::String key;

  bool operator==(IncomingTLS const &) const noexcept;
  bool operator!=(IncomingTLS const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingTLS

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingWSConfig
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingWSConfig
struct IncomingWSConfig final {
  ::rust::Vec<::rust::String> addresses;
  ::pdns::rust::settings::rec::IncomingTLS tls;

  bool operator==(IncomingWSConfig const &) const noexcept;
  bool operator!=(IncomingWSConfig const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$IncomingWSConfig

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Value
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Value
struct Value final {
  bool bool_val;
  ::std::uint64_t u64_val;
  double f64_val;
  ::rust::String string_val;
  ::rust::Vec<::rust::String> vec_string_val;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> vec_forwardzone_val;
  ::rust::Vec<::pdns::rust::settings::rec::AuthZone> vec_authzone_val;
  ::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> vec_trustanchor_val;
  ::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> vec_negativetrustanchor_val;
  ::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> vec_protobufserver_val;
  ::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> vec_dnstap_framestream_server_val;
  ::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> vec_dnstap_nod_framestream_server_val;
  ::rust::Vec<::pdns::rust::settings::rec::RPZ> vec_rpz_val;
  ::rust::Vec<::pdns::rust::settings::rec::SortList> vec_sortlist_val;
  ::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> vec_zonetocache_val;
  ::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> vec_allowedadditionalqtype_val;
  ::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> vec_proxymapping_val;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> vec_forwardingcatalogzone_val;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Value

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$OldStyle
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$OldStyle
struct OldStyle final {
  ::rust::String section;
  ::rust::String name;
  ::rust::String old_name;
  ::rust::String type_name;
  ::pdns::rust::settings::rec::Value value;
  bool overriding;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$OldStyle

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Dnssec
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Dnssec
struct Dnssec final {
  ::std::uint64_t aggressive_nsec_cache_size;
  ::std::uint64_t aggressive_cache_min_nsec3_hit_ratio;
  ::rust::String validation;
  ::rust::Vec<::rust::String> disabled_algorithms;
  bool log_bogus;
  ::std::uint64_t nsec3_max_iterations;
  ::std::uint64_t max_rrsigs_per_record;
  ::std::uint64_t max_nsec3s_per_record;
  ::std::uint64_t max_signature_validations_per_query;
  ::std::uint64_t max_nsec3_hash_computations_per_query;
  ::std::uint64_t aggressive_cache_max_nsec3_hash_cost;
  ::std::uint64_t max_ds_per_zone;
  ::std::uint64_t max_dnskeys;
  ::std::uint64_t signature_inception_skew;
  ::rust::Vec<::rust::String> x_dnssec_names;
  ::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> trustanchors;
  ::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> negative_trustanchors;
  ::rust::String trustanchorfile;
  ::std::uint64_t trustanchorfile_interval;

  bool operator==(Dnssec const &) const noexcept;
  bool operator!=(Dnssec const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Dnssec

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Incoming
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Incoming
struct Incoming final {
  ::rust::Vec<::rust::String> allow_from;
  ::rust::String allow_from_file;
  ::rust::Vec<::rust::String> allow_notify_for;
  ::rust::String allow_notify_for_file;
  ::rust::Vec<::rust::String> allow_notify_from;
  ::rust::String allow_notify_from_file;
  bool allow_no_rd;
  ::std::uint64_t tcp_timeout;
  double distribution_load_factor;
  ::std::uint64_t distribution_pipe_buffer_size;
  ::std::uint64_t distributor_threads;
  ::rust::Vec<::rust::String> edns_padding_from;
  ::rust::String edns_padding_mode;
  ::std::uint64_t edns_padding_tag;
  bool gettag_needs_edns_options;
  ::rust::Vec<::rust::String> listen;
  ::std::uint64_t port;
  bool non_local_bind;
  ::std::uint64_t max_concurrent_requests_per_tcp_connection;
  ::std::uint64_t max_tcp_clients;
  ::std::uint64_t max_tcp_per_client;
  ::std::uint64_t max_tcp_queries_per_connection;
  ::std::uint64_t max_udp_queries_per_round;
  bool pdns_distributes_queries;
  ::rust::Vec<::rust::String> proxy_protocol_from;
  ::rust::Vec<::rust::String> proxy_protocol_exceptions;
  ::std::uint64_t proxy_protocol_maximum_size;
  bool reuseport;
  ::std::uint64_t tcp_fast_open;
  ::std::uint64_t udp_truncation_threshold;
  bool use_incoming_edns_subnet;
  ::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> proxymappings;

  bool operator==(Incoming const &) const noexcept;
  bool operator!=(Incoming const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Incoming

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursor
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursor
struct Recursor final {
  bool any_to_tcp;
  bool allow_trust_anchor_query;
  ::rust::Vec<::pdns::rust::settings::rec::AuthZone> auth_zones;
  ::rust::String chroot;
  ::rust::String config_dir;
  ::rust::String config_name;
  ::rust::String cpu_map;
  bool daemon;
  bool devonly_regression_test_mode;
  ::rust::String dns64_prefix;
  ::rust::String etc_hosts_file;
  ::std::uint64_t event_trace_enabled;
  bool export_etc_hosts;
  ::rust::String export_etc_hosts_search_suffix;
  bool extended_resolution_errors;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> forward_zones;
  ::rust::String forward_zones_file;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> forward_zones_recurse;
  ::rust::String hint_file;
  ::rust::Vec<::rust::String> ignore_unknown_settings;
  ::rust::String include_dir;
  ::std::uint64_t latency_statistic_size;
  ::rust::String lua_config_file;
  ::rust::String lua_global_include_dir;
  ::rust::String lua_dns_script;
  ::std::uint64_t lua_maintenance_interval;
  ::std::uint64_t max_chain_length;
  ::std::uint64_t max_include_depth;
  ::std::uint64_t max_generate_steps;
  ::std::uint64_t max_mthreads;
  ::std::uint64_t max_cnames_followed;
  ::std::uint64_t max_recursion_depth;
  ::std::uint64_t max_total_msec;
  ::std::uint64_t minimum_ttl_override;
  ::rust::String nothing_below_nxdomain;
  ::rust::String public_suffix_list_file;
  bool qname_minimization;
  ::std::uint64_t qname_max_minimize_count;
  ::std::uint64_t qname_minimize_one_label;
  bool root_nx_trust;
  bool save_parent_ns_set;
  ::rust::String security_poll_suffix;
  bool serve_rfc1918;
  bool serve_rfc6303;
  ::rust::String server_id;
  ::rust::String setgid;
  ::rust::String setuid;
  ::rust::String socket_dir;
  ::rust::String socket_group;
  ::rust::String socket_mode;
  ::rust::String socket_owner;
  ::std::uint64_t spoof_nearmiss_max;
  ::std::uint64_t stack_cache_size;
  ::std::uint64_t stack_size;
  ::rust::Vec<::rust::String> stats_api_disabled_list;
  ::rust::Vec<::rust::String> stats_carbon_disabled_list;
  ::rust::Vec<::rust::String> stats_rec_control_disabled_list;
  ::std::uint64_t stats_ringbuffer_entries;
  ::rust::Vec<::rust::String> stats_snmp_disabled_list;
  ::std::uint64_t threads;
  ::std::uint64_t tcp_threads;
  ::rust::String version_string;
  bool write_pid;
  ::std::uint64_t system_resolver_ttl;
  ::std::uint64_t system_resolver_interval;
  bool system_resolver_self_resolve_check;
  ::rust::Vec<::pdns::rust::settings::rec::SortList> sortlists;
  ::rust::Vec<::pdns::rust::settings::rec::RPZ> rpzs;
  ::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> allowed_additional_qtypes;
  ::rust::String lua_start_stop_script;
  ::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> forwarding_catalog_zones;

  bool operator==(Recursor const &) const noexcept;
  bool operator!=(Recursor const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursor

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Webservice
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Webservice
struct Webservice final {
  ::rust::String api_dir;
  ::rust::String api_key;
  bool webserver;
  ::rust::String address;
  ::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> listen;
  ::rust::Vec<::rust::String> allow_from;
  bool hash_plaintext_credentials;
  ::rust::String loglevel;
  ::rust::String password;
  ::std::uint64_t port;

  bool operator==(Webservice const &) const noexcept;
  bool operator!=(Webservice const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Webservice

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Carbon
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Carbon
struct Carbon final {
  ::std::uint64_t interval;
  ::rust::String ns;
  ::rust::String ourname;
  ::rust::String instance;
  ::rust::Vec<::rust::String> server;

  bool operator==(Carbon const &) const noexcept;
  bool operator!=(Carbon const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Carbon

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Outgoing
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Outgoing
struct Outgoing final {
  ::rust::Vec<::rust::String> dont_throttle_names;
  ::rust::Vec<::rust::String> dont_throttle_netmasks;
  ::rust::Vec<::rust::String> dot_to_auth_names;
  bool dot_to_port_853;
  ::rust::Vec<::rust::String> dont_query;
  ::std::uint64_t edns_bufsize;
  bool edns_padding;
  ::rust::Vec<::rust::String> edns_subnet_allow_list;
  bool edns_subnet_harden;
  bool lowercase;
  ::std::uint64_t max_busy_dot_probes;
  ::std::uint64_t max_qperq;
  ::std::uint64_t max_ns_address_qperq;
  ::std::uint64_t max_ns_per_resolve;
  ::std::uint64_t network_timeout;
  ::std::uint64_t non_resolving_ns_max_fails;
  ::std::uint64_t non_resolving_ns_throttle_time;
  ::rust::Vec<::rust::String> source_address;
  ::std::uint64_t server_down_max_fails;
  ::std::uint64_t server_down_throttle_time;
  ::std::uint64_t bypass_server_throttling_probability;
  bool single_socket;
  bool tcp_fast_open_connect;
  ::std::uint64_t tcp_max_idle_ms;
  ::std::uint64_t tcp_max_idle_per_auth;
  ::std::uint64_t tcp_max_queries;
  ::std::uint64_t tcp_max_idle_per_thread;
  ::std::uint64_t udp_source_port_min;
  ::std::uint64_t udp_source_port_max;
  ::rust::Vec<::rust::String> udp_source_port_avoid;

  bool operator==(Outgoing const &) const noexcept;
  bool operator!=(Outgoing const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Outgoing

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Packetcache
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Packetcache
struct Packetcache final {
  bool disable;
  ::std::uint64_t max_entries;
  ::std::uint64_t ttl;
  ::std::uint64_t negative_ttl;
  ::std::uint64_t servfail_ttl;
  ::std::uint64_t shards;

  bool operator==(Packetcache const &) const noexcept;
  bool operator!=(Packetcache const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Packetcache

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Logging
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Logging
struct Logging final {
  bool disable_syslog;
  bool timestamp;
  ::std::uint64_t loglevel;
  bool common_errors;
  bool rpz_changes;
  ::rust::String facility;
  bool protobuf_use_kernel_timestamp;
  bool quiet;
  ::std::uint64_t statistics_interval;
  bool structured_logging;
  ::rust::String structured_logging_backend;
  ::rust::String trace;
  ::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> protobuf_servers;
  ::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> outgoing_protobuf_servers;
  ::std::uint64_t protobuf_mask_v4;
  ::std::uint64_t protobuf_mask_v6;
  ::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> dnstap_framestream_servers;
  ::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> dnstap_nod_framestream_servers;

  bool operator==(Logging const &) const noexcept;
  bool operator!=(Logging const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Logging

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Ecs
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Ecs
struct Ecs final {
  ::rust::Vec<::rust::String> add_for;
  ::std::uint64_t ipv4_bits;
  ::std::uint64_t ipv4_cache_bits;
  ::std::uint64_t ipv6_bits;
  ::std::uint64_t ipv6_cache_bits;
  bool ipv4_never_cache;
  bool ipv6_never_cache;
  ::std::uint64_t minimum_ttl_override;
  ::std::uint64_t cache_limit_ttl;
  ::rust::String scope_zero_address;

  bool operator==(Ecs const &) const noexcept;
  bool operator!=(Ecs const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Ecs

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recordcache
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recordcache
struct Recordcache final {
  ::std::uint64_t max_cache_bogus_ttl;
  ::std::uint64_t max_entries;
  ::std::uint64_t max_ttl;
  bool limit_qtype_any;
  ::std::uint64_t max_rrset_size;
  ::std::uint64_t max_negative_ttl;
  ::std::uint64_t locked_ttl_perc;
  ::std::uint64_t shards;
  ::std::uint64_t refresh_on_ttl_perc;
  ::std::uint64_t serve_stale_extensions;
  ::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> zonetocaches;

  bool operator==(Recordcache const &) const noexcept;
  bool operator!=(Recordcache const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recordcache

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Nod
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Nod
struct Nod final {
  bool tracking;
  bool log;
  ::rust::String lookup;
  ::std::uint64_t db_size;
  ::rust::String history_dir;
  ::std::uint64_t db_snapshot_interval;
  ::rust::Vec<::rust::String> ignore_list;
  ::rust::String ignore_list_file;
  ::rust::String pb_tag;
  bool unique_response_tracking;
  bool unique_response_log;
  ::std::uint64_t unique_response_db_size;
  ::rust::String unique_response_history_dir;
  ::rust::String unique_response_pb_tag;
  ::rust::Vec<::rust::String> unique_response_ignore_list;
  ::rust::String unique_response_ignore_list_file;

  bool operator==(Nod const &) const noexcept;
  bool operator!=(Nod const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Nod

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Snmp
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Snmp
struct Snmp final {
  bool agent;
  ::rust::String daemon_socket;

  bool operator==(Snmp const &) const noexcept;
  bool operator!=(Snmp const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Snmp

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursorsettings
#define CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursorsettings
struct Recursorsettings final {
  ::pdns::rust::settings::rec::Dnssec dnssec;
  ::pdns::rust::settings::rec::Incoming incoming;
  ::pdns::rust::settings::rec::Recursor recursor;
  ::pdns::rust::settings::rec::Webservice webservice;
  ::pdns::rust::settings::rec::Carbon carbon;
  ::pdns::rust::settings::rec::Outgoing outgoing;
  ::pdns::rust::settings::rec::Packetcache packetcache;
  ::pdns::rust::settings::rec::Logging logging;
  ::pdns::rust::settings::rec::Ecs ecs;
  ::pdns::rust::settings::rec::Recordcache recordcache;
  ::pdns::rust::settings::rec::Nod nod;
  ::pdns::rust::settings::rec::Snmp snmp;

  ::rust::String to_yaml_string() const;
  void validate() const;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$settings$rec$Recursorsettings

extern "C" {
bool pdns$rust$settings$rec$cxxbridge1$ForwardZone$operator$eq(ForwardZone const &, ForwardZone const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ForwardZone$operator$ne(ForwardZone const &, ForwardZone const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$AuthZone$operator$eq(AuthZone const &, AuthZone const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$AuthZone$operator$ne(AuthZone const &, AuthZone const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$TrustAnchor$operator$eq(TrustAnchor const &, TrustAnchor const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$TrustAnchor$operator$ne(TrustAnchor const &, TrustAnchor const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$NegativeTrustAnchor$operator$eq(NegativeTrustAnchor const &, NegativeTrustAnchor const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$NegativeTrustAnchor$operator$ne(NegativeTrustAnchor const &, NegativeTrustAnchor const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ProtobufServer$operator$eq(ProtobufServer const &, ProtobufServer const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ProtobufServer$operator$ne(ProtobufServer const &, ProtobufServer const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$DNSTapFrameStreamServer$operator$eq(DNSTapFrameStreamServer const &, DNSTapFrameStreamServer const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$DNSTapFrameStreamServer$operator$ne(DNSTapFrameStreamServer const &, DNSTapFrameStreamServer const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$DNSTapNODFrameStreamServer$operator$eq(DNSTapNODFrameStreamServer const &, DNSTapNODFrameStreamServer const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$DNSTapNODFrameStreamServer$operator$ne(DNSTapNODFrameStreamServer const &, DNSTapNODFrameStreamServer const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$TSIGTriplet$operator$eq(TSIGTriplet const &, TSIGTriplet const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$TSIGTriplet$operator$ne(TSIGTriplet const &, TSIGTriplet const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$RPZ$operator$eq(RPZ const &, RPZ const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$RPZ$operator$ne(RPZ const &, RPZ const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ZoneToCache$operator$eq(ZoneToCache const &, ZoneToCache const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ZoneToCache$operator$ne(ZoneToCache const &, ZoneToCache const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$SubnetOrder$operator$eq(SubnetOrder const &, SubnetOrder const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$SubnetOrder$operator$ne(SubnetOrder const &, SubnetOrder const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$SortList$operator$eq(SortList const &, SortList const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$SortList$operator$ne(SortList const &, SortList const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$AllowedAdditionalQType$operator$eq(AllowedAdditionalQType const &, AllowedAdditionalQType const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$AllowedAdditionalQType$operator$ne(AllowedAdditionalQType const &, AllowedAdditionalQType const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ProxyMapping$operator$eq(ProxyMapping const &, ProxyMapping const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ProxyMapping$operator$ne(ProxyMapping const &, ProxyMapping const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ApiZones$operator$eq(ApiZones const &, ApiZones const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ApiZones$operator$ne(ApiZones const &, ApiZones const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$XFR$operator$eq(XFR const &, XFR const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$XFR$operator$ne(XFR const &, XFR const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$FCZDefault$operator$eq(FCZDefault const &, FCZDefault const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$FCZDefault$operator$ne(FCZDefault const &, FCZDefault const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ForwardingCatalogZone$operator$eq(ForwardingCatalogZone const &, ForwardingCatalogZone const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$ForwardingCatalogZone$operator$ne(ForwardingCatalogZone const &, ForwardingCatalogZone const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$IncomingTLS$operator$eq(IncomingTLS const &, IncomingTLS const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$IncomingTLS$operator$ne(IncomingTLS const &, IncomingTLS const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$IncomingWSConfig$operator$eq(IncomingWSConfig const &, IncomingWSConfig const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$IncomingWSConfig$operator$ne(IncomingWSConfig const &, IncomingWSConfig const &) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$parse_yaml_string(::rust::Str str, ::pdns::rust::settings::rec::Recursorsettings *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$parse_yaml_string_to_allow_from(::rust::Str str, ::rust::Vec<::rust::String> *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$parse_yaml_string_to_forward_zones(::rust::Str str, ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$parse_yaml_string_to_allow_notify_for(::rust::Str str, ::rust::Vec<::rust::String> *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$parse_yaml_string_to_api_zones(::rust::Str str, ::pdns::rust::settings::rec::ApiZones *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$Recursorsettings$to_yaml_string(::pdns::rust::settings::rec::Recursorsettings const &self, ::rust::String *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$map_to_yaml_string(::rust::Vec<::pdns::rust::settings::rec::OldStyle> const &map, ::rust::String *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$forward_zones_to_yaml_string(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const &vec, ::rust::String *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$allow_from_to_yaml_string(::rust::Vec<::rust::String> const &vec, ::rust::String *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$allow_from_to_yaml_string_incoming(::rust::String const &key, ::rust::String const &filekey, ::rust::Vec<::rust::String> const &vec, ::rust::String *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$allow_for_to_yaml_string(::rust::Vec<::rust::String> const &vec, ::rust::String *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$merge(::pdns::rust::settings::rec::Recursorsettings &lhs, ::rust::Str rhs) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$Recursorsettings$validate(::pdns::rust::settings::rec::Recursorsettings const &self) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$AuthZone$validate(::pdns::rust::settings::rec::AuthZone const &self, ::rust::Str field) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$ForwardZone$validate(::pdns::rust::settings::rec::ForwardZone const &self, ::rust::Str field) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$TrustAnchor$validate(::pdns::rust::settings::rec::TrustAnchor const &self, ::rust::Str field) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$NegativeTrustAnchor$validate(::pdns::rust::settings::rec::NegativeTrustAnchor const &self, ::rust::Str field) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$ApiZones$validate(::pdns::rust::settings::rec::ApiZones const &self, ::rust::Str field) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$validate_auth_zones(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::AuthZone> const &vec) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$validate_forward_zones(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const &vec) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$validate_allow_for(::rust::Str field, ::rust::Vec<::rust::String> const &vec) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$validate_allow_notify_for(::rust::Str field, ::rust::Vec<::rust::String> const &vec) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$validate_allow_from(::rust::Str field, ::rust::Vec<::rust::String> const &vec) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$api_read_zones(::rust::Str path, ::std::unique_ptr<::pdns::rust::settings::rec::ApiZones> *return$) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$api_add_auth_zone(::rust::Str file, ::pdns::rust::settings::rec::AuthZone *authzone) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$api_add_forward_zone(::rust::Str file, ::pdns::rust::settings::rec::ForwardZone *forwardzone) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$api_add_forward_zones(::rust::Str file, ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> &forwardzones) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$validate_trustanchors(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> const &vec) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$validate_negativetrustanchors(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> const &vec) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$api_delete_zone(::rust::Str file, ::rust::Str zone) noexcept;

::rust::repr::PtrLen pdns$rust$settings$rec$cxxbridge1$api_delete_zones(::rust::Str file) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Dnssec$operator$eq(Dnssec const &, Dnssec const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Dnssec$operator$ne(Dnssec const &, Dnssec const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Incoming$operator$eq(Incoming const &, Incoming const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Incoming$operator$ne(Incoming const &, Incoming const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Recursor$operator$eq(Recursor const &, Recursor const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Recursor$operator$ne(Recursor const &, Recursor const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Webservice$operator$eq(Webservice const &, Webservice const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Webservice$operator$ne(Webservice const &, Webservice const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Carbon$operator$eq(Carbon const &, Carbon const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Carbon$operator$ne(Carbon const &, Carbon const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Outgoing$operator$eq(Outgoing const &, Outgoing const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Outgoing$operator$ne(Outgoing const &, Outgoing const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Packetcache$operator$eq(Packetcache const &, Packetcache const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Packetcache$operator$ne(Packetcache const &, Packetcache const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Logging$operator$eq(Logging const &, Logging const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Logging$operator$ne(Logging const &, Logging const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Ecs$operator$eq(Ecs const &, Ecs const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Ecs$operator$ne(Ecs const &, Ecs const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Recordcache$operator$eq(Recordcache const &, Recordcache const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Recordcache$operator$ne(Recordcache const &, Recordcache const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Nod$operator$eq(Nod const &, Nod const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Nod$operator$ne(Nod const &, Nod const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Snmp$operator$eq(Snmp const &, Snmp const &) noexcept;
bool pdns$rust$settings$rec$cxxbridge1$Snmp$operator$ne(Snmp const &, Snmp const &) noexcept;
} // extern "C"

bool ForwardZone::operator==(ForwardZone const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ForwardZone$operator$eq(*this, rhs);
}

bool ForwardZone::operator!=(ForwardZone const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ForwardZone$operator$ne(*this, rhs);
}

bool AuthZone::operator==(AuthZone const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$AuthZone$operator$eq(*this, rhs);
}

bool AuthZone::operator!=(AuthZone const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$AuthZone$operator$ne(*this, rhs);
}

bool TrustAnchor::operator==(TrustAnchor const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$TrustAnchor$operator$eq(*this, rhs);
}

bool TrustAnchor::operator!=(TrustAnchor const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$TrustAnchor$operator$ne(*this, rhs);
}

bool NegativeTrustAnchor::operator==(NegativeTrustAnchor const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$NegativeTrustAnchor$operator$eq(*this, rhs);
}

bool NegativeTrustAnchor::operator!=(NegativeTrustAnchor const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$NegativeTrustAnchor$operator$ne(*this, rhs);
}

bool ProtobufServer::operator==(ProtobufServer const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ProtobufServer$operator$eq(*this, rhs);
}

bool ProtobufServer::operator!=(ProtobufServer const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ProtobufServer$operator$ne(*this, rhs);
}

bool DNSTapFrameStreamServer::operator==(DNSTapFrameStreamServer const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$DNSTapFrameStreamServer$operator$eq(*this, rhs);
}

bool DNSTapFrameStreamServer::operator!=(DNSTapFrameStreamServer const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$DNSTapFrameStreamServer$operator$ne(*this, rhs);
}

bool DNSTapNODFrameStreamServer::operator==(DNSTapNODFrameStreamServer const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$DNSTapNODFrameStreamServer$operator$eq(*this, rhs);
}

bool DNSTapNODFrameStreamServer::operator!=(DNSTapNODFrameStreamServer const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$DNSTapNODFrameStreamServer$operator$ne(*this, rhs);
}

bool TSIGTriplet::operator==(TSIGTriplet const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$TSIGTriplet$operator$eq(*this, rhs);
}

bool TSIGTriplet::operator!=(TSIGTriplet const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$TSIGTriplet$operator$ne(*this, rhs);
}

bool RPZ::operator==(RPZ const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$RPZ$operator$eq(*this, rhs);
}

bool RPZ::operator!=(RPZ const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$RPZ$operator$ne(*this, rhs);
}

bool ZoneToCache::operator==(ZoneToCache const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ZoneToCache$operator$eq(*this, rhs);
}

bool ZoneToCache::operator!=(ZoneToCache const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ZoneToCache$operator$ne(*this, rhs);
}

bool SubnetOrder::operator==(SubnetOrder const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$SubnetOrder$operator$eq(*this, rhs);
}

bool SubnetOrder::operator!=(SubnetOrder const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$SubnetOrder$operator$ne(*this, rhs);
}

bool SortList::operator==(SortList const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$SortList$operator$eq(*this, rhs);
}

bool SortList::operator!=(SortList const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$SortList$operator$ne(*this, rhs);
}

bool AllowedAdditionalQType::operator==(AllowedAdditionalQType const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$AllowedAdditionalQType$operator$eq(*this, rhs);
}

bool AllowedAdditionalQType::operator!=(AllowedAdditionalQType const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$AllowedAdditionalQType$operator$ne(*this, rhs);
}

bool ProxyMapping::operator==(ProxyMapping const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ProxyMapping$operator$eq(*this, rhs);
}

bool ProxyMapping::operator!=(ProxyMapping const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ProxyMapping$operator$ne(*this, rhs);
}

bool ApiZones::operator==(ApiZones const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ApiZones$operator$eq(*this, rhs);
}

bool ApiZones::operator!=(ApiZones const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ApiZones$operator$ne(*this, rhs);
}

bool XFR::operator==(XFR const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$XFR$operator$eq(*this, rhs);
}

bool XFR::operator!=(XFR const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$XFR$operator$ne(*this, rhs);
}

bool FCZDefault::operator==(FCZDefault const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$FCZDefault$operator$eq(*this, rhs);
}

bool FCZDefault::operator!=(FCZDefault const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$FCZDefault$operator$ne(*this, rhs);
}

bool ForwardingCatalogZone::operator==(ForwardingCatalogZone const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ForwardingCatalogZone$operator$eq(*this, rhs);
}

bool ForwardingCatalogZone::operator!=(ForwardingCatalogZone const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$ForwardingCatalogZone$operator$ne(*this, rhs);
}

bool IncomingTLS::operator==(IncomingTLS const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$IncomingTLS$operator$eq(*this, rhs);
}

bool IncomingTLS::operator!=(IncomingTLS const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$IncomingTLS$operator$ne(*this, rhs);
}

bool IncomingWSConfig::operator==(IncomingWSConfig const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$IncomingWSConfig$operator$eq(*this, rhs);
}

bool IncomingWSConfig::operator!=(IncomingWSConfig const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$IncomingWSConfig$operator$ne(*this, rhs);
}

::pdns::rust::settings::rec::Recursorsettings parse_yaml_string(::rust::Str str) {
  ::rust::MaybeUninit<::pdns::rust::settings::rec::Recursorsettings> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$parse_yaml_string(str, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::rust::String> parse_yaml_string_to_allow_from(::rust::Str str) {
  ::rust::MaybeUninit<::rust::Vec<::rust::String>> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$parse_yaml_string_to_allow_from(str, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::pdns::rust::settings::rec::ForwardZone> parse_yaml_string_to_forward_zones(::rust::Str str) {
  ::rust::MaybeUninit<::rust::Vec<::pdns::rust::settings::rec::ForwardZone>> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$parse_yaml_string_to_forward_zones(str, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::Vec<::rust::String> parse_yaml_string_to_allow_notify_for(::rust::Str str) {
  ::rust::MaybeUninit<::rust::Vec<::rust::String>> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$parse_yaml_string_to_allow_notify_for(str, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::pdns::rust::settings::rec::ApiZones parse_yaml_string_to_api_zones(::rust::Str str) {
  ::rust::MaybeUninit<::pdns::rust::settings::rec::ApiZones> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$parse_yaml_string_to_api_zones(str, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::String Recursorsettings::to_yaml_string() const {
  ::rust::MaybeUninit<::rust::String> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$Recursorsettings$to_yaml_string(*this, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::String map_to_yaml_string(::rust::Vec<::pdns::rust::settings::rec::OldStyle> const &map) {
  ::rust::MaybeUninit<::rust::String> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$map_to_yaml_string(map, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::String forward_zones_to_yaml_string(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const &vec) {
  ::rust::MaybeUninit<::rust::String> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$forward_zones_to_yaml_string(vec, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::String allow_from_to_yaml_string(::rust::Vec<::rust::String> const &vec) {
  ::rust::MaybeUninit<::rust::String> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$allow_from_to_yaml_string(vec, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::String allow_from_to_yaml_string_incoming(::rust::String const &key, ::rust::String const &filekey, ::rust::Vec<::rust::String> const &vec) {
  ::rust::MaybeUninit<::rust::String> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$allow_from_to_yaml_string_incoming(key, filekey, vec, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

::rust::String allow_for_to_yaml_string(::rust::Vec<::rust::String> const &vec) {
  ::rust::MaybeUninit<::rust::String> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$allow_for_to_yaml_string(vec, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void merge(::pdns::rust::settings::rec::Recursorsettings &lhs, ::rust::Str rhs) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$merge(lhs, rhs);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void Recursorsettings::validate() const {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$Recursorsettings$validate(*this);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void AuthZone::validate(::rust::Str field) const {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$AuthZone$validate(*this, field);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void ForwardZone::validate(::rust::Str field) const {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$ForwardZone$validate(*this, field);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void TrustAnchor::validate(::rust::Str field) const {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$TrustAnchor$validate(*this, field);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void NegativeTrustAnchor::validate(::rust::Str field) const {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$NegativeTrustAnchor$validate(*this, field);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void ApiZones::validate(::rust::Str field) const {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$ApiZones$validate(*this, field);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void validate_auth_zones(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::AuthZone> const &vec) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$validate_auth_zones(field, vec);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void validate_forward_zones(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const &vec) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$validate_forward_zones(field, vec);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void validate_allow_for(::rust::Str field, ::rust::Vec<::rust::String> const &vec) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$validate_allow_for(field, vec);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void validate_allow_notify_for(::rust::Str field, ::rust::Vec<::rust::String> const &vec) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$validate_allow_notify_for(field, vec);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void validate_allow_from(::rust::Str field, ::rust::Vec<::rust::String> const &vec) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$validate_allow_from(field, vec);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

::std::unique_ptr<::pdns::rust::settings::rec::ApiZones> api_read_zones(::rust::Str path) {
  ::rust::MaybeUninit<::std::unique_ptr<::pdns::rust::settings::rec::ApiZones>> return$;
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$api_read_zones(path, &return$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
  return ::std::move(return$.value);
}

void api_add_auth_zone(::rust::Str file, ::pdns::rust::settings::rec::AuthZone authzone) {
  ::rust::ManuallyDrop<::pdns::rust::settings::rec::AuthZone> authzone$(::std::move(authzone));
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$api_add_auth_zone(file, &authzone$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void api_add_forward_zone(::rust::Str file, ::pdns::rust::settings::rec::ForwardZone forwardzone) {
  ::rust::ManuallyDrop<::pdns::rust::settings::rec::ForwardZone> forwardzone$(::std::move(forwardzone));
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$api_add_forward_zone(file, &forwardzone$.value);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void api_add_forward_zones(::rust::Str file, ::rust::Vec<::pdns::rust::settings::rec::ForwardZone> &forwardzones) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$api_add_forward_zones(file, forwardzones);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void validate_trustanchors(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> const &vec) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$validate_trustanchors(field, vec);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void validate_negativetrustanchors(::rust::Str field, ::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> const &vec) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$validate_negativetrustanchors(field, vec);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void api_delete_zone(::rust::Str file, ::rust::Str zone) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$api_delete_zone(file, zone);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

void api_delete_zones(::rust::Str file) {
  ::rust::repr::PtrLen error$ = pdns$rust$settings$rec$cxxbridge1$api_delete_zones(file);
  if (error$.ptr) {
    throw ::rust::impl<::rust::Error>::error(error$);
  }
}

bool Dnssec::operator==(Dnssec const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Dnssec$operator$eq(*this, rhs);
}

bool Dnssec::operator!=(Dnssec const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Dnssec$operator$ne(*this, rhs);
}

bool Incoming::operator==(Incoming const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Incoming$operator$eq(*this, rhs);
}

bool Incoming::operator!=(Incoming const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Incoming$operator$ne(*this, rhs);
}

bool Recursor::operator==(Recursor const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Recursor$operator$eq(*this, rhs);
}

bool Recursor::operator!=(Recursor const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Recursor$operator$ne(*this, rhs);
}

bool Webservice::operator==(Webservice const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Webservice$operator$eq(*this, rhs);
}

bool Webservice::operator!=(Webservice const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Webservice$operator$ne(*this, rhs);
}

bool Carbon::operator==(Carbon const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Carbon$operator$eq(*this, rhs);
}

bool Carbon::operator!=(Carbon const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Carbon$operator$ne(*this, rhs);
}

bool Outgoing::operator==(Outgoing const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Outgoing$operator$eq(*this, rhs);
}

bool Outgoing::operator!=(Outgoing const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Outgoing$operator$ne(*this, rhs);
}

bool Packetcache::operator==(Packetcache const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Packetcache$operator$eq(*this, rhs);
}

bool Packetcache::operator!=(Packetcache const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Packetcache$operator$ne(*this, rhs);
}

bool Logging::operator==(Logging const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Logging$operator$eq(*this, rhs);
}

bool Logging::operator!=(Logging const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Logging$operator$ne(*this, rhs);
}

bool Ecs::operator==(Ecs const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Ecs$operator$eq(*this, rhs);
}

bool Ecs::operator!=(Ecs const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Ecs$operator$ne(*this, rhs);
}

bool Recordcache::operator==(Recordcache const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Recordcache$operator$eq(*this, rhs);
}

bool Recordcache::operator!=(Recordcache const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Recordcache$operator$ne(*this, rhs);
}

bool Nod::operator==(Nod const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Nod$operator$eq(*this, rhs);
}

bool Nod::operator!=(Nod const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Nod$operator$ne(*this, rhs);
}

bool Snmp::operator==(Snmp const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Snmp$operator$eq(*this, rhs);
}

bool Snmp::operator!=(Snmp const &rhs) const noexcept {
  return pdns$rust$settings$rec$cxxbridge1$Snmp$operator$ne(*this, rhs);
}
} // namespace rec
} // namespace settings
} // namespace rust
} // namespace pdns

extern "C" {
void cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$new(::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$drop(::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$len(::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$capacity(::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> const *ptr) noexcept;
::pdns::rust::settings::rec::SubnetOrder const *cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$data(::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$reserve_total(::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$set_len(::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$truncate(::rust::Vec<::pdns::rust::settings::rec::SubnetOrder> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$new(::rust::Vec<::pdns::rust::settings::rec::AuthZone> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$drop(::rust::Vec<::pdns::rust::settings::rec::AuthZone> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$len(::rust::Vec<::pdns::rust::settings::rec::AuthZone> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$capacity(::rust::Vec<::pdns::rust::settings::rec::AuthZone> const *ptr) noexcept;
::pdns::rust::settings::rec::AuthZone const *cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$data(::rust::Vec<::pdns::rust::settings::rec::AuthZone> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$reserve_total(::rust::Vec<::pdns::rust::settings::rec::AuthZone> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$set_len(::rust::Vec<::pdns::rust::settings::rec::AuthZone> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$truncate(::rust::Vec<::pdns::rust::settings::rec::AuthZone> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$new(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$drop(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$len(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$capacity(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const *ptr) noexcept;
::pdns::rust::settings::rec::ForwardZone const *cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$data(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$reserve_total(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$set_len(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$truncate(::rust::Vec<::pdns::rust::settings::rec::ForwardZone> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$new(::rust::Vec<::pdns::rust::settings::rec::FCZDefault> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$drop(::rust::Vec<::pdns::rust::settings::rec::FCZDefault> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$len(::rust::Vec<::pdns::rust::settings::rec::FCZDefault> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$capacity(::rust::Vec<::pdns::rust::settings::rec::FCZDefault> const *ptr) noexcept;
::pdns::rust::settings::rec::FCZDefault const *cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$data(::rust::Vec<::pdns::rust::settings::rec::FCZDefault> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$reserve_total(::rust::Vec<::pdns::rust::settings::rec::FCZDefault> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$set_len(::rust::Vec<::pdns::rust::settings::rec::FCZDefault> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$truncate(::rust::Vec<::pdns::rust::settings::rec::FCZDefault> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$new(::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$drop(::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$len(::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$capacity(::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> const *ptr) noexcept;
::pdns::rust::settings::rec::TrustAnchor const *cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$data(::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$reserve_total(::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$set_len(::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$truncate(::rust::Vec<::pdns::rust::settings::rec::TrustAnchor> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$new(::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$drop(::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$len(::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$capacity(::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> const *ptr) noexcept;
::pdns::rust::settings::rec::NegativeTrustAnchor const *cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$data(::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$reserve_total(::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$set_len(::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$truncate(::rust::Vec<::pdns::rust::settings::rec::NegativeTrustAnchor> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$new(::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$drop(::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$len(::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$capacity(::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> const *ptr) noexcept;
::pdns::rust::settings::rec::ProtobufServer const *cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$data(::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$reserve_total(::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$set_len(::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$truncate(::rust::Vec<::pdns::rust::settings::rec::ProtobufServer> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$new(::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$drop(::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$len(::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$capacity(::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> const *ptr) noexcept;
::pdns::rust::settings::rec::DNSTapFrameStreamServer const *cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$data(::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$reserve_total(::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$set_len(::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$truncate(::rust::Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$new(::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$drop(::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$len(::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$capacity(::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> const *ptr) noexcept;
::pdns::rust::settings::rec::DNSTapNODFrameStreamServer const *cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$data(::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$reserve_total(::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$set_len(::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$truncate(::rust::Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$new(::rust::Vec<::pdns::rust::settings::rec::RPZ> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$drop(::rust::Vec<::pdns::rust::settings::rec::RPZ> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$len(::rust::Vec<::pdns::rust::settings::rec::RPZ> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$capacity(::rust::Vec<::pdns::rust::settings::rec::RPZ> const *ptr) noexcept;
::pdns::rust::settings::rec::RPZ const *cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$data(::rust::Vec<::pdns::rust::settings::rec::RPZ> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$reserve_total(::rust::Vec<::pdns::rust::settings::rec::RPZ> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$set_len(::rust::Vec<::pdns::rust::settings::rec::RPZ> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$truncate(::rust::Vec<::pdns::rust::settings::rec::RPZ> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$new(::rust::Vec<::pdns::rust::settings::rec::SortList> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$drop(::rust::Vec<::pdns::rust::settings::rec::SortList> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$len(::rust::Vec<::pdns::rust::settings::rec::SortList> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$capacity(::rust::Vec<::pdns::rust::settings::rec::SortList> const *ptr) noexcept;
::pdns::rust::settings::rec::SortList const *cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$data(::rust::Vec<::pdns::rust::settings::rec::SortList> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$reserve_total(::rust::Vec<::pdns::rust::settings::rec::SortList> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$set_len(::rust::Vec<::pdns::rust::settings::rec::SortList> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$truncate(::rust::Vec<::pdns::rust::settings::rec::SortList> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$new(::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$drop(::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$len(::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$capacity(::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> const *ptr) noexcept;
::pdns::rust::settings::rec::ZoneToCache const *cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$data(::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$reserve_total(::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$set_len(::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$truncate(::rust::Vec<::pdns::rust::settings::rec::ZoneToCache> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$new(::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$drop(::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$len(::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$capacity(::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> const *ptr) noexcept;
::pdns::rust::settings::rec::AllowedAdditionalQType const *cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$data(::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$reserve_total(::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$set_len(::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$truncate(::rust::Vec<::pdns::rust::settings::rec::AllowedAdditionalQType> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$new(::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$drop(::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$len(::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$capacity(::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> const *ptr) noexcept;
::pdns::rust::settings::rec::ProxyMapping const *cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$data(::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$reserve_total(::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$set_len(::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$truncate(::rust::Vec<::pdns::rust::settings::rec::ProxyMapping> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$new(::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$drop(::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$len(::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$capacity(::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> const *ptr) noexcept;
::pdns::rust::settings::rec::ForwardingCatalogZone const *cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$data(::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$reserve_total(::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$set_len(::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$truncate(::rust::Vec<::pdns::rust::settings::rec::ForwardingCatalogZone> *ptr, ::std::size_t len) noexcept;

void cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$new(::rust::Vec<::pdns::rust::settings::rec::OldStyle> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$drop(::rust::Vec<::pdns::rust::settings::rec::OldStyle> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$len(::rust::Vec<::pdns::rust::settings::rec::OldStyle> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$capacity(::rust::Vec<::pdns::rust::settings::rec::OldStyle> const *ptr) noexcept;
::pdns::rust::settings::rec::OldStyle const *cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$data(::rust::Vec<::pdns::rust::settings::rec::OldStyle> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$reserve_total(::rust::Vec<::pdns::rust::settings::rec::OldStyle> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$set_len(::rust::Vec<::pdns::rust::settings::rec::OldStyle> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$truncate(::rust::Vec<::pdns::rust::settings::rec::OldStyle> *ptr, ::std::size_t len) noexcept;

static_assert(sizeof(::std::unique_ptr<::pdns::rust::settings::rec::ApiZones>) == sizeof(void *), "");
static_assert(alignof(::std::unique_ptr<::pdns::rust::settings::rec::ApiZones>) == alignof(void *), "");
void cxxbridge1$unique_ptr$pdns$rust$settings$rec$ApiZones$null(::std::unique_ptr<::pdns::rust::settings::rec::ApiZones> *ptr) noexcept {
  ::new (ptr) ::std::unique_ptr<::pdns::rust::settings::rec::ApiZones>();
}
::pdns::rust::settings::rec::ApiZones *cxxbridge1$unique_ptr$pdns$rust$settings$rec$ApiZones$uninit(::std::unique_ptr<::pdns::rust::settings::rec::ApiZones> *ptr) noexcept {
  ::pdns::rust::settings::rec::ApiZones *uninit = reinterpret_cast<::pdns::rust::settings::rec::ApiZones *>(new ::rust::MaybeUninit<::pdns::rust::settings::rec::ApiZones>);
  ::new (ptr) ::std::unique_ptr<::pdns::rust::settings::rec::ApiZones>(uninit);
  return uninit;
}
void cxxbridge1$unique_ptr$pdns$rust$settings$rec$ApiZones$raw(::std::unique_ptr<::pdns::rust::settings::rec::ApiZones> *ptr, ::pdns::rust::settings::rec::ApiZones *raw) noexcept {
  ::new (ptr) ::std::unique_ptr<::pdns::rust::settings::rec::ApiZones>(raw);
}
::pdns::rust::settings::rec::ApiZones const *cxxbridge1$unique_ptr$pdns$rust$settings$rec$ApiZones$get(::std::unique_ptr<::pdns::rust::settings::rec::ApiZones> const &ptr) noexcept {
  return ptr.get();
}
::pdns::rust::settings::rec::ApiZones *cxxbridge1$unique_ptr$pdns$rust$settings$rec$ApiZones$release(::std::unique_ptr<::pdns::rust::settings::rec::ApiZones> &ptr) noexcept {
  return ptr.release();
}
void cxxbridge1$unique_ptr$pdns$rust$settings$rec$ApiZones$drop(::std::unique_ptr<::pdns::rust::settings::rec::ApiZones> *ptr) noexcept {
  ptr->~unique_ptr();
}

void cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$new(::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$drop(::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$len(::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$capacity(::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> const *ptr) noexcept;
::pdns::rust::settings::rec::IncomingWSConfig const *cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$data(::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$reserve_total(::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$set_len(::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$truncate(::rust::Vec<::pdns::rust::settings::rec::IncomingWSConfig> *ptr, ::std::size_t len) noexcept;
} // extern "C"

namespace rust {
inline namespace cxxbridge1 {
template <>
Vec<::pdns::rust::settings::rec::SubnetOrder>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::SubnetOrder>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::SubnetOrder>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::SubnetOrder>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$capacity(this);
}
template <>
::pdns::rust::settings::rec::SubnetOrder const *Vec<::pdns::rust::settings::rec::SubnetOrder>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::SubnetOrder>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::SubnetOrder>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::SubnetOrder>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SubnetOrder$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::AuthZone>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::AuthZone>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::AuthZone>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::AuthZone>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$capacity(this);
}
template <>
::pdns::rust::settings::rec::AuthZone const *Vec<::pdns::rust::settings::rec::AuthZone>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::AuthZone>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::AuthZone>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::AuthZone>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AuthZone$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::ForwardZone>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ForwardZone>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ForwardZone>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ForwardZone>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$capacity(this);
}
template <>
::pdns::rust::settings::rec::ForwardZone const *Vec<::pdns::rust::settings::rec::ForwardZone>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ForwardZone>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::ForwardZone>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::ForwardZone>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardZone$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::FCZDefault>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::FCZDefault>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::FCZDefault>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::FCZDefault>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$capacity(this);
}
template <>
::pdns::rust::settings::rec::FCZDefault const *Vec<::pdns::rust::settings::rec::FCZDefault>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::FCZDefault>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::FCZDefault>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::FCZDefault>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$FCZDefault$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::TrustAnchor>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::TrustAnchor>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::TrustAnchor>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::TrustAnchor>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$capacity(this);
}
template <>
::pdns::rust::settings::rec::TrustAnchor const *Vec<::pdns::rust::settings::rec::TrustAnchor>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::TrustAnchor>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::TrustAnchor>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::TrustAnchor>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$TrustAnchor$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::NegativeTrustAnchor>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::NegativeTrustAnchor>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::NegativeTrustAnchor>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::NegativeTrustAnchor>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$capacity(this);
}
template <>
::pdns::rust::settings::rec::NegativeTrustAnchor const *Vec<::pdns::rust::settings::rec::NegativeTrustAnchor>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::NegativeTrustAnchor>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::NegativeTrustAnchor>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::NegativeTrustAnchor>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$NegativeTrustAnchor$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::ProtobufServer>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ProtobufServer>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ProtobufServer>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ProtobufServer>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$capacity(this);
}
template <>
::pdns::rust::settings::rec::ProtobufServer const *Vec<::pdns::rust::settings::rec::ProtobufServer>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ProtobufServer>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::ProtobufServer>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::ProtobufServer>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProtobufServer$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$capacity(this);
}
template <>
::pdns::rust::settings::rec::DNSTapFrameStreamServer const *Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::DNSTapFrameStreamServer>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapFrameStreamServer$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$capacity(this);
}
template <>
::pdns::rust::settings::rec::DNSTapNODFrameStreamServer const *Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::DNSTapNODFrameStreamServer>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$DNSTapNODFrameStreamServer$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::RPZ>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::RPZ>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::RPZ>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::RPZ>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$capacity(this);
}
template <>
::pdns::rust::settings::rec::RPZ const *Vec<::pdns::rust::settings::rec::RPZ>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::RPZ>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::RPZ>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::RPZ>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$RPZ$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::SortList>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::SortList>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::SortList>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::SortList>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$capacity(this);
}
template <>
::pdns::rust::settings::rec::SortList const *Vec<::pdns::rust::settings::rec::SortList>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::SortList>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::SortList>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::SortList>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$SortList$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::ZoneToCache>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ZoneToCache>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ZoneToCache>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ZoneToCache>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$capacity(this);
}
template <>
::pdns::rust::settings::rec::ZoneToCache const *Vec<::pdns::rust::settings::rec::ZoneToCache>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ZoneToCache>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::ZoneToCache>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::ZoneToCache>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ZoneToCache$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::AllowedAdditionalQType>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::AllowedAdditionalQType>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::AllowedAdditionalQType>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::AllowedAdditionalQType>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$capacity(this);
}
template <>
::pdns::rust::settings::rec::AllowedAdditionalQType const *Vec<::pdns::rust::settings::rec::AllowedAdditionalQType>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::AllowedAdditionalQType>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::AllowedAdditionalQType>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::AllowedAdditionalQType>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$AllowedAdditionalQType$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::ProxyMapping>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ProxyMapping>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ProxyMapping>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ProxyMapping>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$capacity(this);
}
template <>
::pdns::rust::settings::rec::ProxyMapping const *Vec<::pdns::rust::settings::rec::ProxyMapping>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ProxyMapping>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::ProxyMapping>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::ProxyMapping>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ProxyMapping$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::ForwardingCatalogZone>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ForwardingCatalogZone>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ForwardingCatalogZone>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::ForwardingCatalogZone>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$capacity(this);
}
template <>
::pdns::rust::settings::rec::ForwardingCatalogZone const *Vec<::pdns::rust::settings::rec::ForwardingCatalogZone>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::ForwardingCatalogZone>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::ForwardingCatalogZone>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::ForwardingCatalogZone>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$ForwardingCatalogZone$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::OldStyle>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::OldStyle>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::OldStyle>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::OldStyle>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$capacity(this);
}
template <>
::pdns::rust::settings::rec::OldStyle const *Vec<::pdns::rust::settings::rec::OldStyle>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::OldStyle>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::OldStyle>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::OldStyle>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$OldStyle$truncate(this, len);
}
template <>
Vec<::pdns::rust::settings::rec::IncomingWSConfig>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$new(this);
}
template <>
void Vec<::pdns::rust::settings::rec::IncomingWSConfig>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::IncomingWSConfig>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::settings::rec::IncomingWSConfig>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$capacity(this);
}
template <>
::pdns::rust::settings::rec::IncomingWSConfig const *Vec<::pdns::rust::settings::rec::IncomingWSConfig>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$data(this);
}
template <>
void Vec<::pdns::rust::settings::rec::IncomingWSConfig>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::settings::rec::IncomingWSConfig>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$set_len(this, len);
}
template <>
void Vec<::pdns::rust::settings::rec::IncomingWSConfig>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$settings$rec$IncomingWSConfig$truncate(this, len);
}
} // namespace cxxbridge1
} // namespace rust
