// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A1__F163

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_256_0_F_T_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1151)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x6ab971d7u, 0xa2d89f86u, 0, 155 }, // 6ab971d7a2d89f86 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1151
    { 0xb4792a09u, 0xf1a84ccdu, 0, 6 }, // b4792a09f1a84ccd = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1151
    { 0xdb16998bu, 0x4e8e4ddau, 0, 109 }, // db16998b4e8e4dda = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1151
    { 0x143aa19eu, 0x5e662b60u, 23, 155 }, // 143aa19e5e662b60 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1151
    { 0x37bc0f66u, 0xe4aa13e9u, 23, 6 }, // 37bc0f66e4aa13e9 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1151
    { 0xa203b2a9u, 0x2c05e3bcu, 23, 86 }, // a203b2a92c05e3bc = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1151
    { 0xd1d80da7u, 0xe2785e52u, 63, 138 }, // d1d80da7e2785e52 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1151
    { 0x7f7e7d0fu, 0xa4df90f2u, 63, 69 }, // 7f7e7d0fa4df90f2 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1151
    { 0xb2f60ddau, 0x95e158f4u, 126, 46 }, // b2f60dda95e158f4 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_F_T_1___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1151
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx1151_mod0
{{3,0,3,0,0,5,4},
 {3,1,0,0,0,0,0},
 {3,0,0,0,0,0,0},
 {6,0,0,0,0,0,0},
 {6,6,0,0,0,0,0},
 {7,8,0,0,0,0,2},
 {6,8,6,0,0,6,0}}
// End of GPU gfx1151_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A1__F163 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 256
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 1

// vim: set fileencoding=utf-8

