// SPDX-FileCopyrightText: Copyright (c) <2025> NVIDIA CORPORATION & AFFILIATES. All rights reserved.
//
// SPDX-License-Identifier: Apache-2.0

#include "cuda_helper.h"
#include "cuda_loader.h"


const char* get_cuda_error(CUresult res) {
    const char* str = nullptr;
    g_cuGetErrorString(res, &str);
    return str ? str : "Unknown error";
}

void try_init_cuda() {
    ErrorGuard guard;
    CUresult res = g_cuInit(0);
    if (res != CUDA_SUCCESS) {
        raise(PyExc_RuntimeError, "cuInit: %s", get_cuda_error(res));
        SavedException exc = save_raised_exception();
        LOG_PYTHON_ERROR("warning", exc, "Failed to initialized CUDA");
    }
}

PyObject* get_max_grid_size(PyObject *self, PyObject *args) {
    int device_id;
    if (!PyArg_ParseTuple(args, "i", &device_id))
        return NULL;

    CUdevice dev;
    CUresult res = g_cuDeviceGet(&dev, device_id);
    if (res != CUDA_SUCCESS)
        return PyErr_Format(PyExc_RuntimeError, "cuDeviceGet: %s", get_cuda_error(res));

    int max_grid_size[3];
    for (int i = 0; i < 3; ++i) {
        res = g_cuDeviceGetAttribute(&max_grid_size[i],
            static_cast<CUdevice_attribute>(CU_DEVICE_ATTRIBUTE_MAX_GRID_DIM_X + i),
            dev);
        if (res != CUDA_SUCCESS) {
            return PyErr_Format(PyExc_RuntimeError,
                                "cuDeviceGetAttribute: %s", get_cuda_error(res));
        }
    }
    return Py_BuildValue("(iii)", max_grid_size[0], max_grid_size[1], max_grid_size[2]);
}

PyObject* get_compute_capability(PyObject *self, PyObject *Py_UNUSED(ignored)) {
    int major, minor;
    CUdevice dev;
    CUresult res = g_cuDeviceGet(&dev, 0);
    if (res != CUDA_SUCCESS) {
        return PyErr_Format(PyExc_RuntimeError, "cuDeviceGet: %s", get_cuda_error(res));
    }
    res = g_cuDeviceGetAttribute(&major, CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MAJOR, dev);
    if (res != CUDA_SUCCESS) {
        return PyErr_Format(PyExc_RuntimeError, "cuDeviceGetAttribute: %s", get_cuda_error(res));
    }
    res = g_cuDeviceGetAttribute(&minor, CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MINOR, dev);
    if (res != CUDA_SUCCESS) {
        return PyErr_Format(PyExc_RuntimeError, "cuDeviceGetAttribute: %s", get_cuda_error(res));
    }
    return Py_BuildValue("(ii)", major, minor);
}

PyObject* get_driver_version(PyObject *self, PyObject *Py_UNUSED(ignored)) {
    int major, minor;
    CUresult res = g_cuDriverGetVersion(&major);
    if (res != CUDA_SUCCESS) {
        return PyErr_Format(PyExc_RuntimeError, "cuDriverGetVersion: %s", get_cuda_error(res));
    }
    minor = (major % 1000) / 10;
    major = major / 1000;
    return Py_BuildValue("(ii)", major, minor);
}

static PyMethodDef functions[] = {
    {"get_compute_capability", get_compute_capability, METH_NOARGS,
        "Get compute capability of the default CUDA device"},
    {"get_driver_version", get_driver_version, METH_NOARGS,
        "Get the cuda driver version"},
    {"_get_max_grid_size", get_max_grid_size, METH_VARARGS,
        "Get max grid size of a CUDA device, given device id"},
    NULL
};

Status cuda_helper_init(PyObject* m) {
    if (PyModule_AddFunctions(m, functions) < 0)
        return ErrorRaised;

    return OK;
}
