(in-package "DM")

(include-book "sym")
(local (include-book "maxine/dihedral"))

;; For n >= 3, consider a regular n-gon with vertices labeled 0, 1, ..., n-1 in clockwise order.  The dihedral group,
;; (dihedral n), is the subgroup of (sym n) of order 2n consisting of the permutations of these vertices that may be
;; realized by rigid motion of the n-gon in 3-dimensional space.  Each of these is either a rotation or a reflection
;; through some axis of symmetry.  The group is generated by 2 elements, the first of which is the basic clockwise
;; rotation by 2*pi/n radians, which has order n:

(defund rot (n) (append (cdr (ninit n)) (list 0)))

(defthm permp-rot
  (implies (posp n)
	   (member-equal (rot n) (slist n))))

(defthm nth-rot
  (implies (and (posp n) (natp i) (< i n))
	   (equal (nth i (rot n))
		  (mod (1+ i) n))))

(defthm index-rot
  (implies (and (posp n) (natp i) (< i n))
	   (equal (index i (rot n))
		  (mod (1- i) n))))

(defthm ord-rot
  (implies (posp n)
	   (equal (ord (rot n) (sym n))
		  n)))

;; The second element is the reflection about the axis of symmetry that passes through the center of the n-gon and 
;; the midpoint of the side connecting vertices 0 and n - 1, which has order 2:

(defund ref (n) (reverse (ninit n)))

(defthm permp-ref
  (implies (posp n)
	   (member-equal (ref n) (slist n))))

(defthm nth-ref
  (implies (and (posp n) (natp i) (< i n))
	   (equal (nth i (ref n))
		  (- (1- n) i))))

(defthm ord-ref
  (implies (and (natp n) (> n 1))
	   (equal (ord (ref n) (sym n))
		  2)))

;; The element list of (dihedral n) consists of the n powers of (rot n) together with the composition of
;; (ref n) with each of these powers:

(defun map-comp-perm (p l n)
  ;; Compose a permutation p with each member of a list l of permutations:
  (if (consp l)
      (cons (comp-perm p (car l) n)
            (map-comp-perm p (cdr l) n))
    ()))

(defun delts (n)
  (append (powers (rot n) (sym n))
          (map-comp-perm (ref n) (powers (rot n) (sym n)) n)))

;; Computing the group operation on (dihedral n) is easy once we have the following:

(defthm comp-perm-ref-rot
  (implies (and (natp n) (>= n 3))
           (equal (comp-perm (ref n) (rot n) n)
	          (comp-perm (inv-perm (rot n) n) (ref n) n))))

;; The rewrite rules required by defsubgroup:

(defthm dlistp-delts
  (implies (and (natp n) (>= n 3))
	   (dlistp (delts n))))

(defthm sublistp-delts-sym
    (implies (posp n)
             (sublistp (delts n) (slist n))))

(defthm consp-delts
    (implies (and (natp n) (>= n 3))
             (consp (delts n))))

(defthm delts-elts-identity
    (implies (and (natp n) (>= n 3))
             (equal (car (delts n)) (e (sym n)))))

(defthm delts-elts-closed
    (implies (and (natp n) (>= n 3)
		  (member-equal x (delts n))
		  (member-equal y (delts n)))
	     (member-equal (comp-perm x y n) (delts n))))

(defthm delts-elts-inverse
      (implies (and (natp n) (>= n 3)
                    (member-equal x (delts n)))
               (member-equal (inv-perm x n) (delts n))))

(defsubgroup dihedral (n)
  (sym n)
  (and (natp n) (>= n 3))
  (delts n))
