#include "util.h"
#include "common.h"
#include <dirent.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <openssl/sha.h>

typedef struct {
    char filename[PATH_MAX];
    unsigned char sha256[SHA256_DIGEST_LENGTH];
} ConfigFile;

int main(int argc, char **argv)
{
    DIR *pdir = NULL;
    struct dirent *pdirent;
    char dir_path[PATH_MAX] = {0};

    snprintf(dir_path, PATH_MAX, "%s%s", argc > 1 ? argv[1] : "", CONFIG_SHELL_IN_CODE_PATH);

    pdir = opendir(dir_path);
    if (pdir == NULL) {
        fprintf(stderr, N_("Error: Failed to open dir %s: %s\n"), dir_path, strerror(errno));
        return ERROR;
    }

    ConfigFile *files = NULL;
    int count = 0;
    int capacity = 0;
    char buff[PATH_MAX] = {0};

    while ((pdirent = readdir(pdir)) != NULL) {
        // Skip . and ..
        if (strcmp(pdirent->d_name, ".") == 0 || strcmp(pdirent->d_name, "..") == 0)
            continue;

        snprintf(buff, PATH_MAX, "%s/%s", dir_path, pdirent->d_name);

        struct stat sbuf;
        if (lstat(buff, &sbuf) == -1) {
            fprintf(stderr, N_("Error: lstat error %s: %s\n"), buff, strerror(errno));
            continue;
        }

        if (S_ISREG(sbuf.st_mode) && str_endsWith(buff, ".sh")) {
            // Resize array if needed
            if (count >= capacity) {
                capacity = capacity == 0 ? 8 : capacity * 2;
                ConfigFile *new_files = realloc(files, sizeof(ConfigFile) * capacity);
                if (!new_files) {
                    fprintf(stderr, N_("Error: Failed to allocate memory for ConfigFile\n"));
                    closedir(pdir);
                    free(files);
                    return ERROR;
                }
                files = new_files;
            }

            // Calculate SHA256
            unsigned char sha256_hash[SHA256_DIGEST_LENGTH];
            if (calculate_sha256(buff, sha256_hash) < 0) {
                fprintf(stderr, N_("Error: Failed to calculate SHA256 for %s\n"), buff);
                closedir(pdir);
                free(files);
                return ERROR;
            }

            // Store file info
            strncpy(files[count].filename, buff, PATH_MAX - 1);
            files[count].filename[PATH_MAX - 1] = '\0';
            memcpy(files[count].sha256, sha256_hash, SHA256_DIGEST_LENGTH);
            count++;
        }
    }

    closedir(pdir);

    // Check if any shell files were found
    if (count == 0) {
        fprintf(stderr, N_("Error: No shell files found in %s\n"), dir_path);
        free(files);
        return ERROR;
    }

    // Write output file
    FILE *output = fopen("config_sha256.c", "w");
    if (!output) {
        fprintf(stderr, N_("Error: Failed to create output file: %s\n"), strerror(errno));
        free(files);
        return ERROR;
    }

    fprintf(output, "#include <openssl/sha.h>\n\n");
    fprintf(output, "unsigned char config_sha256[][SHA256_DIGEST_LENGTH] = {\n");
    for (int i = 0; i < count; i++) {
        fprintf(output, "    {");
        for (int j = 0; j < SHA256_DIGEST_LENGTH; j++) {
            fprintf(output, "0x%02X", files[i].sha256[j]);
            if (j < SHA256_DIGEST_LENGTH - 1)
                fprintf(output, ",");
        }
        fprintf(output, "}, // %s\n", files[i].filename);
    }
    fprintf(output, "};\n\n");
    fprintf(output, "int config_sha256_count = %d;\n", count);

    fclose(output);
    free(files);

    printf("Successfully generated config_sha256.c with %d shell files\n", count);
    return 0;
}
