#ifndef sjr_deb822_h_
#define sjr_deb822_h_ 1

#include <sjr/parser.h>

#include <istream>
#include <string>

#include <iostream>

namespace sjr {
namespace deb822 {

template<
	typename CharT_ = char,
	typename Traits_ = std::char_traits<CharT_> >
class key :
	public std::basic_string<CharT_, Traits_>
{
public:
	typedef CharT_ char_type;
	typedef Traits_ traits_type;
	typedef std::basic_string<char_type, traits_type> string_type;

	key(void) : string_type() { }
	key(string_type const &k) : string_type(k) { }
};

template<
	typename CharT_ = char,
	typename Traits_ = std::char_traits<CharT_> >
class value :
	public std::basic_string<CharT_, Traits_>
{
public:
	typedef CharT_ char_type;
	typedef Traits_ traits_type;
	typedef std::basic_string<char_type, traits_type> string_type;

	value(void) : string_type() { }
	value(string_type const &k) : string_type(k) { }
};

template<
	typename CharT_ = char,
	typename Traits_ = std::char_traits<CharT_> >
class pair
{
public:
	typedef sjr::deb822::key<CharT_, Traits_> key_type;
	typedef sjr::deb822::value<CharT_, Traits_> value_type;

	key_type key;
	value_type value;
};

}

template<
	typename CharT_ = char,
	typename Traits_ = std::char_traits<CharT_> >
class deb822_parser_impl :
	public parser_impl<deb822::pair<CharT_, Traits_> >
{
public:
	typedef CharT_ char_type;
	typedef Traits_ traits_type;

	typedef std::basic_string<char_type, traits_type> string_type;
	typedef std::basic_istream<char_type, traits_type> stream_type;

	deb822_parser_impl(stream_type &);
	virtual ~deb822_parser_impl(void) throw();

	typedef parser_impl<deb822::pair<CharT_, Traits_> > impl_type;

	typedef typename impl_type::value_type value_type;
	typedef typename impl_type::reference reference;
	typedef typename impl_type::pointer pointer;

	virtual reference operator*(void) const;
	virtual pointer operator->(void) const;
	virtual deb822_parser_impl &operator++(void);
	virtual bool operator!(void) const throw();

private:
	stream_type &stream;
	value_type obj;
	bool fake_failure;
};

template<typename CharT_, typename Traits_>
deb822_parser_impl<CharT_, Traits_>::deb822_parser_impl(
		std::basic_istream<char_type, traits_type> &stream) :
	stream(stream),
	fake_failure(false)
{
	stream >> std::noskipws;
	++*this;
	return;
}

template<typename CharT_, typename Traits_>
deb822_parser_impl<CharT_, Traits_>::~deb822_parser_impl(void) throw()
{
	if(fake_failure)
		stream.clear(stream.rdstate() & ~std::ios::failbit);
	return;
}

template<typename CharT_, typename Traits_>
typename deb822_parser_impl<CharT_, Traits_>::reference
deb822_parser_impl<CharT_, Traits_>::operator*(void) const
{
	return obj;
}

template<typename CharT_, typename Traits_>
typename deb822_parser_impl<CharT_, Traits_>::pointer
deb822_parser_impl<CharT_, Traits_>::operator->(void) const
{
	return &obj;
}

template<typename CharT_, typename Traits_>
deb822_parser_impl<CharT_, Traits_> &
deb822_parser_impl<CharT_, Traits_>::operator++(void)
{
	typename stream_type::sentry sentry(stream);
	if(!sentry)
		return *this;

	string_type line;
	if(!std::getline(stream, line))
		return *this;
	
	if(line.empty())
	{
		stream.setstate(std::ios::failbit);
		fake_failure = true;
		return *this;
	}

	typedef typename string_type::size_type pos_type;

	pos_type const colon = line.find(':');
	if(colon == string_type::npos)
	{
		stream.setstate(std::ios::failbit|std::ios::badbit);
		return *this;
	}

	typename value_type::key_type key = line.substr(0, colon);
	typename value_type::value_type value = line.substr(
			line.find_first_not_of(" ", colon + 1));

	while(stream.peek() == traits_type::to_int_type(' '))
	{
		if(!std::getline(stream, line))
			return *this;
		if(line.empty())
		{
			stream.setstate(std::ios::failbit);
			fake_failure = true;
			return *this;
		}

		if(line == " .")
			line = " ";

		line[0] = '\n';
		value += line;
	}

	obj.key = key;
	obj.value = value;

	return *this;
}

template<typename CharT_, typename Traits_>
bool deb822_parser_impl<CharT_, Traits_>::operator!(void) const throw()
{
	return !stream;
}

typedef deb822_parser_impl<> deb822_parser;
typedef deb822_parser_impl<wchar_t> deb822_wparser;

}

#endif
