/*
 * Copyright (C) 2001-2024 Jacek Sieka, arnetheduck on gmail point com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef DCPLUSPLUS_DCPP_QUEUE_ITEM_H
#define DCPLUSPLUS_DCPP_QUEUE_ITEM_H

#include <airdcpp/queue/QueueItemBase.h>
#include <airdcpp/queue/QueueDownloadInfo.h>

#include <airdcpp/core/classes/FastAlloc.h>
#include <airdcpp/core/classes/IncrementingIdCounter.h>
#include <airdcpp/user/HintedUser.h>
#include <airdcpp/hash/value/MerkleTree.h>
#include <airdcpp/core/classes/Segment.h>
#include <airdcpp/util/Util.h>

namespace dcpp {

class QueueItem : public QueueItemBase {
public:
	using TokenMap = unordered_map<QueueToken, QueueItemPtr>;
	using StringMap = unordered_map<string *, QueueItemPtr, noCaseStringHash, noCaseStringEq>;
	using TTHMap = unordered_multimap<TTHValue *, QueueItemPtr>;
	using ItemBoolList = vector<pair<QueueItemPtr, bool>>;

	struct HashComp {
		explicit HashComp(const TTHValue& s) : a(s) { }
		bool operator()(const QueueItemPtr& q) const noexcept { return a == q->getTTH(); }

		HashComp& operator=(const HashComp&) = delete;
	private:
		const TTHValue& a;
	};

	struct AlphaSortOrder {
		bool operator()(const QueueItemPtr& left, const QueueItemPtr& right) const noexcept;
	};

	struct SizeSortOrder {
		bool operator()(const QueueItemPtr& left, const QueueItemPtr& right) const noexcept;
	};

	struct PrioSortOrder {
		bool operator()(const QueueItemPtr& left, const QueueItemPtr& right) const noexcept;
	};

	enum FileFlags {
		/** Normal download, no flags set */
		FLAG_NORMAL				= 0x00, 
		/** This is a user file listing download */
		FLAG_USER_LIST			= 0x01,
		/** The file list is downloaded to use for directory download (used with USER_LIST) */
		FLAG_DIRECTORY_DOWNLOAD = 0x02,
		/** The file is downloaded to be viewed in the gui */
		FLAG_CLIENT_VIEW		= 0x04,
		/** Match the queue against this list */
		FLAG_MATCH_QUEUE		= 0x08,
		/** The file list downloaded was actually an .xml.bz2 list */
		FLAG_XML_BZLIST			= 0x10,
		/** Only download a part of the file list */
		FLAG_PARTIAL_LIST 		= 0x20,
		/** Open directly with an external program after the file has been downloaded */
		FLAG_OPEN				= 0x40,
		/** Recursive partial list */
		FLAG_RECURSIVE_LIST		= 0x80,
		/** TTH list for partial bundle sharing */
		FLAG_TTHLIST_BUNDLE		= 0x100,
		/** A private file that won't be added in share and it's not available via partial sharing */
		FLAG_PRIVATE			= 0x200,
	};

	enum Status {
		STATUS_NEW, // not added in queue yet
		STATUS_QUEUED,
		STATUS_DOWNLOADED,
		STATUS_VALIDATION_RUNNING, // the file is being validated by the completion hooks
		STATUS_VALIDATION_ERROR, // hook validation failed (see the error pointer for more information)
		STATUS_COMPLETED, // no validation errors, ready for sharing
	};

	static bool isFailedStatus(Status aStatus) noexcept;

	class Source : public Flags {
	public:
		enum {
			FLAG_NONE				= 0x00,
			FLAG_FILE_NOT_AVAILABLE = 0x01,
			FLAG_REMOVED			= 0x04,
			FLAG_NO_TTHF			= 0x08,
			FLAG_BAD_TREE			= 0x10,
			FLAG_SLOW_SOURCE		= 0x20,
			FLAG_NO_TREE			= 0x40,
			FLAG_NO_NEED_PARTS		= 0x80,
			FLAG_PARTIAL			= 0x100,
			FLAG_TTH_INCONSISTENCY	= 0x200,
			FLAG_UNTRUSTED			= 0x400,
			FLAG_MASK				= FLAG_FILE_NOT_AVAILABLE
				| FLAG_REMOVED | FLAG_BAD_TREE | FLAG_SLOW_SOURCE
				| FLAG_NO_TREE | FLAG_TTH_INCONSISTENCY | FLAG_UNTRUSTED
		};

		Source(const HintedUser& aUser) : user(aUser) { }

		bool operator==(const UserPtr& aUser) const { return user == aUser; }

		// Update the hinted download hub URL based if the provided one can't be used
		bool updateDownloadHubUrl(const OrderedStringSet& aOnlineHubs, string& hubUrl_, bool aAllowUrlChange) const noexcept;

		// Update the hinted source URL
		void setHubUrl(const string& aHubUrl) noexcept;

		static string formatError(const Flags& aFlags) noexcept;
		const HintedUser& getUser() const noexcept {
			return user;
		}

		const OrderedStringSet& getBlockedHubs() const noexcept {
			return blockedHubs;
		}

		void addBlockedHub(const string& aHubUrl) noexcept {
			blockedHubs.insert(aHubUrl);
		}

		const PartsInfo* getPartsInfo() const noexcept {
			return partsInfo.empty() ? nullptr : &partsInfo;
		}

		void setPartsInfo(const PartsInfo& aPartsInfo) noexcept {
			partsInfo = aPartsInfo;
		}

		bool validateHub(const OrderedStringSet& aOnlineHubs, bool aAllowUrlChange, string& lastError_) const noexcept;
		bool validateHub(const string& aHubUrl, bool aAllowUrlChange) const noexcept;
	private:
		PartsInfo partsInfo;

		HintedUser user;

		OrderedStringSet blockedHubs;
	};

	using SourceList = vector<Source>;
	using SourceIter = SourceList::iterator;

	using SourceConstIter = SourceList::const_iterator;

	using SegmentSet = set<Segment>;
	using SegmentConstIter = SegmentSet::const_iterator;
	
	QueueItem(const string& aTarget, int64_t aSize, Priority aPriority, Flags::MaskType aFlag, time_t aAdded, const TTHValue& tth, const string& aTempTarget);

	~QueueItem() override;

	bool usesSmallSlot() const noexcept;

	// Select a random item from the list to search for alternates
	static QueueItemPtr pickSearchItem(const QueueItemList& aItems) noexcept;

	void save(OutputStream &save, string tmp, string b32tmp) const;
	int countOnlineUsers() const noexcept;
	void getOnlineUsers(HintedUserList& l) const noexcept;
	bool hasSegment(const QueueDownloadQuery& aQuery, string& lastError_, bool aAllowOverlap) noexcept;
	bool isPausedPrio() const noexcept override;

	SourceList& getSources() noexcept { return sources; }
	const SourceList& getSources() const noexcept { return sources; }
	SourceList& getBadSources() noexcept { return badSources; }
	const SourceList& getBadSources() const noexcept { return badSources; }

	string getTargetFileName() const noexcept;
	string getFilePath() const noexcept;

	SourceIter getSource(const UserPtr& aUser) noexcept { return find(sources.begin(), sources.end(), aUser); }
	SourceIter getBadSource(const UserPtr& aUser) noexcept { return find(badSources.begin(), badSources.end(), aUser); }
	SourceConstIter getSource(const UserPtr& aUser) const noexcept { return find(sources.begin(), sources.end(), aUser); }
	SourceConstIter getBadSource(const UserPtr& aUser) const noexcept { return find(badSources.begin(), badSources.end(), aUser); }

	bool isSource(const UserPtr& aUser) const noexcept { return getSource(aUser) != sources.end(); }
	bool isBadSource(const UserPtr& aUser) const noexcept { return getBadSource(aUser) != badSources.end(); }
	bool isBadSourceExcept(const UserPtr& aUser, Flags::MaskType exceptions, bool& isBad_) const noexcept;
	
	void getChunksVisualisation(vector<Segment>& running, vector<Segment>& downloaded, vector<Segment>& done) const noexcept;

	bool isChunkDownloaded(const Segment& aSegment) const noexcept;

	/**
	 * Is specified parts needed by this download?
	 */
	bool isNeededPart(const PartsInfo& partsInfo, int64_t blockSize) const noexcept;

	/**
	 * Get shared parts info, max 255 parts range pairs
	 */
	void getPartialInfo(PartsInfo& partialInfo, int64_t blockSize) const noexcept;

	uint64_t getDownloadedBytes() const noexcept;
	uint64_t getDownloadedSegments() const noexcept;
	double getDownloadedFraction() const noexcept;
	
	void addDownload(Download* d) noexcept;

	// Erase a single download
	void removeDownload(const Download* d) noexcept;

	// Erase all downloads from this user
	void removeDownloads(const UserPtr& aUser) noexcept;
	
	/** Next segment that is not done and not being downloaded, zero-sized segment returned if there is none is found */
	Segment getNextSegment(int64_t blockSize, int64_t wantedSize, int64_t aLastSpeed, const PartsInfo* aPartsInfo, bool allowOverlap) const noexcept;
	Segment checkOverlaps(int64_t blockSize, int64_t aLastSpeed, const PartsInfo* aPartsInfo, bool allowOverlap) const noexcept;
	
	void addFinishedSegment(const Segment& segment) noexcept;
	void resetDownloaded() noexcept;
	
	// Check that all segments have been downloaded (unsafe)
	bool segmentsDone() const noexcept;

	// The file has been flagged as downloaded
	bool isDownloaded() const noexcept;

	// File finished downloading and all validation hooks have completed (safe)
	bool isCompleted() const noexcept;

	bool isRunning() const noexcept {
		return !isWaiting();
	}
	bool isWaiting() const noexcept {
		return downloads.empty();
	}

	bool isFilelist() const noexcept;

	bool hasPartialSharingTarget() noexcept;

	string getListName() const noexcept;
	const string& getListDirectoryPath() const noexcept;
	string getStatusString(int64_t aDownloadedBytes, bool aIsWaiting) const noexcept;

	const string& getTempTarget() noexcept;
	void setTempTarget(const string& aTempTarget) noexcept;

	GETSET(TTHValue, tthRoot, TTH);
	GETSET(SegmentSet, done, Done);	
	IGETSET(uint64_t, fileBegin, FileBegin, 0);
	IGETSET(uint64_t, nextPublishingTime, NextPublishingTime, 0);
	IGETSET(uint8_t, maxSegments, MaxSegments, 1);
	IGETSET(BundlePtr, bundle, Bundle, nullptr);
	GETSET(string, lastSource, LastSource);
	IGETSET(Status, status, Status, STATUS_NEW);
	IGETSET(ActionHookRejectionPtr, hookError, HookError, nullptr);
	
	Priority calculateAutoPriority() const noexcept;

	uint64_t getAverageSpeed() const noexcept;
	uint64_t getSecondsLeft() const noexcept;

	int64_t getBlockSize() noexcept;
	void setBlockSize(int64_t aBlockSize) noexcept { blockSize = aBlockSize; }

	QueueItem& operator=(const QueueItem&) = delete;

	static IncrementingIdCounter<QueueToken> idCounter;
private:
	friend class QueueManager;
	friend class UserQueue;
	SourceList sources;
	SourceList badSources;
	string tempTarget;

	void addSource(const HintedUser& aUser) noexcept;
	void blockSourceHub(const HintedUser& aUser) noexcept;
	bool validateHub(const UserPtr& aUser, const string& aUrl) const noexcept;
	void removeSource(const UserPtr& aUser, Flags::MaskType reason) noexcept;

	bool matchesDownloadType(QueueDownloadType aType) const noexcept;
	bool allowSegmentedDownloads() const noexcept;
	bool allowUrlChange() const noexcept;

	static uint8_t getMaxSegments(int64_t aFileSize) noexcept;

	int64_t blockSize = -1;
};

} // namespace dcpp

#endif // !defined(QUEUE_ITEM_H)