﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/clouddirectory/CloudDirectoryRequest.h>
#include <aws/clouddirectory/CloudDirectory_EXPORTS.h>
#include <aws/clouddirectory/model/ObjectAttributeUpdate.h>
#include <aws/clouddirectory/model/ObjectReference.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CloudDirectory {
namespace Model {

/**
 */
class UpdateObjectAttributesRequest : public CloudDirectoryRequest {
 public:
  AWS_CLOUDDIRECTORY_API UpdateObjectAttributesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateObjectAttributes"; }

  AWS_CLOUDDIRECTORY_API Aws::String SerializePayload() const override;

  AWS_CLOUDDIRECTORY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that is associated with the <a>Directory</a>
   * where the object resides. For more information, see <a>arns</a>.</p>
   */
  inline const Aws::String& GetDirectoryArn() const { return m_directoryArn; }
  inline bool DirectoryArnHasBeenSet() const { return m_directoryArnHasBeenSet; }
  template <typename DirectoryArnT = Aws::String>
  void SetDirectoryArn(DirectoryArnT&& value) {
    m_directoryArnHasBeenSet = true;
    m_directoryArn = std::forward<DirectoryArnT>(value);
  }
  template <typename DirectoryArnT = Aws::String>
  UpdateObjectAttributesRequest& WithDirectoryArn(DirectoryArnT&& value) {
    SetDirectoryArn(std::forward<DirectoryArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The reference that identifies the object.</p>
   */
  inline const ObjectReference& GetObjectReference() const { return m_objectReference; }
  inline bool ObjectReferenceHasBeenSet() const { return m_objectReferenceHasBeenSet; }
  template <typename ObjectReferenceT = ObjectReference>
  void SetObjectReference(ObjectReferenceT&& value) {
    m_objectReferenceHasBeenSet = true;
    m_objectReference = std::forward<ObjectReferenceT>(value);
  }
  template <typename ObjectReferenceT = ObjectReference>
  UpdateObjectAttributesRequest& WithObjectReference(ObjectReferenceT&& value) {
    SetObjectReference(std::forward<ObjectReferenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The attributes update structure.</p>
   */
  inline const Aws::Vector<ObjectAttributeUpdate>& GetAttributeUpdates() const { return m_attributeUpdates; }
  inline bool AttributeUpdatesHasBeenSet() const { return m_attributeUpdatesHasBeenSet; }
  template <typename AttributeUpdatesT = Aws::Vector<ObjectAttributeUpdate>>
  void SetAttributeUpdates(AttributeUpdatesT&& value) {
    m_attributeUpdatesHasBeenSet = true;
    m_attributeUpdates = std::forward<AttributeUpdatesT>(value);
  }
  template <typename AttributeUpdatesT = Aws::Vector<ObjectAttributeUpdate>>
  UpdateObjectAttributesRequest& WithAttributeUpdates(AttributeUpdatesT&& value) {
    SetAttributeUpdates(std::forward<AttributeUpdatesT>(value));
    return *this;
  }
  template <typename AttributeUpdatesT = ObjectAttributeUpdate>
  UpdateObjectAttributesRequest& AddAttributeUpdates(AttributeUpdatesT&& value) {
    m_attributeUpdatesHasBeenSet = true;
    m_attributeUpdates.emplace_back(std::forward<AttributeUpdatesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_directoryArn;
  bool m_directoryArnHasBeenSet = false;

  ObjectReference m_objectReference;
  bool m_objectReferenceHasBeenSet = false;

  Aws::Vector<ObjectAttributeUpdate> m_attributeUpdates;
  bool m_attributeUpdatesHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudDirectory
}  // namespace Aws
