/*******************************************************************************

  Recipient has requested a license and Intel Corporation ("Intel") is
  willing to grant a license for the software entitled Linux Base Driver
  for the Intel(R) PRO/1000 Familty of Adapters (e1000) (the "Software")
  being provided by Intel Corporation.
  
  The following definitions apply to this License:
  
  "Licensed Patents" means patent claims licensable by Intel Corporation
  which are necessarily infringed by the use or sale of the Software alone
  or when combined with the operating system referred to below.
  "Recipient" means the party to whom Intel delivers this Software.
  "Licensee" means Recipient and those third parties that receive a license
  to any operating system available under the GNU Public License version
  2.0 or later.
  
  Copyright (c) 1999-2001 Intel Corporation All rights reserved.
  
  The license is provided to Recipient and Recipient's Licensees under
  the following terms.
  
  Redistribution and use in source and binary forms of the Software,
  with or without modification, are permitted provided that the following
  conditions are met:
  
  Redistributions of source code of the Software may retain the above
  copyright notice, this list of conditions and the following disclaimer.
  Redistributions in binary form of the Software may reproduce the above
  copyright notice, this list of conditions and the following disclaimer in
  the documentation and/or other materials provided with the distribution.
  Neither the name of Intel Corporation nor the names of its contributors
  shall be used to endorse or promote products derived from this Software
  without specific prior written permission.
  
  Intel hereby grants Recipient and Licensees a non-exclusive, worldwide,
  royalty-free patent license under Licensed Patents to make, use, sell,
  offer to sell, import and otherwise transfer the Software, if any, in
  source code and object code form. This license shall include changes
  to the Software that are error corrections or other minor changes
  to the Software that do not add functionality or features when the
  Software is incorporated in any version of a operating system that has
  been distributed under the GNU General Public License 2.0 or later.
  This patent license shall apply to the combination of the Software and
  any operating system licensed under the GNU Public License version 2.0
  or later if, at the time Intel provides the Software to Recipient, such
  addition of the Software to the then publicly available versions of such
  operating system available under the GNU Public License version 2.0 or
  later (whether in gold, beta or alpha form) causes such combination to
  be covered by the Licensed Patents. The patent license shall not apply
  to any other combinations which include the Software. No hardware per
  se is licensed hereunder.
  
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
  THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR ITS CONTRIBUTORS
  BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  THE POSSIBILITY OF SUCH DAMAGE.

*******************************************************************************/

/**********************************************************************
 *                                                                     *
 * INTEL CORPORATION                                                   *
 *                                                                     *
 * This software is supplied under the terms of the license included   *
 * above.  All use of this software must be in accordance with the     *
 * terms of that license.                                              *
 *                                                                     *
 * Module Name:  diag.c                                               *
 *                                                                     *
 * Abstract:     Header file for data structure definition for common  *
 *               diagnostics support                                   *
 *                                                                     *
 **********************************************************************/

#include "e1000.h"

static u32 IntrCount;

void SetLoopBackMode(struct e1000_adapter *adapter, short LoopbackType);
idiag_pro_stat_t Osdep_LoopBackTest(struct e1000_adapter *adapter,
                idiag_e1000_diag_loopback_test_param_t *Param, int LbType);

extern void e1000_selective_hibernate_adapter(struct net_device *netdev);
extern void e1000_selective_wakeup_adapter(struct net_device *netdev);
extern void e1000_hibernate_adapter(struct net_device *netdev);
extern void e1000_wakeup_adapter(struct net_device *netdev);
extern int e1000_xmit_lbtest_frame(struct sk_buff *skb, struct e1000_adapter * adapter);
extern int e1000_rcv_lbtest_frame(struct e1000_adapter * adapter,
               unsigned int frame_size);
static void e1000_create_lbtest_frame(struct sk_buff *skb, 
               unsigned int frame_size);

#define RegPatternTest(Register, Mask, WriteVal)            \
        for (Pattern = 0; Pattern < Ntests; Pattern++)        \
        {                                                    \
            E1000_WRITE_REG(&adapter->shared, Register,                        \
                (TestPattern[Pattern] & WriteVal));            \
            ReadValue = E1000_READ_REG(&adapter->shared, Register);            \
            if (ReadValue != (TestPattern[Pattern] &         \
                        WriteVal & Mask))                    \
            {                                                \
                Param->Reg =                                \
                (adapter->shared.mac_type < e1000_82543) ?        \
                E1000_82542_##Register :            \
                E1000_##Register;        \
                Param->WriteValue =                         \
                (TestPattern[Pattern] & WriteVal);            \
                Param->ReadValue = ReadValue;                \
                return (IDIAG_PRO_STAT_TEST_FAILED);        \
            }                                                \
        }

#define RegSetAndCheck(Register, Mask, WriteVal)            \
        E1000_WRITE_REG(&adapter->shared, Register, WriteVal);                \
        ReadValue = E1000_READ_REG(&adapter->shared, Register);                \
        if ((WriteVal & Mask) != (ReadValue & Mask))        \
        {                                                    \
            Param->Reg =                                        \
            (adapter->shared.mac_type < e1000_82543) ?            \
            E1000_82542_##Register :                           \
            E1000_##Register;                              \
            Param->WriteValue = WriteVal & Mask;            \
            Param->ReadValue = ReadValue & Mask;            \
            return (IDIAG_PRO_STAT_TEST_FAILED);             \
        }
            
/****************************************************************************
* Name:       DiagRegTest 
*
* Description: This routine Performs diagnostic tests on the hardware
*         registers 
*
* Author:      IntelCorporation
*
* Born on Date:   1/23/2001 
*
* Arguments:   
*      adapter     Board dependent data structure 
*      Param       Param structure thru which failed diagnostic test
*                  results are reported.
*
* Returns: 
*     idiag_pro_stat_t    Status of the Diagnostic Test.
*
* Modification log:
* Date      Who  Description
* --------  ---  -------------------------------------------------------- 
*
****************************************************************************/

idiag_pro_stat_t
DiagRegTest(struct e1000_adapter *adapter,
    idiag_e1000_diag_reg_test_param_t *Param)
{
    u32 ReadValue, Pattern;
    u32 Ntests = 4;
    u32 TestPattern[4] = {0x5A5A5A5A, 0xA5A5A5A5, 0x00000000, 0xFFFFFFFF};
    u32 i;

    /* The status register is Read Only
     * So a write should fail.
     * Some bits that get toggled are ignored
     */
#ifdef DIAG_DEBUG
    DEBUGOUT("Register Test\n");
#endif
    ReadValue = (E1000_READ_REG(&adapter->shared, STATUS) & (0xFFFFF833));
    E1000_WRITE_REG(&adapter->shared, STATUS, (0xFFFFFFFF));
    if(ReadValue != (E1000_READ_REG(&adapter->shared, STATUS) & (0xFFFFF833))) {
            Param->Reg = E1000_STATUS;
            Param->WriteValue = 0xFFFFFFFF;
            Param->ReadValue = ReadValue;

        return(IDIAG_PRO_STAT_TEST_FAILED);
    }

    /* Register Pattern Test */
    RegPatternTest(FCAL, 0xFFFFFFFF, 0xFFFFFFFF);
    RegPatternTest(FCAH, 0x0000FFFF, 0xFFFFFFFF);
    RegPatternTest(FCT, 0x0000FFFF, 0xFFFFFFFF);
    RegPatternTest(VET, 0x0000FFFF, 0xFFFFFFFF);

    if (adapter->shared.mac_type >= e1000_82543) {
        RegSetAndCheck(RCTL, 0x06DFB3FE, 0xFFFFFFFF);
    }
    else
    {
        RegSetAndCheck(RCTL, 0xFFFFFFFF, 0x01FFFFFF);
    }

    if(adapter->shared.mac_type >= e1000_82543)
    {
        for (i = 0; i < E1000_RAR_ENTRIES; i++)
        {
            // RegPatternTest(Rar[i].Low, 0xFFFFFFFF, 0xFFFFFFFF);
            // RegPatternTest(Rar[i].High, 0x8003FFFF, 0xFFFFFFFF);
            RegPatternTest(RA + ((i << 1) << 2), 0xFFFFFFFF, 0xFFFFFFFF);
            RegPatternTest(RA + (((i << 1) + 1) << 2), 0x8003FFFF, 0xFFFFFFFF);
        }
    }

    for (i = 0; i < E1000_MC_TBL_SIZE; i++)
    {
        // RegPatternTest(Mta[i], 0xFFFFFFFF, 0xFFFFFFFF);
        RegPatternTest(MTA + (i << 2), 0xFFFFFFFF, 0xFFFFFFFF);
    }

    RegSetAndCheck(RCTL, 0xFFFFFFFF, 0x00000000);

    RegPatternTest(RDTR, 0x0000FFFF, 0xFFFFFFFF);

    if (adapter->shared.mac_type >= e1000_82543) {
        RegPatternTest(RDBAL, 0xFFFFFFF0, 0xFFFFFFFF);
    } else {
        RegPatternTest(RDBAL, 0xFFFFF000, 0xFFFFFFFF);
    }

    RegPatternTest(RDBAH, 0xFFFFFFFF, 0xFFFFFFFF);
    RegPatternTest(RDLEN, 0x000FFF80, 0x000FFFFF);
    RegPatternTest(RDH, 0x0000FFFF, 0x0000FFFF);
    RegPatternTest(RDT, 0x0000FFFF, 0x0000FFFF);

    RegPatternTest(FCRTH, 0x0000FFF8, 0x0000FFF8);

    RegPatternTest(FCTTV, 0x0000FFFF, 0x0000FFFF);

    if (adapter->shared.mac_type >= e1000_82543) 
    {
        RegPatternTest(TXCW, 0xC000FFFF, 0x0000FFFF);
    }
    else
    {
        RegPatternTest(TXCW, 0x0000FFFF, 0x0000FFFF);
    }

        
    RegSetAndCheck(RCTL, 0x06DFB3FE, 0x003FFFFB);
    RegSetAndCheck(TCTL, 0xFFFFFFFF, 0x00000000);
        
    RegPatternTest(TIPG, 0x3FFFFFFF, 0x3FFFFFFF);    
    
    if (adapter->shared.mac_type >= e1000_82543) 
    {    
        RegPatternTest(TDBAL, 0xFFFFFFF0, 0xFFFFFFFF);    
    }
    else
    {
        RegPatternTest(TDBAL, 0xFFFFF000, 0xFFFFFFFF);    
    }

    RegPatternTest(TDBAH, 0xFFFFFFFF, 0xFFFFFFFF);    

    RegPatternTest(TDLEN, 0x000FFF80, 0x000FFFFF);    
    
    if (adapter->shared.mac_type >= e1000_82543)
        RegPatternTest(TIDV, 0x0000FFFF, 0x0000FFFF);

    return (IDIAG_PRO_STAT_OK);
}

/****************************************************************************
* Name:       DiagFifoTest 
*
* Description: This routine Performs diagnostic tests on the Packet buffer  
*         Memory 
*
* Author:      IntelCorporation
*
* Born on Date:   1/23/2001 
*
* Arguments:   
*      adapter        Board dependent data structure 
*      Param        Param structure thru which failed diagnostic test
*                  results are reported.
*
* Returns: 
*     idiag_pro_stat_t    Status of the Diagnostic Test.
*
* Modification log:
* Date      Who  Description
* --------  ---  -------------------------------------------------------- 
*
****************************************************************************/
            
idiag_pro_stat_t
DiagFifoTest(struct e1000_adapter *adapter,
        idiag_e1000_diag_fifo_test_param_t *Param)
{
    return (IDIAG_PRO_STAT_OK);
}

/****************************************************************************
* Name:       DiagEepromTest 
*
* Description: This routine Performs diagnostic tests on the contents of  
*         on board EEPROM 
*
* Author:      IntelCorporation
*
* Born on Date:   1/23/2001 
*
* Arguments:   
*      adapter        Board dependent data structure 
*      Param        Param structure thru which failed diagnostic test
*                  results are reported.
*
* Returns: 
*     idiag_pro_stat_t    Status of the Diagnostic Test.
*
* Modification log:
* Date      Who  Description
* --------  ---  -------------------------------------------------------- 
*
****************************************************************************/

idiag_pro_stat_t
DiagEepromTest(struct e1000_adapter *adapter,
        idiag_e1000_diag_eeprom_test_param_t *Param)
{
    u16 Checksum = 0;
    u16 Iteration;

#ifdef DIAG_DEBUG
    DEBUGOUT("EEPROM Test\n");
#endif

    /* Read and add up the contents of the EEPROM */
    for (Iteration = 0; Iteration < (EEPROM_CHECKSUM_REG + 1); Iteration++)
             Checksum += e1000_read_eeprom(&adapter->shared, Iteration);
    
    /* If Checksum is not Correct return error else test passed */
    if(Checksum != (u16) EEPROM_SUM) {
        Param->ActualChecksum = Checksum;
        Param->ExpectedChecksum = EEPROM_SUM;
        return(IDIAG_PRO_STAT_TEST_FAILED);
    } else {
        return(IDIAG_PRO_STAT_OK);
    }
    
}

void
DiagIntrRoutine(int irq, void *data, struct pt_regs *regs)
{
    struct net_device *netdev = (struct net_device *)data;
    struct e1000_adapter * adapter = netdev->priv;
    uint32_t IcrContents;

    /* Disable interrupts */
    E1000_WRITE_REG(&adapter->shared, IMC, ~0);

    IcrContents = E1000_READ_REG(&adapter->shared, ICR);

    E1000_DBG("DiagIntr\n");

    IntrCount++;

    return;
}

/****************************************************************************
* Name:       DiagIntrTest 
*
* Description: This routine Performs diagnostic tests on the Interrupt 
*         registers  
*
* Author:      IntelCorporation
*
* Born on Date:   1/23/2001 
*
* Arguments:   
*      adapter        Board dependent data structure 
*      Param        Param structure thru which failed diagnostic test
*                  results are reported.
*
* Returns: 
*     idiag_pro_stat_t    Status of the Diagnostic Test.
*
* Modification log:
* Date      Who  Description
* --------  ---  -------------------------------------------------------- 
*
****************************************************************************/

idiag_pro_stat_t
DiagIntrTest(struct e1000_adapter *adapter,
        idiag_e1000_diag_intr_test_param_t *Param)
{
    u32 IcrContents;    
    u32 Count;    
    u32 Mask;    

#ifdef DIAG_DEBUG
    DEBUGOUT("Intr Test\n");
#endif
    
    IcrContents = E1000_READ_REG(&adapter->shared, ICR);

    *Param = IDIAG_E1000_INTR_TEST_OK;
    IntrCount = 0;

    /* Disable all the interrupts */
    E1000_WRITE_REG(&adapter->shared, IMC, 0xFFFFFFFF);

    msec_delay(10);

    /* Intr register is clear on read, 
     * So read the Icr register once more And verify the
     * Contents are zero. The contents should be zero
     * becoz all the interrupts are disbled and Icr 
     * was read earlier before clearing all its contents
     */
    IcrContents = E1000_READ_REG(&adapter->shared, ICR);

    if (IcrContents != 0)
    {
        /* Something is wrong, Icr has to be zero. 
         * Since it is not, we cannot execute
         * any further interrupt related tests.
         */ 
        *Param = IDIAG_E1000_INTR_TEST_NOT_EXEC;
        return (IDIAG_PRO_STAT_TEST_FAILED);
    }

    /* Test each interrupt */
    for(Count = 0; Count < 10; Count++)
    {
        /* Disable interrupts */
        E1000_WRITE_REG(&adapter->shared, IMC, 0xFFFFFFFF);
        msec_delay(10);

        /* clear the intr cause register */
        IcrContents = E1000_READ_REG(&adapter->shared, ICR);
        
        /* Test Interrupts while disabled */
        Mask = 0x0001 << Count;

        /* Disable an interrupt by writing 1 to 
         * corresponding bit of Imc 
         */
        E1000_WRITE_REG(&adapter->shared, IMC, (u32) Mask);
        msec_delay(10);

        /* Cause an interrupt by writing 1 to 
         * corresponding bit of Imc 
         */
        E1000_WRITE_REG(&adapter->shared, ICS, (u32) Mask);
        msec_delay(10);
        
        /* Read the Cause register contents */
        IcrContents = E1000_READ_REG(&adapter->shared, ICR);

        /* If there is an Intr reported over the Bus then
         * the test failed, becoz the caused intr was disabled
         * and should not be reported by the device on the bus
         */ 
        if(IntrCount > 0)
        {
            *Param = IDIAG_E1000_INTR_TEST_FAILED_WHILE_DISABLED;
            break;
        }

        /* Test Interrupts While enabled */
        E1000_WRITE_REG(&adapter->shared, IMS, (u32) Mask);
        msec_delay(10);
        
        E1000_WRITE_REG(&adapter->shared, ICS, (u32) Mask);
        msec_delay(10);

        IcrContents = E1000_READ_REG(&adapter->shared, ICR);

        /* The Intr was enabled So the device should have put
         * an intr on the bus.
         */                         
        if(IntrCount == 0) {
            *Param = IDIAG_E1000_INTR_TEST_FAILED_WHILE_ENABLED;
            break;
        }
        /* Reset the Count for next test */
        IntrCount = 0;
        
        /* Try to cause any other interrupt which is disabled */
        E1000_WRITE_REG(&adapter->shared, IMC, (u32) ~Mask);
        msec_delay(10);

        E1000_WRITE_REG(&adapter->shared, ICS, (u32) ~Mask);
        msec_delay(10);

        IcrContents = E1000_READ_REG(&adapter->shared, ICR);

        /* Intr should not be reported on the bus as the caused
         * intr's are disabled
         */
        if(IntrCount != 0) {
            *Param = IDIAG_E1000_INTR_TEST_FAILED_MASKED_ENABLED;
            break;
        }
    }

    return ((*Param == IDIAG_E1000_INTR_TEST_OK) ? IDIAG_PRO_STAT_OK:
            IDIAG_PRO_STAT_TEST_FAILED);
}


/****************************************************************************
* Name:       DiagLinkTest 
*
* Description: This routine reports Link Status  
*
* Author:      IntelCorporation
*
* Born on Date:   1/23/2001 
*
* Arguments:   
*      adapter        Board dependent data structure 
*      Param        Param structure thru which failed diagnostic test
*                  results are reported.
*
* Returns: 
*     idiag_pro_stat_t    Status of the Diagnostic Test.
*
* Modification log:
* Date      Who  Description
* --------  ---  -------------------------------------------------------- 
*
****************************************************************************/
        
idiag_pro_stat_t
DiagLinkTest(struct e1000_adapter *adapter,
        idiag_e1000_diag_link_test_param_t *Param)
{
#ifdef DIAG_DEBUG
    DEBUGOUT("Link Test\n");
#endif
    
    /* Check link status */
    adapter->shared.get_link_status = TRUE;
    e1000_check_for_link(&adapter->shared);

    if(E1000_READ_REG(&adapter->shared, STATUS) & E1000_STATUS_LU)
        *Param = IDIAG_E1000_LINK_TEST_UP;
    else
        *Param = IDIAG_E1000_LINK_TEST_DOWN;

    return IDIAG_PRO_STAT_OK;
}


/****************************************************************************
* Name:       DiagLoopbackTest 
*
* Description: This routine Performs Loopback diagnostic tests. 
*
* Author:      IntelCorporation
*
* Born on Date:   1/23/2001 
*
* Arguments:   
*      adapter        Board dependent data structure 
*      Param        Param structure thru which failed diagnostic test
*                  results are reported.
*
* Returns: 
*     idiag_pro_stat_t    Status of the Diagnostic Test.
*
* Modification log:
* Date      Who  Description
* --------  ---  -------------------------------------------------------- 
*
****************************************************************************/

idiag_pro_stat_t
DiagLoopbackTest(struct e1000_adapter *adapter,
        idiag_e1000_diag_loopback_test_param_t *Param)
{
    u32 IcrContents;

#ifdef DIAG_DEBUG
    DEBUGOUT(" LOOPBACK TEST \n");
#endif

    /* Disable Interrupts*/    
    E1000_WRITE_REG(&adapter->shared, IMC, 0xFFFFFFFF);

    /* Clear all interrupts */
    IcrContents = E1000_READ_REG(&adapter->shared, ICR);
    
    if(Param->mode & IDIAG_E1000_DIAG_MAC_LB) {
        /* Setup MAC loopback mode */
        SetLoopBackMode(adapter, IDIAG_E1000_DIAG_MAC_LB);

        /* Do the Loopback Test */
        Osdep_LoopBackTest(adapter, Param, IDIAG_E1000_DIAG_MAC_LB);
    }
    if(Param->mode & IDIAG_E1000_DIAG_TCVR_LB) {
        /* Setup External loopback mode */
        SetLoopBackMode(adapter, IDIAG_E1000_DIAG_TCVR_LB);

        /* Do the Loopback Test */
        Osdep_LoopBackTest(adapter, Param, IDIAG_E1000_DIAG_TCVR_LB);
    }
    
    /* Check Params */
    if(Param->mac != IDIAG_E1000_LOOPBACK_TEST_OK &&
       Param->tcvr != IDIAG_E1000_LOOPBACK_TEST_OK &&
         Param->ext != IDIAG_E1000_LOOPBACK_TEST_OK) {
    
        return IDIAG_PRO_STAT_TEST_FAILED;
    } else {
        return IDIAG_PRO_STAT_OK;
    }
}

void SetLoopBackMode(struct e1000_adapter *adapter, short LoopbackType)
{
    u32 RctlContents;
    u16 MiiCtrlReg;

    RctlContents = E1000_READ_REG(&adapter->shared, RCTL);        

    /* Clear LBM bits */
    RctlContents &= 0xFFFFFF3F;        

    /* Store Bad Packets */
    RctlContents |= E1000_RCTL_SBP;
    RctlContents |= E1000_RCTL_BAM;

    switch(LoopbackType)
    {
        case IDIAG_E1000_DIAG_MAC_LB:
            RctlContents |= E1000_RCTL_LBM_MAC;
            E1000_WRITE_REG(&adapter->shared, RCTL, RctlContents);
            break;

        case IDIAG_E1000_DIAG_TCVR_LB:
            RctlContents |= E1000_RCTL_LBM_TCVR;
            E1000_WRITE_REG(&adapter->shared, RCTL, RctlContents);
            if(adapter->shared.media_type == e1000_media_type_copper) {
                    MiiCtrlReg = e1000_read_phy_reg(&adapter->shared, PHY_CTRL);
                    MiiCtrlReg |= MII_CR_LOOPBACK;
                    e1000_write_phy_reg(&adapter->shared, PHY_CTRL, MiiCtrlReg);
                    msec_delay(100);
            }
            break;
    }
}

idiag_pro_stat_t
GetPhyInformation(struct e1000_adapter * adapter, 
                    struct device_diagnostics *Param)
{
#ifdef DIAG_DEBUG
    DEBUGOUT("Get PHY Register Information\n");
#endif
     if (Param == NULL)
        return IDIAG_PRO_STAT_BAD_PARAM;

#ifdef DIAG_DEBUG
    DEBUGOUT1("line_speed -              <%x>\n", adapter->link_speed);
    DEBUGOUT1("media_type -              <%x>\n", adapter->shared.media_type);
    DEBUGOUT1("cable_length -            <%x>\n", adapter->phy_info.cable_length);
    DEBUGOUT1("extended_10b_t_distance - <%x>\n", adapter->phy_info.extended_10bt_distance);
    DEBUGOUT1("cable_polarity -          <%x>\n", adapter->phy_info.cable_polarity);
    DEBUGOUT1("polarity_reversal -       <%x>\n", adapter->phy_info.polarity_correction);
    DEBUGOUT1("Idle_errors -             <%x>\n", adapter->idle_errors);
    DEBUGOUT1("receive_errors -          <%x>\n", adapter->receive_errors);
    DEBUGOUT1("mdi_x_mode -              <%x>\n", adapter->phy_info.mdix_mode);
    DEBUGOUT1("local_rx -                <%x>\n", adapter->phy_info.local_rx);
    DEBUGOUT1("remote_rx -               <%x>\n", adapter->phy_info.remote_rx);
    DEBUGOUT1("pci_mode -                <%x>\n", adapter->bus_type);
    DEBUGOUT1("bus_speed -               <%x>\n", adapter->bus_speed);
    DEBUGOUT1("slot_size -               <%x>\n", adapter->bus_width);
#endif

    Param->line_speed              = adapter->link_speed;
    Param->media_type              = adapter->shared.media_type;
    Param->cable_length            = adapter->phy_info.cable_length;
    Param->extended_10b_t_distance = adapter->phy_info.extended_10bt_distance;
    Param->cable_polarity          = adapter->phy_info.cable_polarity;
    Param->polarity_reversal       = adapter->phy_info.polarity_correction;
    Param->idle_errors             = adapter->idle_errors;
    Param->link_reset              = adapter->phy_info.link_reset;
    Param->receive_errors          = adapter->receive_errors;
    Param->mdi_x_mode              = adapter->phy_info.mdix_mode;
    Param->local_rx                = adapter->phy_info.local_rx;
    Param->remote_rx               = adapter->phy_info.remote_rx;
    Param->pci_mode                = adapter->shared.bus_type;
    Param->bus_speed               = adapter->shared.bus_speed;
    Param->slot_size               = adapter->shared.bus_width;
    
    return (IDIAG_PRO_STAT_OK);
}

/* OS dependent diagnostic code */

#include "e1000.h"

void
e1000_diag_ioctl(struct net_device *netdev, struct ifreq *ifr)
{
    uint32_t icr;
    struct e1000_adapter * adapter = netdev->priv;
    uint8_t stop_adapter = 0;

    idiag_pro_data_t *DiagData = (idiag_pro_data_t *)ifr->ifr_data;

    if(DiagData->interface_ver != IDIAG_PRO_VERSION) {
        /* incorrect diagnostics interface version */
        DiagData->status = IDIAG_PRO_STAT_NOT_SUPPORTED;
        return;
    }

    if((DiagData->cmd != IDIAG_PRO_IDENTIFY_DRIVER) &&
       (DiagData->driver_id != IDIAG_E1000_DRIVER)) {
        /* incorrect driver identifier */
        DiagData->status = IDIAG_PRO_STAT_NOT_SUPPORTED;
        return;
    }

    if ((DiagData->cmd == IDIAG_E1000_DIAG_REG_TEST) || 
        (DiagData->cmd == IDIAG_E1000_DIAG_FIFO_TEST) ||
        (DiagData->cmd == IDIAG_E1000_DIAG_XSUM_TEST) ||
        (DiagData->cmd == IDIAG_E1000_DIAG_INTR_TEST) ||
        (DiagData->cmd == IDIAG_E1000_DIAG_LOOPBACK_TEST)){
        stop_adapter = 1;
    }
    
    if (stop_adapter == 1) {
        /* Save the adapter state before starting the test */
        e1000_hibernate_adapter(netdev);

        E1000_WRITE_REG(&adapter->shared, IMC, 0xFFFFFFFF);
        icr = E1000_READ_REG(&adapter->shared, ICR);

        /* Hook up diagnostic irq */
        if (request_irq(netdev->irq, &DiagIntrRoutine, SA_SHIRQ,
                    "e1000", netdev) != 0) {
            E1000_ERR("Requesting diagnostic interrrupt routine FAILED\n");
            return;
        }
    }

    switch (DiagData->cmd) {

    case IDIAG_PRO_IDENTIFY_DRIVER:
        DiagData->driver_id = IDIAG_E1000_DRIVER;
        DiagData->status = IDIAG_PRO_STAT_OK;
        break;    

    case IDIAG_E1000_DIAG_REG_TEST:
        DiagData->status = DiagRegTest(adapter, (idiag_e1000_diag_reg_test_param_t *)
                                       &DiagData->diag_param);
        break;

    case IDIAG_E1000_DIAG_FIFO_TEST:
        DiagData->status = DiagFifoTest(adapter, (idiag_e1000_diag_fifo_test_param_t *)
                                        &DiagData->diag_param);
        break;

    case IDIAG_E1000_DIAG_XSUM_TEST:
        DiagData->status = DiagEepromTest(adapter, (idiag_e1000_diag_eeprom_test_param_t *)
                                          &DiagData->diag_param);
        break;

    case IDIAG_E1000_DIAG_INTR_TEST:
        DiagData->status = DiagIntrTest(adapter, (idiag_e1000_diag_intr_test_param_t *)
                                        &DiagData->diag_param);
        break;

    case IDIAG_E1000_DIAG_LOOPBACK_TEST:

        /* If interface is not open, we open here because
        loopback test needs descriptor resources, etc. */ 
        e1000_selective_wakeup_adapter(netdev);

        /* Stop the interface from calling Tx entry point */
        netif_stop_queue(netdev);

        msec_delay(50);

        DiagData->status = DiagLoopbackTest(adapter, (idiag_e1000_diag_loopback_test_param_t *)
                                            &DiagData->diag_param);
                                        
        /* 
        Mask off interrupts. Prevent LSC (Link Status Change) interrupt
        from going to the bus after Phy Reset.
        We don't want to see 2 cases:
        case 1: Spurious interrupts. Linux fires up intr handler when irq is 
        unhooked.
        case 2: interrupt test failure. Unexpected fired interrupts will fail
        interrupt test.
        */
        E1000_WRITE_REG(&adapter->shared, IMC, ~0);

        /* Will generate LSC intr. Won't go to bus. */
        if (adapter->shared.media_type == e1000_media_type_copper)
            e1000_phy_reset(&adapter->shared);

        e1000_selective_hibernate_adapter(netdev);
        break;

    case IDIAG_E1000_DIAG_LINK_TEST:
        DiagData->status = DiagLinkTest(adapter, (idiag_e1000_diag_link_test_param_t *)
                                         &DiagData->diag_param);
        break;

    case IDIAG_E1000_DIAG_GET_PHY_INFORMATION:
        DiagData->status = GetPhyInformation(adapter, 
                    (struct device_diagnostics *) &DiagData->diag_param);
        break;

    default:
        DiagData->status = IDIAG_PRO_STAT_NOT_SUPPORTED;
        break;
    }

    if (stop_adapter == 1) {
        E1000_WRITE_REG(&adapter->shared, IMC, 0xFFFFFFFF);
        icr = E1000_READ_REG(&adapter->shared, ICR);

        /* Unhook diagnostic irq */
        free_irq(netdev->irq, netdev);

        e1000_wakeup_adapter(netdev);
    }

    return;
}

idiag_pro_stat_t
Osdep_LoopBackTest(struct e1000_adapter *adapter,
                idiag_e1000_diag_loopback_test_param_t *Param, int LbType)
{
    u16 xmit_stat, rcv_stat;
    struct sk_buff *skb;
    unsigned int frame_size = 1024;

    skb = alloc_skb(frame_size, GFP_ATOMIC);

    skb_put(skb, frame_size);

    e1000_create_lbtest_frame(skb, frame_size);
 
    e1000_clear_hw_cntrs(&adapter->shared);

    xmit_stat = e1000_xmit_lbtest_frame(skb, adapter);

    if (xmit_stat != 1) {
        switch(LbType) {
        case IDIAG_E1000_DIAG_MAC_LB:
            Param->mac = IDIAG_E1000_LOOPBACK_TEST_NOT_EXEC;
            break;
        case IDIAG_E1000_DIAG_TCVR_LB:
            Param->tcvr = IDIAG_E1000_LOOPBACK_TEST_NOT_EXEC;
            break;
        case IDIAG_E1000_DIAG_EXT_LB:
            Param->ext = IDIAG_E1000_LOOPBACK_TEST_NOT_EXEC;
            break;
        }
        return (IDIAG_PRO_STAT_TEST_FAILED);
    }

    msec_delay(100);

    rcv_stat = e1000_rcv_lbtest_frame(adapter, frame_size);
    
    if (rcv_stat != 1) {
        switch(LbType) {
        case IDIAG_E1000_DIAG_MAC_LB:
            Param->mac = IDIAG_E1000_LOOPBACK_TEST_FAILED;
            break;
        case IDIAG_E1000_DIAG_TCVR_LB:
            Param->tcvr = IDIAG_E1000_LOOPBACK_TEST_FAILED;
            break;
        case IDIAG_E1000_DIAG_EXT_LB:
            Param->ext = IDIAG_E1000_LOOPBACK_TEST_FAILED;
            break;
        }
        return (IDIAG_PRO_STAT_TEST_FAILED);
    }

    switch(LbType) {
    case IDIAG_E1000_DIAG_MAC_LB:
        Param->mac = IDIAG_E1000_LOOPBACK_TEST_OK;
        break;
    case IDIAG_E1000_DIAG_TCVR_LB:
        Param->tcvr = IDIAG_E1000_LOOPBACK_TEST_OK;
        break;
    case IDIAG_E1000_DIAG_EXT_LB:
        Param->ext = IDIAG_E1000_LOOPBACK_TEST_OK;
        break;
    }
    return (IDIAG_PRO_STAT_OK);
}

static void
e1000_create_lbtest_frame(struct sk_buff *skb, unsigned int frame_size)
{
    memset(skb->data, 0xFF, frame_size);
    frame_size = (frame_size % 2) ? (frame_size - 1) : frame_size;
    memset(&skb->data[frame_size/2], 0xAA, frame_size/2 - 1);
    memset(&skb->data[frame_size/2 + 10], 0xBE, 1);
    memset(&skb->data[frame_size/2 + 12], 0xAF, 1);
} 
