; Cute Mouse Driver - a tiny mouse driver
; Copyright (c) 1997-2002 Nagy Daniel <nagyd@users.sourceforge.net>
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;

CTMVER		equ	"2.0"
CTMRELEASE	equ	"2.0 alpha 2"
driverversion	equ	705h	; Microsoft driver version

WARN
%NOINCL
LOCALS
.model tiny
		assume	ss:nothing
;.286
;.386

FASTER_CODE	 = 0		; Optimize by speed instead size
OVERFLOW_PROTECT = 0		; Prevent internal variables overflow
FOOLPROOF	 = 1		; Check passed to driver arguments correctness

USE_286		equ	<(@CPU and 4)>
USE_386		equ	<(@CPU and 8)>

_ARG_DI_	equ	<word ptr [bp]>
_ARG_SI_	equ	<word ptr [bp+2]>
_ARG_BP_	equ	<word ptr [bp+4]>

IF USE_286

_ARG_BX_	equ	<word ptr [bp+8]>
_ARG_DX_	equ	<word ptr [bp+10]>
_ARG_CX_	equ	<word ptr [bp+12]>
_ARG_AX_	equ	<word ptr [bp+14]>
_ARG_ES_	equ	<word ptr [bp+16]>
_ARG_DS_	equ	<word ptr [bp+18]>
_ARG_OFFS_	=	24

PUSHALL		equ	<pusha>
POPALL		equ	<popa>

ELSE ;---------- USE_286

_ARG_BX_	equ	<word ptr [bp+6]>
_ARG_DX_	equ	<word ptr [bp+8]>
_ARG_CX_	equ	<word ptr [bp+10]>
_ARG_AX_	equ	<word ptr [bp+12]>
_ARG_ES_	equ	<word ptr [bp+14]>
_ARG_DS_	equ	<word ptr [bp+16]>
_ARG_OFFS_	=	22

PUSHALL		equ	<push	ax cx dx bx bp si di>
POPALL		equ	<pop	di si bp bx dx cx ax>

ENDIF ;---------- USE_286

OPCODE_MOV_AX	equ	<db 0B8h>
OPCODE_MOV_CX	equ	<db 0B9h>
OPCODE_MOV_DX	equ	<db 0BAh>
OPCODE_MOV_BX	equ	<db 0BBh>
OPCODE_MOV_AL	equ	<db 0B0h>
OPCODE_MOV_CL	equ	<db 0B1h>
OPCODE_MOV_DL	equ	<db 0B2h>
OPCODE_MOV_AH	equ	<db 0B4h>

include asm.mac
include hll.mac
include macro.mac
include BIOS/area0.def
include convert/digit.mac
include DOS/MCB.def
include DOS/PSP.def
include DOS/file.mac
include DOS/io.mac
include DOS/mem.mac
include hard/PIC8259A.def
include hard/UART.def

PS2serv		macro	serv,errlabel
		mov	ax,serv
		int	15h
	IFNB <errlabel>
		jc	errlabel
		test	ah,ah
		jnz	errlabel
	ENDIF
endm

POINT		struc
  X		dw	0
  Y		dw	0
ends

.code
		org	0
TSRstart	label
		org	100h
start:		jmp	real_start


; UNINITIALIZED DATA 

;!!! WARNING: don't init variables in uninitialized data section because
;		it will not be present in the executable image

		org	PSP_TSR		; reuse part of PSP
TSRdata?	label

;----- application state begins here -----

;!!! WARNING: variables order between StartSaveArea and LenDefArea must be
;		syncronized with variables order after StartDefArea

		evendata
StartSaveArea = $

mickey8		POINT	?		; mickeys per 8 pixel ratios
;;*doublespeed	dw	?		; double-speed threshold (mickeys/sec)
senscoeff	POINT	?		; mickeys sensitivity, ~[1/3..3.0]*256
sensval		POINT	?		; original 001A sensitivity values-1
startscan	dw	?		; screen mask/cursor start scanline
endscan		dw	?		; cursor mask/cursor end scanline

;---------- hotspot, screenmask and cursormask must follow as is

hotspot		POINT	?		; cursor bitmap hot spot
screenmask	db	2*16 dup (?)	; user defined screen mask
cursormask	db	2*16 dup (?)	; user defined cursor mask
nocursorcnt	db	?		; 0=cursor enabled, else hide counter
;;*nolightpen?	db	?		; 0=emulate light pen
		evendata
LenDefArea = $ - StartSaveArea		; initialized by softreset_21

rangemax	POINT	?		; horizontal/vertical range max
upleft		POINT	?		; upper left of update region
lowright	POINT	?		; lower right of update region
pos		POINT	?		; virtual cursor position
granpos		POINT	?		; granulated virtual cursor position
UIR@		dd	?		; user interrupt routine address

		evendata
StartClearArea = $

sensround	POINT	?		; rounding error in applying
					;  sensitivity for mickeys
rounderr	POINT	?		; same in conversion mickeys to pixels
		evendata
LenClearArea1 = $ - StartClearArea	; cleared by setpos_04

rangemin	POINT	?		; horizontal/vertical range min
		evendata
LenClearArea2 = $ - StartClearArea	; cleared by setupvideo

cursortype	db	?		; 0 - software, else hardware
callmask	db	?		; user interrupt routine call mask
mickeys		POINT	?		; mouse move since last access
BUTTLASTSTATE	struc
  counter	dw	?
  lastrow	dw	?
  lastcol	dw	?
ends
buttpress	BUTTLASTSTATE	?,?,?
buttrelease	BUTTLASTSTATE	?,?,?
wheel		BUTTLASTSTATE	?	; wheel counter since last access
wheelUIR	db	?		; wheel counter for UIR
		evendata
LenClearArea3 = $ - StartClearArea	; cleared by softreset_21
LenSaveArea = $ - StartSaveArea

;----- registers values for RIL -----

;!!! WARNING: registers order and RGROUPDEF contents must be fixed

		evendata
StartVRegsArea = $

regs_SEQC	db	5 dup (?)
reg_MISC	db	?
regs_CRTC	db	25 dup (?)
regs_ATC	db	21 dup (?)
regs_GRC	db	9 dup (?)
reg_FC		db	?
reg_GPOS1	db	?
reg_GPOS2	db	?
		evendata
LenVRegsArea = $ - StartVRegsArea

		evendata
StartDefVRegsArea = $

def_SEQC	db	5 dup (?)
def_MISC	db	?
def_CRTC	db	25 dup (?)
def_ATC		db	21 dup (?)
def_GRC		db	9 dup (?)
def_FC		db	?
def_GPOS1	db	?
def_GPOS2	db	?
		evendata
ERRIF ($-StartDefVRegsArea ne LenVRegsArea) "VRegs area contents corrupted!"

;----- old interrupt vectors -----

oldint33	dd	?		; old INT 33 handler address
oldIRQaddr	dd	?		; old IRQ handler address


; INITIALIZED DATA 

		evendata
TSRdata		label

StartDefArea = $
		POINT	<8,16>			; mickey8
		POINT	<1*256,1*256>		; senscoeff
		POINT	<49,49>			; sensval
;;*		dw	64			; doublespeed
		dw	77FFh,7700h		; startscan, endscan
		POINT	<0,0>			; hotspot
		dw	0011111111111111b	; screenmask
		dw	0001111111111111b
		dw	0000111111111111b
		dw	0000011111111111b
		dw	0000001111111111b
		dw	0000000111111111b
		dw	0000000011111111b
		dw	0000000001111111b
		dw	0000000000111111b
		dw	0000000000011111b
		dw	0000000111111111b
		dw	0000000011111111b
		dw	0011000011111111b
		dw	1111100001111111b
		dw	1111100001111111b
		dw	1111110011111111b
		dw	0000000000000000b	; cursormask
		dw	0100000000000000b
		dw	0110000000000000b
		dw	0111000000000000b
		dw	0111100000000000b
		dw	0111110000000000b
		dw	0111111000000000b
		dw	0111111100000000b
		dw	0111111110000000b
		dw	0111110000000000b
		dw	0110110000000000b
		dw	0100011000000000b
		dw	0000011000000000b
		dw	0000001100000000b
		dw	0000001100000000b
		dw	0000000000000000b
		db	1			; nocursorcnt
;;*		db	0			; nolightpen?
		evendata
ERRIF ($-StartDefArea ne LenDefArea) "Defaults area contents corrupted!"

;----- driver and video state begins here -----

		evendata
granumask	POINT	<-1,-1>

textbuf		label	word
buffer@		dd	?		; pointer to screen sprite copy
cursor@		dw	-1		; cursor sprite offset in videoseg;
					; -1=cursor not drawn
videoseg	dw	0		; 0=not supported video mode

UIRunlock	db	1		; 0=user intr routine is in progress
videolock	db	1		; drawing: 1=ready,0=busy,-1=busy+queue
newcursor	db	0		; 1=force cursor redraw

;----- table of pointers to registers values for RIL -----

REGSET		struc
  rgroup	dw	?
  regnum	db	?
  regval	db	?
ends
		evendata
		dw	(vdata1end-vdata1)/(size REGSET)
vdata1		REGSET	<10h,1>,<10h,3>,<10h,4>,<10h,5>,<10h,8>,<08h,2>
vdata1end	label
		dw	(vdata2end-vdata2)/(size REGSET)
vdata2		REGSET	<10h,1,0>,<10h,4,0>,<10h,5,1>,<10h,8,0FFh>,<08h,2,0Fh>
vdata2end	label

RGROUPDEF	struc
  port@		dw	?
  regs@		dw	?
  def@		dw	?
  regscnt	db	1
  rmodify?	db	0
ends
		evendata
videoregs@	label
	RGROUPDEF <3D4h,regs_CRTC,def_CRTC,25>	; CRTC
	RGROUPDEF <3C4h,regs_SEQC,def_SEQC,5>	; Sequencer
	RGROUPDEF <3CEh,regs_GRC, def_GRC, 9>	; Graphics controller
	RGROUPDEF <3C0h,regs_ATC, def_ATC, 20>	; VGA attrib controller
	RGROUPDEF <3C2h,reg_MISC, def_MISC>	; VGA misc output and input
	RGROUPDEF <3DAh,reg_FC,   def_FC>	; Feature Control
	RGROUPDEF <3CCh,reg_GPOS1,def_GPOS1>	; Graphics 1 Position
	RGROUPDEF <3CAh,reg_GPOS2,def_GPOS2>	; Graphics 2 Position


; IRQ HANDLERS 

;

IRQhandler	proc
		assume	ds:nothing,es:nothing
		cld
		push	ds es
		PUSHALL
		MOVSEG	ds,cs,,DGROUP
		OPCODE_MOV_CX
IOdone		db	?,0			; processed bytes counter
IRQproc		label	byte			; "mov al,OCW2<OCW2_EOI>"
		j	PS2proc			;  if serial mode
		out	PIC1_OCW2,al		; {20h} end of interrupt

		OPCODE_MOV_DX
IO_address	dw	?			; UART IO address
		push	dx
		movidx	dx,LSR_index
		 in	al,dx			; {3FDh} LSR: get status
		xchg	bx,ax			; OPTIMIZE: instead MOV BL,AL
		pop	dx
		movidx	dx,RBR_index
		 in	al,dx			; {3F8h} flush receive buffer

		testflag bl,mask LSR_break+mask LSR_FE+mask LSR_OE
	if_ nz					; if break/framing/overrun
		xor	cx,cx			;  errors then restart
		mov	[IOdone],cl		;  sequence: clear counter
	end_
		shr	bl,LSR_RBF+1
	if_ carry				; process data if data ready
		db	0E8h			; CALL NEAR to mouseproc
mouseproc	dw	MSMproc-$-2
	end_
@rethandler_:	jmp	@rethandler
IRQhandler	endp
		assume	ds:DGROUP

;
;!!! WARNING: below buffer for copy of screen sprite when serial protocols

bufferSERIAL	label				; requires 3*16 bytes

;
;				Enable PS/2
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	AX, BX, DX
; Call:	INT 15/C200, INT 21/25
;
enablePS2	proc
		DOSSetIntr 68h+12,,,DGROUP:IRQhandler
		mov	bh,1			; set mouse on
		PS2serv	0C200h
		ret
enablePS2	endp

;
;				Disable PS/2
;
;
; In:	none
; Out:	none
; Use:	oldIRQaddr
; Modf:	AX, BX, DX
; Call:	INT 15/C200, INT 21/25
;
disablePS2	proc
		mov	bh,0			; set mouse off
		PS2serv	0C200h
		push	ds
		lds	dx,[oldIRQaddr]
		DOSSetIntr 68h+12
		pop	ds
		ret
disablePS2	endp

;

PS2proc		proc
		in	al,60h

		push	ax
		out_	PIC2_OCW2,%OCW2<OCW2_EOI>	; {0A0h} end of interrupt
		out_	PIC2_OCW3,%OCW3<,,,OCW3_ISR>	; {0A0h} select ISR read mode
		in	al,PIC2_ISR			; {0A0h} read ISR
		or	al,al				; if last serviced
	if_ zero					;  IRQ in PIC2 then
		out_	PIC1_OCW2,%OCW2<OCW2_specEOI,,2>; {20h} EOI for IRQ2
	end_
		pop	ax

		jcxz	@@PS2_1
PS2WHEELCODE	label	byte	
@@PS2_2:	loop	@@PS2update		; "loop @PS2_3" if wheel
		mov	[PS2_X],al
		j	@@exitIRQnext

@PS2_3:		loop	@@PS2_4
		mov	[PS2_Y],al
		j	@@exitIRQnext

@@PS2_1:	xor	al,00001000b	; =8	; synchro check
		test	al,11001000b	; =0C8h	; if first byte and movement
	if_ zero				;   values not overflowed
		mov	[PS2_buttons],al
@@exitIRQnext:	inc	[IOdone]		; request next byte
	end_
		j	@rethandler_

@@PS2_4:	mov	ah,al
		OPCODE_MOV_AL
PS2_Y		db	?

@@PS2update:	;mov	ch,0
		mov	cl,al
		mov	[IOdone],ch		; request next 3/4 bytes
		OPCODE_MOV_AL
PS2_buttons	db	?
		OPCODE_MOV_BX
PS2_X		db	?,0

		testflag al,00100000b	; =20h	; check Y sign bit
	if_ nz
		mov	ch,-1
	end_
		testflag al,00010000b	; =10h	; check X sign bit
	if_ nz
		mov	bh,-1
	end_
		neg	cx			; reverse Y movement
		call	swapbuttons
		j	@rethandler_
PS2proc		endp
ERRIF ($-bufferSERIAL lt 3*16) "PS/2 handler too small for buffer!"

;
;!!! WARNING: below buffer for copy of screen sprite when PS2 protocol

bufferPS2	label				; requires 3*16 bytes

;
;			Enable serial interrupt in PIC
;
;
; In:	none
; Out:	none
; Use:	IO_address
; Modf:	AX, DX, SI, IOdone, MSLTbuttons
; Call:	INT 21/25
;
enableUART	proc
;---------- set new IRQ handler
		OPCODE_MOV_AX
IRQintnum	dw	2500h			; INT number of selected IRQ
		mov	dx,offset DGROUP:IRQhandler
		int	21h			; set INT in DS:DX

;---------- set communication parameters
		mov	si,[IO_address]
		movidx	dx,LCR_index,si
		 out_	dx,%LCR{LCR_DLAB=1}	; {3FBh} LCR: DLAB on
		xchg	dx,si			; 1200 baud rate
		 outw	dx,96			; {3F8h},{3F9h} divisor latch
		xchg	dx,si
		 OPCODE_MOV_AX
LCRset		 LCR	<0,,LCR_noparity,0,2>	; {3FBh} LCR: DLAB off, 7/8N1
		 MCR	<,,,1,1,1,1>		; {3FCh} MCR: DTR/RTS/OUTx on
		 out	dx,ax

;---------- prepare UART for interrupts
		movidx	dx,RBR_index,si,LCR_index
		 in	al,dx			; {3F8h} flush receive buffer
		movidx	dx,IER_index,si,RBR_index
		 out_	dx,%IER{IER_DR=1},%FCR<>; {3F9h} IER: enable DR intr
						; {3FAh} FCR: disable FIFO
		dec	ax			; OPTIMIZE: instead MOV AL,0
		mov	[IOdone],al
		mov	[MSLTbuttons],al
;----------
		in	al,PIC1_IMR		; {21h} get IMR
		db	24h			; AND AL,byte
notPIC1state	db	?			; clear bit to enable interrupt
		out	PIC1_IMR,al		; {21h} enable serial interrupts
		ret
enableUART	endp

;
;			Disable serial interrupt of PIC
;
;
; In:	none
; Out:	none
; Use:	IO_address, oldIRQaddr
; Modf:	AX, DX
; Call:	INT 21/25
;
disableUART	proc
		in	al,PIC1_IMR		; {21h} get IMR
		db	0Ch			; OR AL,byte
PIC1state	db	?			; set bit to disable interrupt
		out	PIC1_IMR,al		; {21h} disable serial interrupts
;----------
		movidx	dx,LCR_index,[IO_address] ; {3FBh} LCR: DLAB off
		 out_	dx,%LCR<>,%MCR<>	; {3FCh} MCR: DTR/RTS/OUT2 off
		movidx	dx,IER_index,,LCR_index
		 ;mov	al,IER<>
		 out	dx,al			; {3F9h} IER: interrupts off

;---------- restore old IRQ handler
		mov	ax,[IRQintnum]		; AH=25h
		push	ds
		lds	dx,[oldIRQaddr]
		int	21h			; set INT in DS:DX
		pop	ds
		ret
disableUART	endp

;
;		Process mouse bytes the Microsoft/Logitech way
;

MSLTproc	proc
		testflag al,01000000b	; =40h	; synchro check
	if_ nz					; if first byte
		mov	[IOdone],1		; request next 2/3 bytes
		mov	[MSLT_1],al
MSLTCODE1	label	byte			; "ret" if not LT/WM
		xchg	ax,cx			; OPTIMIZE: instead MOV AL,CL
		sub	al,3
	 if_ zero				; if first byte after 3 bytes
		mov	[MSLTbuttons],al	; then release middle button
	 end_
@@MSLTret:	ret
	end_

		jcxz	@@MSLTret		; skip nonfirst byte at start
		inc	[IOdone]		; request next byte
		loop	@@MSLT_3
		mov	[MSLT_X],al		; keep X movement LO
		ret

@@MSLT_3:	loop	@@LTWM_4
		xchg	cx,ax			; OPTIMIZE: instead MOV CL,AL
		OPCODE_MOV_AH
MSLT_1		db	?			; first byte of MS/LT protocol

		;mov	al,0			; CX was 0
		shr	ax,2			; bits 1-0 - X movement HI
		db	0Ch			; OR AL,byte
MSLT_X		db	?
		mov	bh,ah
		cbw
		xchg	bx,ax			; BX=X movement

		mov	al,0
		shr	ax,2			; bits 3-2 - Y movement HI
		or	al,cl
		mov	cl,ah			; bits 5-4 - L/R buttons
		cbw
		xchg	cx,ax			; CX=Y movement

		flipflag al,[MSLTbuttons]
		maskflag al,00000011b	; =3	; L/R buttons change mask
		mov	dl,al
		or	dl,bl			; nonzero if L/R buttons state
		or	dl,cl			;  changed or mouse moved
MSLTCODE2	label	byte
		j	@@MSLTupdate		; "jnz" if MS3
		setflag	al,00000100b	; =4	; empty event toggles button
		j	@@MSLTupdate

@@LTWM_4:	;mov	ch,0
		mov	[IOdone],ch		; request next 3/4 bytes
		mov	ah,al
MSLTCODE3	label	byte			; if LT "mov cl,3" else
		mov	cl,3			; if WM "mov cl,2" else "ret"
		shr	al,cl			; (MS defines only 3 bytes)
		flipflag al,[MSLTbuttons]
		maskflag al,00000100b	; =4
		jz	@@MSLTret		; exit if button 3 not changed
		xor	cx,cx
		xor	bx,bx

@@MSLTupdate:	db	034h			; XOR AL,byte
MSLTbuttons	db	?			; buttons state for MS3/LT
		mov	[MSLTbuttons],al
		j	swapbuttons
MSLTproc	endp

;
;		Process mouse bytes the Mouse Systems way
;

MSMproc		proc
		jcxz	@@MSM_1
		cbw
		dec	cx
		jz	@@MSM_2
		dec	cx
		jz	@@MSM_3
		loop	@@MSM_5

@@MSM_4:	add	ax,[MSM_X]
@@MSM_2:	mov	[MSM_X],ax
		j	@@MSMnext

@@MSM_1:	flipflag al,10000111b	; =87h	; sync check: AL should
		testflag al,11111000b	; =0F8h	;  be equal to 10000lmr
		jnz	@@MSMret
		mov	[MSM_buttons],al	; bits 2-0 - L/M/R buttons
		;j	@@MSMnext

@@MSM_3:	mov	[MSM_Y],ax
@@MSMnext:	inc	[IOdone]		; request next byte
@@MSMret:	ret

@@MSM_5:	;mov	ch,0
		mov	[IOdone],ch		; request next 5 bytes
		OPCODE_MOV_CL
MSM_buttons	db	?
		db	05h			; ADD AX,word
MSM_Y		dw	?
		OPCODE_MOV_BX
MSM_X		dw	?
		xchg	cx,ax
		neg	cx
		testflag al,00000110b	; =6	; check the L and M buttons
	if_ odd					; if buttons not same
		flipflag al,00000110b	; =6	; swap them
	end_
		;j	swapbuttons
MSMproc		endp
ERRIF ($-bufferPS2 lt 3*16) "Serial handler too small for buffer!"

;
;			Update mouse status
;
;
; In:	AL			(new buttons state)
;	AH			(wheel movemement)
;	BX			(X mouse movement)
;	CX			(Y mouse movement)
; Out:	none
; Use:	callmask, granpos, mickeys, UIR@
; Modf:	AX, CX, DX, BX, SI, DI, wheel, wheelUIR, UIRunlock
; Call:	updateposition, updatebutton, refreshcursor
;
swapbuttons	proc
		testflag al,00000011b	; =3	; check the L and R buttons
	if_ odd					; if buttons not same
		db	034h			; XOR AL,byte
SWAPMASK	db	00000011b	; =3	; 0 if (PS2 xor LEFTHAND)
	end_
		;j	mouseupdate
swapbuttons	endp

mouseupdate	proc
		db	025h			; AND AX,word
buttonsmask	db	00000111b
wheelmask	db	0
		xchg	di,ax			; keep buttons new state in DI

;---------- recalculate mickey counters and screen position
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		MOVREG_	bx,<offset X>
		call	updateposition

		xchg	ax,cx
		MOVREG_	bl,<offset Y>		; OPTIMIZE: BL instead BX
		call	updateposition
		or	cl,al			; bit 0=mickeys change flag

;---------- recalculate wheel movement
		mov	ax,[mickeys.Y]
		xchg	ax,di
		xchg	dx,ax			; OPTIMIZE: instead MOV DX,AX
		mov	al,dh
		flipflag al,00001000b	; =8
		sub	al,00001000b	; =8	; sign extension AL[0:3]->AL
	if_ nz					; if wheel moved
		cbw
		mov	si,offset DGROUP:wheel
		add	[si].counter,ax		; wheel counter
		add	[wheelUIR],al		; same, but used for the UIR
		mov	al,10000000b	; =80h	; bit 7=wheel movement flag
		xor	bx,bx
		call	@lastpos
	end_

;---------- recalculate buttons state
		mov	dh,dl			; DH=buttons new state
		xchg	dl,[buttstatus]
		xor	dl,dh			; DL=buttons change state
IF FASTER_CODE
	if_ nz
ENDIF
		xor	bx,bx			; buttpress array index
		mov	al,00000010b		; indicate that 1 is pressed
		call	updatebutton
		mov	al,00001000b		; indicate that 2 is pressed
		call	updatebutton
		mov	al,00100000b		; indicate that 3 is pressed
		call	updatebutton
IF FASTER_CODE
	end_
ENDIF

;---------- call User Interrupt Routine (CX=events mask)
		dec	[UIRunlock]
	if_ zero				; if user proc not running
		and	cl,[callmask]
	 if_ nz					; if there is a user events
		OPCODE_MOV_BX
buttstatus	db	0,0			; buttons status
		xchg	bh,[wheelUIR]
		mov	ax,[granpos.X]
		mov	dx,[granpos.Y]
		xchg	ax,cx
		mov	si,[mickeys.X]
		;mov	di,[mickeys.Y]
		push	ds
		sti
		call	[UIR@]
		pop	ds
	 end_
		call	refreshcursor
	end_
;----------
		inc	[UIRunlock]
		ret
mouseupdate	endp

;
; In:	AX			(mouse movement)
;	BX			(offset X/offset Y)
; Out:	AX			(1 - mickey counter changed)
; Use:	mickey8, rangemax, rangemin, granumask, senscoeff
; Modf:	DX, SI, sensround, mickeys, rounderr, pos, granpos
;
updateposition	proc
		test	ax,ax
		jz	@@uposret
		mov	si,ax
	if_ sign
		neg	ax
	end_

;---------- apply sensitivity (SI=movement, AX=abs(SI))
		mov	dx,word ptr senscoeff[bx] ; ~[1/3..3.0]*256
		cmp	ax,4
	if_ be
		 mov	ax,si
		 cmp	dh,1		; skip [-4..4] movements
		 jae	@@newmickeys	;  when sensitivity >= 1.0
		 imul	dx		; =mickeys*sensitivity
	else_
		cmp	ax,12
	 if_ ae
IF FASTER_CODE
		 mov	ax,dx
		 shl	ax,1
		 add	ax,dx		; =sensitivity*3
	 else_
		mul	dx
		shr	ax,2		; =sensitivity*min(12,abs(mickeys))/4
	 end_
ELSE
		 mov	ax,12
	 end_
		mul	dx
		shr	ax,2		; =sensitivity*min(12,abs(mickeys))/4
ENDIF
		imul	si		; DX:AX=mickeys*newsensitivity
	end_
		add	al,byte ptr sensround[bx]
		mov	byte ptr sensround[bx],al
		mov	al,ah		; remove 256 multiplier from
		mov	ah,dl		;  sensitivity: AX=DX:AX/256
		adc	ax,0		; add carry from previous adding

;---------- apply mickeys per 8 pixels ratio to calculate cursor position
@@newmickeys:	add	word ptr mickeys[bx],ax
IF FASTER_CODE
		mov	si,word ptr mickey8[bx]
		cmp	si,8
	if_ ne
		shl	ax,3
		dec	si
	andif_ gt
		add	ax,word ptr rounderr[bx]
		inc	si
		cwd
		idiv	si
		mov	word ptr rounderr[bx],dx
		test	ax,ax
		jz	@@uposdone
	end_
ELSE
		shl	ax,3
		add	ax,word ptr rounderr[bx]
		cwd
		idiv	word ptr mickey8[bx]
		mov	word ptr rounderr[bx],dx
ENDIF
		add	ax,word ptr pos[bx]

;---------- cut new position by virtual ranges and save
@savecutpos:	mov	dx,word ptr rangemax[bx]
		cmp	ax,dx
		jge	@@cutpos
		mov	dx,word ptr rangemin[bx]
		cmp	ax,dx
	if_ le
@@cutpos:	xchg	ax,dx			; OPTIMIZE: instead MOV AX,DX
	end_
		mov	word ptr pos[bx],ax	; new position
		and	al,byte ptr granumask[bx]
		mov	word ptr granpos[bx],ax	; new granulated position
@@uposdone:	mov	ax,1
@@uposret:	ret
updateposition	endp

;
; In:	AL			(unrolled press bit mask)
;	CL			(unrolled buttons change state)
;	DL			(buttons change state)
;	DH			(buttons new state)
;	BX			(buttpress array index)
; Out:	CL
;	DX			(shifted state)
;	BX			(next index)
; Use:	granpos
; Modf:	AX, SI, buttpress, buttrelease
;
updatebutton	proc
		shr	dx,1
	if_ carry				; if button changed
		mov	si,offset DGROUP:buttpress
		test	dl,dl
	 if_ ns					; if button not pressed
		shl	al,1			; indicate that it released
		mov	si,offset DGROUP:buttrelease
	 end_
		inc	[si+bx].counter
@lastpos:	or	cl,al
		mov	ax,[granpos.Y]
		mov	[si+bx].lastrow,ax
		mov	ax,[granpos.X]
		mov	[si+bx].lastcol,ax
	end_
		add	bx,size BUTTLASTSTATE	; next button
		ret
updatebutton	endp

; END OF IRQ HANDLERS 


; INT 10 HANDLER 

		evendata
RILtable	dw	offset DGROUP:RIL_F0	; RIL functions
		dw	offset DGROUP:RIL_F1
		dw	offset DGROUP:RIL_F2
		dw	offset DGROUP:RIL_F3
		dw	offset DGROUP:RIL_F4
		dw	offset DGROUP:RIL_F5
		dw	offset DGROUP:RIL_F6
		dw	offset DGROUP:RIL_F7

int10handler	proc
		assume	ds:nothing,es:nothing
		cld
		test	ah,ah			; set video mode?
		jz	@@setmode
		cmp	ah,11h			; font manipulation function
		je	@@setnewfont
		cmp	ax,4F02h		; VESA set video mode?
		je	@@setmode
		cmp	ah,0F0h			; RIL func requested?
		jb	@@jmpold10
		cmp	ah,0F7h
		jbe	@@RIL
		cmp	ah,0FAh
		je	@@RIL_FA
@@jmpold10:	db	0EAh			; JMP FAR to old INT 10 handler
oldint10	dd	?

@@setnewfont:	cmp	al,10h
		jb	@@jmpold10
		cmp	al,20h
		jae	@@jmpold10
		;j	@@setmode

;========== set video mode or activate font
@@setmode:	push	ax
		mov	ax,2
		pushf				;!!! Logitech MouseWare
		push	cs			;  Windows driver workaround
		call	handler33		; hide mouse cursor
		pop	ax
		pushf
		call	cs:[oldint10]
		push	ds es
		PUSHALL
		MOVSEG	ds,cs,,DGROUP
		mov	[nocursorcnt],1		; normalize hide counter
		call	setupvideo
@@exitINT10:	jmp	@rethandler

;========== RIL
@@RIL:		push	ds es
		PUSHALL
		MOVSEG	ds,cs,,DGROUP
		mov	bp,sp
		mov	al,ah
		maskflag ax,0Fh			;!!! AH must be 0 for RIL_*
		mov	si,ax
		shl	si,1
		call	RILtable[si]
		j	@@exitINT10
;----------
@@RIL_FA:	MOVSEG	es,cs,,DGROUP		; RIL FA - Interrogate driver
		mov	bx,offset DGROUP:RILversion
		iret
int10handler	endp
		assume	ds:DGROUP

;
; RIL F0 - Read one register
;
;
; In:	DX			(group index)
;	BX			(register #)
; Out:	BL			(value)
; Use:	videoregs@
; Modf:	AL, SI
; Call:	none
;
RIL_F0		proc
		mov	si,dx
		mov	si,videoregs@[si].regs@
		cmp	dx,20h
	if_ below				; if not single register
		add	si,bx
	end_
		lodsb
		mov	byte ptr [_ARG_BX_],al
		ret
RIL_F0		endp

;
; RIL F1 - Write one register
;
;
; In:	DX			(group index)
;	BL			(value for single reg)
;	BL			(register # otherwise)
;	BH			(value otherwise)
; Out:	BL			(value)
; Use:	none
; Modf:	AX
; Call:	RILwrite
;
RIL_F1		proc
		mov	ah,bl
		cmp	dx,20h
		jae	RILwrite		; jump if single registers
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		mov	byte ptr [_ARG_BX_],ah
		;j	RILwrite
RIL_F1		endp

;
; In:	DX			(group index)
;	AL			(register # for regs group)
;	AH			(value to write)
; Out:	none
; Use:	videoregs@
; Modf:	AL, DX, BX, DI
; Call:	RILoutAH, RILgroupwrite
;
RILwrite	proc
		xor	bx,bx
		mov	di,dx
		cmp	dx,20h
		mov	dx,videoregs@[di].port@
		mov	videoregs@[di].rmodify?,dl ; OPTIMIZE: DL instead 1
		mov	di,videoregs@[di].regs@
	if_ below				; if not single register
		mov	bl,al
	end_
		mov	[di+bx],ah
		jae	RILoutAH
		;j	RILgroupwrite
RILwrite	endp

;
; In:	DX			(IO port)
;	AL			(register #)
;	AH			(value to write)
; Out:	none
; Use:	videoregs@
; Modf:	none
; Call:	none
;
RILgroupwrite	proc
		cmp	dl,0C0h
	if_ ne					; if not ATTR controller
		out	dx,ax
		ret
	end_
		push	ax dx
		mov	dx,videoregs@[(size RGROUPDEF)*5].port@
		in	al,dx			; {3DAh} force address mode
		pop	dx ax
		out	dx,al			; {3C0h} select ATC register
RILoutAH:	xchg	al,ah
		out	dx,al			; {3C0h} modify ATC register
		xchg	al,ah
		ret
RILgroupwrite	endp

;
; RIL F2 - Read register range
;
;
; In:	CH			(starting register #)
;	CL			(# of registers)
;	DX			(group index: 0,8,10h,18h)
;	ES:BX			(buffer, CL bytes size)
; Out:	none
; Use:	videoregs@
; Modf:	AX, CX, SI, DI
; Call:	none
;
RIL_F2		proc
		assume	es:nothing
		mov	di,bx
		mov	si,dx
		mov	si,videoregs@[si].regs@
		mov	al,ch
		;mov	ah,0
		add	si,ax
RILmemcopy:	sti
		mov	ch,0
		shr	cx,1
		rep	movsw
		adc	cx,cx
		rep	movsb
		ret
RIL_F2		endp

;
; RIL F3 - Write register range
;
;
; In:	CH			(starting register #)
;	CL			(# of registers, >0)
;	DX			(group index: 0,8,10h,18h)
;	ES:BX			(buffer, CL bytes size)
; Out:	none
; Use:	videoregs@
; Modf:	AX, CX, DX, BX, DI
; Call:	RILgroupwrite
;
RIL_F3		proc
		assume	es:nothing
		mov	di,dx
		mov	dx,videoregs@[di].port@
		mov	videoregs@[di].rmodify?,dl ; OPTIMIZE: DL instead 1
		mov	di,videoregs@[di].regs@
RILgrouploop:	xor	ax,ax
		xchg	al,ch
		add	di,ax
	countloop_
		mov	ah,es:[bx]
		mov	[di],ah
		inc	bx
		inc	di
		call	RILgroupwrite
		inc	ax			; OPTIMIZE: AX instead AL
	end_
		ret
RIL_F3		endp

;
; RIL F4 - Read register set
;
;
; In:	CX			(# of registers, >0)
;	ES:BX			(table of registers records)
; Out:	none
; Use:	videoregs@
; Modf:	AL, CX, BX, DI
; Call:	none
;
RIL_F4		proc
		assume	es:nothing
		sti
		mov	di,bx
	countloop_
		mov	bx,es:[di]
		movadd	di,,2
		mov	bx,videoregs@[bx].regs@
		mov	al,es:[di]
		inc	di
		xlat
		stosb
	end_
		ret
RIL_F4		endp

;
; RIL F5 - Write register set
;
;
; In:	CX			(# of registers, >0)
;	ES:BX			(table of registers records)
; Out:	none
; Use:	none
; Modf:	AX, CX, DX, SI
; Call:	RILwrite
;
RIL_F5		proc
		assume	es:nothing
		mov	si,bx
	countloop_
		lods	word ptr es:[si]
		xchg	dx,ax			; OPTIMIZE: instead MOV DX,AX
		lods	word ptr es:[si]
		call	RILwrite
	end_
		ret
RIL_F5		endp

;
; RIL F7 - Define registers default
;
;
; In:	DX			(group index)
;	ES:BX			(table of one-byte entries)
; Out:	none
; Use:	videoregs@
; Modf:	CL, SI, DI, ES, DS
; Call:	RILmemcopy
;
RIL_F7		proc
		assume	es:nothing
		mov	si,bx
		mov	di,dx
		mov	cl,videoregs@[di].regscnt
		mov	videoregs@[di].rmodify?,cl ; OPTIMIZE: CL instead 1
		mov	di,videoregs@[di].def@
		push	es ds
		pop	es ds
		j	RILmemcopy
RIL_F7		endp

;
; RIL F6 - Revert registers to default
;
;
; In:	none
; Out:	none
; Use:	videoregs@
; Modf:	AX, CX, DX, BX, SI, DI, ES
; Call:	RILgrouploop
;
RIL_F6		proc
		MOVSEG	es,ds,,DGROUP
		mov	si,offset DGROUP:videoregs@+(size RGROUPDEF)*8

@@R6loop:	sub	si,size RGROUPDEF
		xor	cx,cx
		xchg	cl,[si].rmodify?
		jcxz	@@R6next

		mov	bx,[si].def@
		mov	di,[si].regs@
		mov	dx,[si].port@
		mov	cl,[si].regscnt
		;mov	ch,0
		loop	@@R6group
		mov	al,[bx]
		stosb
		out	dx,al
		;j	@@R6next		; OPTIMIZE: single regs
		j	@@R6loop		;  handled first

@@R6group:	inc	cx			; OPTIMIZE: CX instead CL
		;mov	ch,0
		call	RILgrouploop

@@R6next:	cmp	si,offset DGROUP:videoregs@
		ja	@@R6loop
		ret
RIL_F6		endp

; END OF INT 10 HANDLER 


;
;			Draw mouse cursor
;

drawcursor	proc
		mov	cx,[videoseg]
		jcxz	@@drawret		; exit if nonstandard mode

		xor	cx,cx
		cmp	[nocursorcnt],cl	; OPTIMIZE: CL instead 0
		jnz	restorescreen		; jump if cursor disabled

		mov	ax,[granumask.Y]
		xchg	cl,[newcursor]		; remove redraw request
						; CX=force cursor request
		inc	ax
		mov	bx,[granpos.Y]		; cursor position Y
		mov	ax,[granpos.X]		; cursor position X
		jz	graphcursor		; jump if graphics mode

;========== text mode cursor
		mov	si,8			; OPTIMIZE: instead -[granumask.Y]
		call	checkifseen
		jc	restorescreen		; jump if not in seen area

		call	gettxtoffset
		cmp	di,[cursor@]
	if_ eq					; exit if position not changed
		jcxz	@@drawret		;  and cursor not forced
	end_
		push	di
		call	restorescreen
		;MOVSEG	es,[videoseg],,nothing
		pop	di

		cmp	[cursortype],ch		; OPTIMIZE: CH instead 0
	if_ nz
;---------- position hardware text mode cursor
		shr	di,1
		mov	dx,videoregs@[0].port@	; CRTC port
		mov	ax,di
		out_	dx,0Fh,al		; cursor position lo
		xchg	ax,di			; OPTIMIZE: instead MOV AX,DI
		out_	dx,0Eh,ah		; cursor position hi
		ret
	end_

;---------- draw software text mode cursor
		mov	[cursor@],di
		mov	ax,es:[di]		; save char under cursor
		mov	textbuf[2],ax
		and	ax,[startscan]
		xor	ax,[endscan]
		stosw				; draw to new position
		mov	textbuf[0],ax
@@drawret:	ret
drawcursor	endp

;
;			Restore old screen contents
;

restorescreen	proc
		les	di,dword ptr [cursor@]
		assume	es:nothing
		inc	di
	if_ nz					; if cursor drawn
		sub	[cursor@],di		; OPTIMIZE: instead MOV -1
		mov	ax,[granumask.Y]
		dec	di
		inc	ax

	 if_ zero
;---------- graphics mode
		call	restoresprite
		jmp	restorevregs
	 end_

;---------- text mode
		mov	si,offset DGROUP:textbuf
		lodsw
		cmp	ax,es:[di]
	 if_ eq					; if screen not changed
		movsw				; restore old text char/attrib
	 end_
	end_
@drawret:	ret
restorescreen	endp

;
;		Draw graphics mode mouse cursor
;

graphcursor	proc
		sub	ax,[hotspot.X]		; virtual X
		sub	bx,[hotspot.Y]		; virtual Y
		mov	si,16			; cursor height
		push	ax
		call	checkifseen
		pop	ax
		jc	restorescreen		; jump if not in seen area

		xchg	ax,bx
		xor	dx,dx
		neg	ax
	if_ lt
		neg	ax
		xchg	ax,dx
	end_
		mov	[spritetop],ax
		mov	ax,[screenheight]
		cmp	si,ax
	if_ ge
		xchg	si,ax			; OPTIMIZE: instead MOV SI,AX
	end_
		sub	si,dx			; =spriteheight
		push	si			;  =min(16-ax,screenheight-dx)
		call	getgroffset
		pop	dx

; cx=force request, bx=X, di=line offset, si=nextrow, dx=spriteheight

		add	di,bx
		les	ax,dword ptr [cursor@]
		assume	es:nothing
		inc	ax
	if_ nz					; if cursor drawn
		db	081h,0FFh		; CMP DI,word
cursorpos	dw	?
	 if_ eq
		cmp	dx,[spriteheight]
	 andif_ eq				; exit if position not changed
		jcxz	@drawret		;  and cursor not forced
	 end_
		push	bx dx di
		dec	ax
		xchg	di,ax			; OPTIMIZE: instead MOV DI,AX
		call	restoresprite
		;MOVSEG	es,[videoseg],,nothing
		pop	di dx
	else_
		push	bx
		call	updatevregs
	end_
		pop	bx

; bx=X, di=line offset+bx, si=nextrow, dx=spriteheight, es=videoseg

		mov	[cursorpos],di
		mov	[spriteheight],dx
		mov	[nextrow],si
		sub	di,bx
		push	dx

;---------- precompute sprite parameters
		push	bx
		mov	cl,[bitmapshift]
		mov	dx,[cursorwidth]
		sar	bx,cl			; left sprite offset (signed)
		mov	ax,[scanline]
		add	dx,bx			; right sprite offset
		cmp	dx,ax
	if_ ae
		xchg	dx,ax			; DX=min(DX,scanline)
	end_

		pop	ax			; =cursorX
		sub	cl,3			; mode 0Dh=1, other=0
		and	ax,[granumask.X]	; fix for mode 4/5
		sar	ax,cl			; sprite shift for non 13h modes
		neg	bx			; sprite shift for 13h mode
	if_ lt					; if left sprite offset>0
		add	dx,bx
		sub	di,bx
		mov	bl,0
		and	al,7			; shift in byte (X%8)
	end_

		inc	bx			; OPTIMIZE: BX instead BL
		sub	al,8			; if cursorX>0
		mov	bh,al			; ...then BH=-(8-X%8)
		push	bx			; ...else BH=-(8-X)=-(8+|X|)

;---------- save screen sprite and draw cursor at new cursor position
		mov	[spritewidth],dx
		mov	[cursor@],di
		mov	al,0D6h			; screen source
		call	copysprite		; save new sprite

		OPCODE_MOV_BX
spritetop	dw	?
		pop	cx ax			; CL/CH=sprite shift
						; AX=[spriteheight]
						; SI=[nextrow]
		shl	bx,1			; mask offset
	countloop_ ,ax
		push	ax cx bx si di
		mov	si,[spritewidth]
		mov	dx,word ptr screenmask[bx]
		mov	bx,word ptr cursormask[bx]
		call	makerow
		pop	di si bx cx ax
		add	di,si
		xor	si,[nextxor]
		movadd	bx,,2
	end_
;----------
		;j	restorevregs
graphcursor	endp

;
;		Restore graphics card video registers
;

restorevregs	proc
		mov	bx,offset DGROUP:vdata1
		j	@writevregs
restorevregs	endp

;
;		Save & update graphics card video registers
;

updatevregs	proc
		mov	bx,offset DGROUP:vdata1
		mov	ah,0F4h			; read register set
		call	@registerset

		mov	bx,offset DGROUP:vdata2
@writevregs:	mov	ah,0F5h			; write register set

@registerset:	; if planar videomode [0Dh-12h] then "push es" else "ret"
		db	?
		MOVSEG	es,ds,,DGROUP
		mov	cx,[bx-2]
		int	10h
		pop	es
		ret
updatevregs	endp

;

restoresprite	proc
		call	updatevregs
		mov	al,0D7h			; screen destination
		;j	copysprite		; restore old sprite
restoresprite	endp

;
;		Copy screen sprite back and forth
;
;
; In:	AL			(0D6h/0D7h-screen source/dest.)
;	ES:DI			(pointer to video memory)
; Out:	CX = 0
;	BX = 0
; Use:	buffer@
; Modf:	AX, DX
; Call:	none
;
copysprite	proc	C uses si di ds es
		assume	es:nothing
		cmp	al,0D6h
		mov	NEXTOFFSCODE[1],al
		OPCODE_MOV_AX
nextrow		dw	?			; next row offset
		OPCODE_MOV_BX
spriteheight	dw	?			; sprite height in lines
		lds	si,[buffer@]
		assume	ds:nothing
	if_ eq
		push	ds es
		pop	ds es			; DS:SI=screen
		xchg	si,di			; ES:DI=buffer
	end_

	countloop_ ,bx
		OPCODE_MOV_CX
spritewidth	dw	?			; seen part of sprite in bytes
		movsub	dx,ax,cx
		rep	movsb
NEXTOFFSCODE	db	01h,?			; ADD SI,DX/ADD DI,DX
		db	035h			; XOR AX,word
nextxor		dw	?
	end_
		ret
copysprite	endp
		assume	ds:DGROUP

;
;		Transform the cursor mask row to screen
;
;
; In:	DX = screenmask[row]
;	BX = cursormask[row]
;	SI = [spritewidth]
;	CL			(sprite shift when mode 13h)
;	CH			(sprite shift when non 13h modes)
;	ES:DI			(video memory pointer)
; Out:	none
; Use:	bitmapshift
; Modf:	AX, CX, DX, BX, SI, DI
; Call:	none
;
makerow		proc
		assume	es:nothing
		cmp	[bitmapshift],1		; =1 for 13h mode
	if_ eq
;----------
	 countloop_ ,si
		shl	bx,cl			; if MSB=0
		sbb	al,al			; ...then AL=0
		and	al,0Fh			; ...else AL=0Fh (WHITE color)
		shl	dx,cl
	  if_ carry				; if most sign bit nonzero
		xor	al,es:[di]
	  end_
		stosb
		mov	cl,1
	 end_ countloop
		ret
	end_ if

;---------- display cursor row in modes other than 13h
makerowno13:	mov	ax,0FFh
	loop_
		stc
		rcl	dx,1
		rcl	al,1			; al:dh:dl shifted screenmask
		shl	bx,1
		rcl	ah,1			; ah:bh:bl shifted cursormask
		inc	ch
	until_ zero
		xchg	dh,bl			; al:bl:dl - ah:bh:dh

	countloop_ ,si
		push	dx
		mov	dx,es
		cmp	dh,0A0h
	 if_ ne					; if not planar mode 0Dh-12h
		and	al,es:[di]
		xor	al,ah
		stosb
	 else_
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		out_	3CEh,5,0		; set write mode
		out_	,3,08h			; data ANDed with latched data
		xchg	es:[di],cl
		out_	,,18h			; data XORed with latched data
		xchg	es:[di],ch
		inc	di
	 end_
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		pop	bx
	end_ countloop
		ret
makerow		endp

;
;	Return graphic mode video memory offset to line start
;
;
; In:	DX			(Y coordinate in pixels)
; Out:	DI			(video memory offset)
;	SI			(offset to next row)
; Use:	videoseg, scanline
; Modf:	AX, DX, ES
; Call:	@getoffsret
; Info:	4/5 (320x200x4)   byte offset = (y/2)*80 + (y%2)*2000h + (x*2)/8
;	  6 (640x200x2)   byte offset = (y/2)*80 + (y%2)*2000h + x/8
;	0Dh (320x200x16)  byte offset = y*40 + x/8, bit offset = 7 - (x % 8)
;	0Eh (640x200x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;	0Fh (640x350x4)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;	10h (640x350x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;	11h (640x480x2)   byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;	12h (640x480x16)  byte offset = y*80 + x/8, bit offset = 7 - (x % 8)
;	13h (320x200x256) byte offset = y*320 + x
;	HGC (720x348x2)   byte offset = (y%4)*2000h + (y/4)*90 + x/8
;						    bit offset = 7 - (x % 8)
;
getgroffset	proc
		xor	di,di
		mov	ax,[scanline]
		MOVSEG	es,di,,BIOS
		mov	si,ax			; [nextrow]
		cmp	byte ptr videoseg[1],0A0h
		je	@getoffsret		; jump if not videomode 4-6
		mov	si,2000h
		sar	dx,1			; DX=Y/2
		jnc	@getoffsret
		mov	di,si			; DI=(Y%2)*2000h
		mov	si,-(2000h-80)
		j	@getoffsret
getgroffset	endp

;
;		Return text mode video memory offset
;
;
; In:	AX/BX			(cursor position X/Y)
; Out:	DI			(video memory offset=row*[0:44Ah]*2+column*2)
; Use:	0:44Ah, 0:44Eh, bitmapshift
; Modf:	AX, DX, BX, ES
; Call:	getpageoffset
;
gettxtoffset	proc
		MOVSEG	es,0,dx,BIOS
		xchg	di,ax			; OPTIMIZE: instead MOV DI,AX
		mov	al,[bitmapshift]
		dec	ax			; OPTIMIZE: AX instead AL
		xchg	cx,ax
		sar	di,cl			; DI=column*2
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		sar	ax,2			; AX=row*2=Y/4
		mov	dx,[VIDEO_width]	; screen width

@getoffsret:	imul	dx			; AX=row*screen width
		add	ax,[VIDEO_pageoff]	; add video page offset
		add	di,ax
		ret
gettxtoffset	endp

;
;		Check if cursor seen and not in update region
;
;
; In:	AX/BX			(cursor position X/Y)
;	SI			(cursor height)
; Out:	Carry flag		(cursor not seen or in update region)
;	BX			(cursor X aligned at byte in video memory)
;	SI			(cursor Y+height)
; Use:	scanline, screenheight, upleft, lowright
; Modf:	DX
; Call:	none
;
checkifseen	proc	C uses cx
;---------- check if cursor shape seen on the screen
		add	si,bx
		jle	@@retunseen		; fail if Y+height<=0
		cmp	bx,[screenheight]
		jge	@@retunseen		; fail if Y>maxY

		OPCODE_MOV_CL
bitmapshift	db	?			; mode 13h=1, 0Dh=4, other=3
		OPCODE_MOV_DX
cursorwidth	dw	?			; cursor width in bytes
		sar	ax,cl
		add	dx,ax
		jle	@@retunseen		; fail if X+width<=0
		cmp	ax,[scanline]
		jge	@@retunseen		; fail if X>maxX

;---------- check if cursor shape not intersects with update region
		shl	ax,cl
		cmp	bx,[lowright.Y]
		jg	@@retseen		; ok if Y below
		cmp	si,[upleft.Y]
		jle	@@retseen		; ok if Y+height above

		cmp	ax,[lowright.X]
		jg	@@retseen		; ok if X from the right
		shl	dx,cl
		cmp	dx,[upleft.X]
		jle	@@retseen		; ok if X+width from the left

@@retunseen:	stc
		ret
@@retseen:	clc
		ret
checkifseen	endp


; INT 33 HANDLER SERVICES 

;

setupvideo	proc
		mov	si,LenClearArea2/2	; clear area 2
ERRIF (LenClearArea2 mod 2 ne 0) "LenClearArea2 must be even!"
		j	@setvideo
setupvideo	endp

;
; 21 - Software reset
;
;
; In:	none
; Out:	[AX] = 21h/FFFFh	(not installed/installed)
;	[BX] = 2/3/FFFFh	(number of buttons)
; Use:	0:449h, 0:44Ah, 0:463h, 0:484h, 0:487h, 0:488h, 0:4A8h
; Modf:	StartSaveArea, screenheight, granumask, buffer@, videoseg, cursorwidth,
;	scanline, nextxor, bitmapshift, @registerset, rangemax, StartClearArea
; Call:	hidecursor, @savecutpos
;
softreset_21	proc
		mov	[_ARG_BX_],3
buttonscnt	equ	byte ptr [$-2]		; buttons count (2/3)
		mov	[_ARG_AX_],0FFFFh
						; restore default area
		memcopy	LenDefArea,ds,DGROUP,DGROUP:StartSaveArea,ds,,DGROUP:StartDefArea
		call	hidecursor		; restore screen contents
		mov	si,LenClearArea3/2	; clear area 3
ERRIF (LenClearArea3 mod 2 ne 0) "LenClearArea3 must be even!"

;---------- setup video regs values for current video mode
@setvideo:	push	si
		MOVSEG	es,ds,,DGROUP
		MOVSEG	ds,0,ax,BIOS
		mov	ax,[CRTC_base]		; base IO address of CRTC
		mov	videoregs@[0].port@,ax	; 3D4h/3B4h
		add	ax,6			; Feature Control register
		mov	videoregs@[(size RGROUPDEF)*5].port@,ax
		mov	al,[VIDEO_mode]		; current video mode
		push	ax
;----------
	block_
		mov	ah,9
		cmp	al,11h			; VGA videomodes?
	 breakif_ ae

		cbw				; OPTIMIZE: instead MOV AH,0
		cmp	al,0Fh			; 0F-10 videomodes?
	 if_ ae
		testflag [VIDEO_control],mask VCTRL_RAM_64K
	  breakif_ zero				; break if only 64K of VRAM
		mov	ah,2
	 else_
		cmp	al,4			; not color text modes?
	 andif_ below
		mov	cl,[VIDEO_switches]	; get display combination
		maskflag cl,mask VIDSW_feature0+mask VIDSW_display
		cmp	cl,9			; EGA+ECD/MDA?
		je	@@lines350
		cmp	cl,3			; MDA/EGA+ECD?
	  if_ eq
@@lines350:	mov	ah,13h
	  end_
	 end_ if
	end_ block
;----------
		lds	si,[VIDEO_ptrtable@]
		assume	ds:nothing
		lds	si,[si].VIDEO_paramtbl@
		add	ah,al
		mov	al,(offset VPARAM_SEQC) shl 2
		shr	ax,2
		add	si,ax			; SI += (AL+AH)*64+5

		mov	di,offset DGROUP:StartDefVRegsArea
		push	di
		mov	al,3
		stosb				; def_SEQ[0]=3
		memcopy	50			; copy default registers value
		mov	al,0
		stosb				; def_ATC[20]=0; VGA only
		memcopy	9			; def_GRC
		;mov	ah,0
		stosw				; def_FC=0, def_GPOS1=0
		inc	ax			; OPTIMIZE: instead MOV AL,1
		stosb				; def_GPOS2=1

		pop	si			; copy current registers value
		memcopy	LenVRegsArea,,,DGROUP:StartVRegsArea,es,DGROUP

		dec	ax			; OPTIMIZE: instead MOV AL,0
		mov	cx,8
		mov	di,offset DGROUP:videoregs@[0].rmodify?
	countloop_
		stosb
		add	di,(size RGROUPDEF)-1
	end_

;---------- set parameters for current video mode
; mode	 seg   screen  cell scan planar  VX/
;			    line	byte
;  0	B800h  640x200 16x8   -    -	  -
;  1	B800h  640x200 16x8   -    -	  -
;  2	B800h  640x200	8x8   -    -	  -
;  3	B800h  640x200	8x8   -    -	  -
;  4	B800h  320x200	2x1   80   no	  8
;  5	B800h  320x200	2x1   80   no	  8
;  6	B800h  640x200	1x1   80   no	  8
;  7	B000h  640x200	8x8   -    -	  -
; 0Dh	A000h  320x200	2x1   40  yes	 16
; 0Eh	A000h  640x200	1x1   80  yes	  8
; 0Fh	A000h  640x350	1x1   80  yes	  8
; 10h	A000h  640x350	1x1   80  yes	  8
; 11h	A000h  640x480	1x1   80  yes	  8
; 12h	A000h  640x480	1x1   80  yes	  8
; 13h	A000h  320x200	2x1  320   no	  2
; other     0  640x200	1x1   -    -	  -
;
		pop	ax			; current video mode
; mode 0-3
		mov	dx,0B8FFh		; B800h: [0-3]
		mov	cx,0304h		; 16x8: [0-1]
		mov	di,200			; x200: [4-6,0Dh-0Eh,13h]
		cmp	al,2
	if_ ae
		dec	cx			; 8x8: [2-3,7]
		cmp	al,4
	andif_ ae
; mode 7
		cmp	al,7
		jne	@@checkgraph
		mov	dh,0B0h			; B000h: [7]
	end_

@@settext:	mov	ch,1
		mov	bh,0F8h
		shl	dl,cl

		MOVSEG	es,0,ax,BIOS
		add	al,[VIDEO_lastrow]	; screen height-1
	if_ nz					; zero on old machines
		inc	ax			; OPTIMIZE: AX instead AL
IF USE_286
		shl	ax,3
ELSE
		mov	ah,8
		mul	ah
ENDIF
		xchg	di,ax			; OPTIMIZE: instead MOV DI,AX
	end_
		mov	ax,[VIDEO_width]	; screen width
		j	@@setcommon

; mode 4-6
@@checkgraph:	mov	ah,0C3h			; RET opcode for [4-6,13h]
		;mov	cx,0303h		; sprite: 3 bytes/row
		;mov	dx,0B8FFh		; B800h: [4-6]/1x1: [6,0Eh-12h]
		;mov	di,200			; x200: [4-6,0Dh-0Eh,13h]
		mov	si,2000h xor -(2000h-80) ; [nextxor] for [4-6]
		;MOVSEG	es,ds,,DGROUP
		OPCODE_MOV_BX
BUFFADDR	dw	offset DGROUP:bufferPS2
		cmp	al,6
		je	@@setgraphics
		jb	@@set2x1

; in modes 0Dh-13h screen contents under cursor sprite will be
; saved at free space in video memory (A000h segment)

		mov	dh,0A0h			; A000h: [0Dh-13h]
		MOVSEG	es,0A000h,bx,nothing
		xor	si,si			; [nextxor] for [0Dh-13h]
		cmp	al,13h
		ja	@@nonstandard
		je	@@mode13
; mode 8-0Dh
		cmp	al,0Dh
		jb	@@nonstandard
		mov	ah,06h			; PUSH ES opcode for [0Dh-12h]
		mov	bx,3E82h		; 16002: [0Dh-0Eh]
		je	@@set320
; mode 0Eh-12h
		cmp	al,0Fh
		jb	@@setgraphics
		mov	di,350			; x350: [0Fh-10h]
		mov	bh,7Eh			; 32386: [0Fh-10h]
		cmp	al,11h
		jb	@@setgraphics
		mov	di,480			; x480: [11h-12h]
		mov	bh,9Eh			; 40578: [11h-12h]
		j	@@setgraphics
; mode 13h
@@mode13:	;mov	bl,0
		mov	bh,0FAh			; =320*200
		mov	cx,1000h		; sprite: 16 bytes/row

@@set320:	inc	cx			; OPTIMIZE: instead INC CL
@@set2x1:	dec	dx			; OPTIMIZE: instead MOV DL,-2

@@setgraphics:	saveFAR	[buffer@],es,bx
		mov	[nextxor],si
		mov	byte ptr [@registerset],ah
		j	@@setgcommon

@@nonstandard:	;mov	cl,3
		;mov	dl,0FFh
		;mov	di,200

;;+++++ for text modes: dh := 0B8h, j @@settext

		mov	dh,0			; no video segment

@@setgcommon:	mov	ax,640			; virtual screen width
		mov	bh,0FFh			; Y granularity
		shr	ax,cl

@@setcommon:	mov	[screenheight],di
		mov	[scanline],ax		; screen line width in bytes
		mov	[bitmapshift],cl	; log2(screen/memory ratio)
						;  (mode 13h=1, 0-1/0Dh=4, other=3)
		mov	byte ptr [cursorwidth],ch ; cursor width in bytes
		mov	byte ptr [granumask.X],dl
		mov	byte ptr [granumask.Y],bh
		mov	byte ptr videoseg[1],dh
		shl	ax,cl
		pop	si

;---------- set ranges and center cursor (AX=screenwidth, DI=screenheight)
		mov	cx,ax
		dec	ax
		mov	[rangemax.X],ax		; set right X range
		shr	cx,1			; X middle

		mov	dx,di
		dec	di
		mov	[rangemax.Y],di		; set lower Y range
		shr	dx,1			; Y middle

;---------- set cursor position (CX=X, DX=Y, SI=area size to clear)
@setpos:	;cli
		MOVSEG	es,ds,,DGROUP
		mov	di,offset DGROUP:StartClearArea
		xchg	cx,si
		xor	ax,ax
		rep	stosw

		xchg	ax,dx			; OPTIMIZE: instead MOV AX,DX
		MOVREG_	bx,<offset Y>
		call	@savecutpos
		xchg	ax,si			; OPTIMIZE: instead MOV AX,SI
		MOVREG_	bl,<offset X>		; OPTIMIZE: BL instead BX
		jmp	@savecutpos
softreset_21	endp

;
; 1F - Disable mouse driver
;
;
; In:	none
; Out:	[AX] = 1Fh/FFFFh	(success/unsuccess)
;	[ES:BX]			(old int33 handler)
; Use:	oldint33, oldint10
; Modf:	AX, CX, DX, BX, DS, ES, disabled?, nocursorcnt
; Call:	INT 21/35, INT 21/25, disablePS2/disableUART, hidecursor
;
disabledrv_1F	proc
		les	ax,[oldint33]
		assume	es:nothing
		mov	[_ARG_ES_],es
		mov	[_ARG_BX_],ax

		db	0E8h			; CALL NEAR to disableproc
disableproc	dw	disablePS2-$-2

		mov	al,[disabled?]
		test	al,al
	if_ zero				; if driver not disabled
		mov	[buttstatus],al
		inc	ax			; OPTIMIZE: instead MOV AL,1
		mov	[nocursorcnt],al	; normalize hide counter
		call	hidecursor		; restore screen contents

;---------- check if INT 33 or INT 10 were intercepted
;	    (i.e. handlers segment not equal to CS)
		mov	cx,cs
		DOSGetIntr 33h
		mov	dx,es
		cmp	dx,cx
		jne	althandler_18

		;mov	ah,35h
		mov	al,10h
		int	21h
		movsub	ax,es,cx
		jne	althandler_18

		inc	ax			; OPTIMIZE: instead MOV AL,1
		mov	[disabled?],al
		lds	dx,[oldint10]
		assume	ds:nothing
		DOSSetIntr 10h			; restore old INT 10 handler
	end_ if
		ret
disabledrv_1F	endp
		assume	ds:DGROUP

;
; 18 - Set alternate User Interrupt Routine
;
;
; In:	CX			(call mask)
;	ES:DX			(FAR routine)
; Out:	[AX] = 18h/FFFFh	(success/unsuccess)
;
althandler_18	proc
		assume	es:nothing
		mov	[_ARG_AX_],0FFFFh
		ret
althandler_18	endp

;
; 19 - Get alternate User Interrupt Routine
;
;
; In:	CX			(call mask)
; Out:	[CX]			(0=not found)
;	[BX:DX]			(FAR routine)
;
althandler_19	proc
		mov	[_ARG_CX_],0
		ret
althandler_19	endp

;
; 00 - Reset driver and read status
;
;
; In:	none
; Out:	[AX] = 0/FFFFh		(not installed/installed)
;	[BX] = 2/3/FFFFh	(number of buttons)
; Use:	none
; Modf:	none
; Call:	softreset_21, enabledriver_20
;
resetdriver_00	proc
		call	softreset_21
		;j	enabledriver_20
resetdriver_00	endp

;
; 20 - Enable mouse driver
;
;
; In:	none
; Out:	[AX] = 20h/FFFFh	(success/unsuccess)
; Use:	none
; Modf:	AX, CX, DX, BX, ES, disabled?, oldint10
; Call:	INT 21/35, INT 21/25, setupvideo, enablePS2/enableUART
;
enabledriver_20	proc
		xor	cx,cx
		xchg	cl,[disabled?]
	if_ ncxz
;---------- set new INT 10 handler
		DOSGetIntr 10h
		saveFAR	[oldint10],es,bx
		;mov	al,10h
		DOSSetIntr ,,,DGROUP:int10handler
	end_
;----------
		call	setupvideo
		db	0E9h			; JMP NEAR to enableproc
enableproc	dw	enablePS2-$-2
enabledriver_20	endp

;
; 03 - Get cursor position, buttons status and wheel counter
;
;
; In:	none
; Out:	[BL]			(buttons status)
;	[BH]			(wheel movement counter)
;	[CX]			(X - column)
;	[DX]			(Y - row)
; Use:	buttstatus, granpos
; Modf:	AX, CX, DX, wheel.counter
; Call:	@retBCDX
;
status_03	proc
		xor	ax,ax
		xchg	ax,[wheel.counter]
		mov	ah,al
		mov	al,[buttstatus]
		mov	cx,[granpos.X]
		mov	dx,[granpos.Y]
		j	@retBCDX
status_03	endp

;
; 05 - Get button press data
;
;
; In:	BX			(button number; -1 for wheel)
; Out:	[AL]			(buttons status)
;	[AH]			(wheel movement counter)
;	[BX]			(press times or wheel counter, if BX was -1)
;	[CX]			(last press X)
;	[DX]			(last press Y)
; Use:	none
; Modf:	CX, buttpress, wheel
; Call:	@retbuttstat
;
pressdata_05	proc
		mov	cx,offset DGROUP:buttpress-(size BUTTLASTSTATE)
		j	@retbuttstat
pressdata_05	endp

;
; 06 - Get button release data
;
;
; In:	BX			(button number; -1 for wheel)
; Out:	[AL]			(buttons status)
;	[AH]			(wheel movement counter)
;	[BX]			(release times or wheel counter, if BX was -1)
;	[CX]			(last release X)
;	[DX]			(last release Y)
; Use:	buttstatus
; Modf:	AX, CX, DX, BX, SI, buttrelease, wheel
; Call:	none
;
releasedata_06	proc
		mov	cx,offset DGROUP:buttrelease-(size BUTTLASTSTATE)
@retbuttstat:	mov	ah,byte ptr [wheel.counter]
		mov	al,[buttstatus]
		mov	[_ARG_AX_],ax
		mov	si,offset DGROUP:wheel
		inc	bx
		jz	@@retlastpos		; jump if BX was -1
IF FOOLPROOF
		cmp	bx,2+1
	if_ be
ENDIF
ERRIF (6 ne size BUTTLASTSTATE) "BUTTLASTSTATE structure size changed!"
		mov	si,cx
		shl	bx,1
		add	si,bx			; SI+BX=buttrelease
		shl	bx,1			;  +button*size BUTTLASTSTATE

@@retlastpos:	xor	ax,ax
		xchg	[si+bx.counter],ax
		mov	cx,[si+bx.lastcol]
		mov	dx,[si+bx.lastrow]
@retBCDX:	mov	[_ARG_DX_],dx
@retBCX:	mov	[_ARG_CX_],cx
@retBX:		mov	[_ARG_BX_],ax
IF FOOLPROOF
	end_ if
ENDIF
		ret
releasedata_06	endp

;
; 0B - Get motion counters
;
;
; In:	none
; Out:	[CX]			(number of mickeys mouse moved
;	[DX]			 horizontally/vertically since last call)
; Use:	none
; Modf:	mickeys
; Call:	@retBCDX
;
mickeys_0B	proc
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		xor	cx,cx
		xor	dx,dx
		xchg	[mickeys.X],cx
		xchg	[mickeys.Y],dx
		j	@retBCDX
mickeys_0B	endp

;
; 11 - Check wheel support and get capabilities flags
;
;
; In:	none
; Out:	[AX] = 574Dh ('WM')
;	[CX]			(capabilities flag, bits 1..15 reserved)
;	[BX]			(capabilities flag, all bits reserved)
; Use:	none
; Modf:	AX
; Call:	@retBCX
;
wheelAPI_11	proc
		mov	[_ARG_AX_],574Dh
		db	0B9h			; MOV CX,word
wheelflags	db	0,0
		xor	ax,ax
		j	@retBCX
wheelAPI_11	endp

;
; 15 - Get driver storage requirements
;
;
; In:	none
; Out:	[BX]			(buffer size)
; Use:	LenSaveArea
; Modf:	AX
; Call:	@retBX
;
storagereq_15	proc
		mov	ax,LenSaveArea
		j	@retBX
storagereq_15	endp

;
; 1B - Get mouse sensitivity
;
;
; In:	none
; Out:	[BX]			(horizontal sensitivity, 1..100)
;	[CX]			(vertical sensitivity, 1..100)
;	[DX]			(speed threshold in mickeys/second)
; Use:	sensval, /doublespeed/
; Modf:	AX, CX, DX
; Call:	@retBCDX
;
sensitivity_1B	proc
		mov	ax,[sensval.X]
		mov	cx,[sensval.Y]
		inc	ax
		inc	cx
		xor	dx,dx
;;*		mov	dx,[doublespeed]
		j	@retBCDX
sensitivity_1B	endp

;
; 1E - Get display page
;
;
; In:	none
; Out:	[BX]			(display page number)
; Use:	0:462h
; Modf:	AX, DS
; Call:	@retBX
;
videopage_1E	proc
		MOVSEG	ds,0,ax,BIOS
		mov	al,[VIDEO_pageno]
		j	@retBX
videopage_1E	endp
		assume	ds:DGROUP

;
; 01 - Show mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	AX, lowright.Y
; Call:	cursorstatus
;
showcursor_01	proc
		neg	ax			; AL=AH=-1
		mov	byte ptr lowright.Y[1],al ; place update region
		j	cursorstatus		;  outside seen screen area
showcursor_01	endp

;
; 02 - Hide mouse cursor
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	AX
; Call:	cursorstatus
;
hidecursor_02	proc
		dec	ax			; AL=1,AH=0
		;j	cursorstatus
hidecursor_02	endp

;
; Hint:	request to cursor redraw (instead refresh) is useful in cases when
;	interrupt handlers try to hide, then show cursor while cursor
;	drawing is in progress
;
cursorstatus	proc
		add	al,[nocursorcnt]
		sub	ah,al			; exit if "already enabled"
		jz	@showret		;   or "counter overflow"
		mov	[nocursorcnt],al
		inc	ah			; jump if cursor changed
		jz	redrawcursor		;  between enabled/disabled
		ret
cursorstatus	endp

;
; 07 - Set horizontal cursor range
;
;
; In:	CX			(min X)
;	DX			(max X)
; Out:	none
; Use:	none
; Modf:	BX
; Call:	@setnewrange
;
hrange_07	proc
		MOVREG_	bx,<offset X>
		j	@setnewrange
hrange_07	endp

;
; 08 - Set vertical cursor range
;
;
; In:	CX			(min Y)
;	DX			(max Y)
; Out:	none
; Use:	pos
; Modf:	CX, DX, BX, rangemin, rangemax
; Call:	setpos_04
;
vrange_08	proc
		MOVREG_	bx,<offset Y>
IF FOOLPROOF
@setnewrange:	xchg	ax,cx			; OPTIMIZE: instead MOV AX,CX
		cmp	ax,dx
	if_ ge
		xchg	ax,dx
	end_
		mov	word ptr rangemin[bx],ax
ELSE
@setnewrange:	mov	word ptr rangemin[bx],cx
ENDIF
		mov	word ptr rangemax[bx],dx
		mov	cx,[pos.X]
		mov	dx,[pos.Y]
		;j	setpos_04
vrange_08	endp

;
; 04 - Position mouse cursor
;
;
; In:	CX			(X - column)
;	DX			(Y - row)
; Out:	none
; Use:	none
; Modf:	SI
; Call:	@setpos, refreshcursor
;
setpos_04	proc
		mov	si,LenClearArea1/2	; clear area 1
ERRIF (LenClearArea1 mod 2 ne 0) "LenClearArea1 must be even!"
		call	@setpos
		;j	refreshcursor
setpos_04	endp

;

refreshcursor	proc
		sub	[videolock],1
		jc	@showret		; was 0: drawing in progress
		js	@@refreshdone		; was -1: queue already used
		sti

	loop_
		call	drawcursor
@@refreshdone:	inc	[videolock]		; drawing stopped
	until_ nz				; loop until queue empty

		cli
@showret:	ret
refreshcursor	endp

;
; 09 - Define graphics cursor
;
;
; In:	BX			(hot spot X)
;	CX			(hot spot Y)
;	ES:DX			(pointer to bitmaps)
; Out:	none
; Use:	none
; Modf:	AX, CX, BX, SI, DI, ES, hotspot, screenmask, cursormask
; Call:	@showret, redrawcursor
;
graphcursor_09	proc
		assume	es:nothing
;---------- compare user shape with internal area
		mov	si,offset DGROUP:hotspot
		lodsw
		cmp	ax,bx
	if_ eq
		lodsw
		xor	ax,cx
	andif_ eq
		mov	di,dx
		;mov	ah,0
		mov	al,16+16
		xchg	ax,cx
		repe	cmpsw
		je	@showret		; exit if cursor not changed
		xchg	cx,ax			; OPTIMIZE: instead MOV CX,AX
	end_

;---------- copy user shape to internal area
		push	ds ds es
		pop	ds es
		mov	di,offset DGROUP:hotspot
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		stosw
		xchg	ax,cx			; OPTIMIZE: instead MOV AX,CX
		stosw
		memcopy	2*(16+16),,,,,,dx
		pop	ds
		;j	redrawcursor
graphcursor_09	endp

;

redrawcursor	proc
hidecursor:	mov	[newcursor],1		; force cursor redraw
		j	refreshcursor
redrawcursor	endp

;
; 0A - Define text cursor
;
;
; In:	BX			(0 - SW, else HW text cursor)
;	CX			(screen mask/start scanline)
;	DX			(cursor mask/end scanline)
; Out:	none
; Use:	none
; Modf:	AX, CX, BX, cursortype, startscan, endscan
; Call:	INT 10/01, @showret, redrawcursor
;
textcursor_0A	proc
		xchg	cx,bx
	if_ ncxz				; if hardware cursor
		mov	ch,bl
		mov	cl,dl
		mov	ah,1
		int	10h			; set cursor shape & size
		mov	cl,1
	end_
		cmp	cl,[cursortype]
	if_ eq
		cmp	bx,[startscan]
	andif_ eq
		cmp	dx,[endscan]
		je	@showret		; exit if cursor not changed
	end_
;----------
		mov	[cursortype],cl
		mov	[startscan],bx
		mov	[endscan],dx
		j	redrawcursor
textcursor_0A	endp

;
; 10 - Define screen region for updating
;
;
; In:	CX, DX			(X/Y of upper left corner)
;	SI, DI			(X/Y of lower right corner)
; Out:	none
; Use:	none
; Modf:	AX, CX, DX, DI, upleft, lowright
; Call:	redrawcursor
;
updateregion_10	proc
		mov	ax,[_ARG_SI_]
IF FOOLPROOF
		cmp	cx,ax
	if_ ge
		xchg	cx,ax
	end_
		mov	[upleft.X],cx
		mov	[lowright.X],ax
		xchg	ax,di			; OPTIMIZE: instead MOV AX,DI
		cmp	dx,ax
	if_ ge
		xchg	dx,ax
	end_
		mov	[upleft.Y],dx
		mov	[lowright.Y],ax
ELSE
		mov	[upleft.X],cx
		mov	[upleft.Y],dx
		mov	[lowright.X],ax
		mov	[lowright.Y],di
ENDIF
		j	redrawcursor
updateregion_10	endp

;
; 16 - Save driver state
;
;
; In:	BX			(buffer size)
;	ES:DX			(buffer)
; Out:	none
; Use:	StartSaveArea
; Modf:	CX, SI, DI
; Call:	none
;
savestate_16	proc
		assume	es:nothing
IF FOOLPROOF
;;-		cmp	bx,LenSaveArea		;!!! TurboPascal IDE
;;-		jb	@stateret		;  workaround: it not init BX
ENDIF
		memcopy	LenSaveArea,,,dx,,,DGROUP:StartSaveArea
@stateret:	ret
savestate_16	endp

;
; 17 - Restore driver state
;
;
; In:	BX			(buffer size)
;	ES:DX			(saved state buffer)
; Out:	none
; Use:	none
; Modf:	SI, DI, DS, ES, StartSaveArea
; Call:	@stateret, redrawcursor
;
restorestate_17	proc
		assume	es:nothing
IF FOOLPROOF
;;-		cmp	bx,LenSaveArea		;!!! TurboPascal IDE
;;-		jb	@stateret		;  workaround: it not init BX
ENDIF
;---------- do nothing if SaveArea is not changed
;;*		mov	si,offset DGROUP:StartSaveArea
;;*		mov	di,dx
;;*		mov	cx,LenSaveArea/2
;;*ERRIF (LenSaveArea mod 2 ne 0) "LenSaveArea must be even!"
;;*		repe	cmpsw
;;*		je	@stateret

;---------- change SaveArea
		push	es dx
		MOVSEG	es,ds,,DGROUP
		pop	si ds
		assume	ds:nothing
		memcopy	LenSaveArea,,,DGROUP:StartSaveArea
		j	redrawcursor
restorestate_17	endp
		assume	ds:DGROUP

;
; 0D - Light pen emulation ON
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	none
; Call:	lightpenoff_0E
;
;;*lightpenon_0D	proc
;;*		mov	al,0
;;*		;j	lightpenoff_0E
;;*lightpenon_0D	endp

;
; 0E - Light pen emulation OFF
;
;
; In:	none
; Out:	none
; Use:	none
; Modf:	nolightpen?
; Call:	none
;
;;*lightpenoff_0E	proc
;;*		mov	[nolightpen?],al	; OPTIMIZE: AL instead nonzero
;;*		ret
;;*lightpenoff_0E	endp

;
; 14 - Exchange User Interrupt Routines
;
;
; In:	CX			(new call mask)
;	ES:DX			(new FAR routine)
; Out:	[CX]			(old call mask)
;	[ES:DX]			(old FAR routine)
; Use:	callmask, UIR@
; Modf:	AX
; Call:	UIR_0C
;
exchangeUIR_14	proc
		assume	es:nothing
		;mov	ah,0
		mov	al,[callmask]
		mov	[_ARG_CX_],ax
		mov	ax,word ptr UIR@[0]
		mov	[_ARG_DX_],ax
		mov	ax,word ptr UIR@[2]
		mov	[_ARG_ES_],ax
		;j	UIR_0C
exchangeUIR_14	endp

;
; 0C - Define User Interrupt Routine
;
;
; In:	CX			(call mask)
;	ES:DX			(FAR routine)
; Out:	none
; Use:	none
; Modf:	UIR@, callmask
; Call:	none
;
UIR_0C		proc
		assume	es:nothing
		saveFAR [UIR@],es,dx
		mov	[callmask],cl
		ret
UIR_0C		endp

;
; 0F - Set mickeys/pixels ratios
;
;
; In:	CX			(number of mickeys per 8 pix
;	DX			 horizontally/vertically)
; Out:	none
; Use:	none
; Modf:	mickey8
; Call:	none
;
sensitivity_0F	proc
IF FOOLPROOF
		test	dx,dx
	if_ nz					; ignore wrong ratio
	andif_ ncxz				; ignore wrong ratio
ENDIF
		mov	[mickey8.X],cx
		mov	[mickey8.Y],dx
IF FOOLPROOF
	end_
ENDIF
		ret
sensitivity_0F	endp

;
; 1A - Set mouse sensitivity
;
;
; In:	BX			(horizontal sensitivity, 1..100)
;	CX			(vertical sensitivity, 1..100)
;	DX			(speed threshold in mickeys/second, ignored)
; Out:	none
; Use:	none
; Modf:	AX, CX, BX
; Call:	senscalc
;
sensitivity_1A	proc
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		MOVREG_	bx,<offset X>
		call	senscalc
		xchg	ax,cx			; OPTIMIZE: instead MOV AX,CX
		MOVREG_	bl,<offset Y>		; OPTIMIZE: BL instead BX
		;j	senscalc
sensitivity_1A	endp

;
; In:	AX			(001A value)
;	BX			(offset X/offset Y)
; Out:	none
; Modf:	AX, DX, DI, sensval, senscoeff
; Call:	none
; Hint:	sensval=AX-1, senscoeff=[(sensval^2/3600+1/3)*256]
;
senscalc	proc
		dec	ax
		cmp	ax,100			; ignore original values
	if_ below				;  outside [1..100]
		mov	word ptr sensval[bx],ax
		mul	ax			; DX:AX=V^2 (0<=X^2<10000)
		add	ax,3600/3
		;adc	dx,0			; DX:AX=V^2+1200
		;mov	dh,0
		;xchg	dh,dl
		xchg	dl,al
		xchg	dl,ah			; DX:AX=(V^2+1200)*256
		mov	di,3600
		div	di			; AX=(V^2+1200)*256/3600
		mov	word ptr senscoeff[bx],ax
	end_
		ret
senscalc	endp

;
; 13 - Define double-speed threshold
;
;
; In:	DX			(speed threshold in mickeys/second)
; Out:	none
; Use:	none
; Modf:	/DX/, /doublespeed/
; Call:	none
;
doublespeed_13	proc
;;*		test	dx,dx
;;*	if_ zero
;;*		mov	dl,64
;;*	end_
;;*		mov	[doublespeed],dx
		;ret
doublespeed_13	endp

;
; 0D 0E 11 12 18 19 1C 1D - Null function for not implemented calls
;

nullfunc	proc
		ret
nullfunc	endp

;
;				INT 33 handler
;

		evendata
handler33table	dw	offset DGROUP:resetdriver_00
		dw	offset DGROUP:showcursor_01
		dw	offset DGROUP:hidecursor_02
		dw	offset DGROUP:status_03
		dw	offset DGROUP:setpos_04
		dw	offset DGROUP:pressdata_05
		dw	offset DGROUP:releasedata_06
		dw	offset DGROUP:hrange_07
		dw	offset DGROUP:vrange_08
		dw	offset DGROUP:graphcursor_09
		dw	offset DGROUP:textcursor_0A
		dw	offset DGROUP:mickeys_0B
		dw	offset DGROUP:UIR_0C
		dw	offset DGROUP:nullfunc	;lightpenon_0D
		dw	offset DGROUP:nullfunc	;lightpenoff_0E
		dw	offset DGROUP:sensitivity_0F
		dw	offset DGROUP:updateregion_10
		dw	offset DGROUP:wheelAPI_11
		dw	offset DGROUP:nullfunc	;12 - large graphics cursor
		dw	offset DGROUP:doublespeed_13
		dw	offset DGROUP:exchangeUIR_14
		dw	offset DGROUP:storagereq_15
		dw	offset DGROUP:savestate_16
		dw	offset DGROUP:restorestate_17
		dw	offset DGROUP:althandler_18
		dw	offset DGROUP:althandler_19
		dw	offset DGROUP:sensitivity_1A
		dw	offset DGROUP:sensitivity_1B
		dw	offset DGROUP:nullfunc	;1C - InPort mouse only
		dw	offset DGROUP:nullfunc	;1D - define display page #
		dw	offset DGROUP:videopage_1E
		dw	offset DGROUP:disabledrv_1F
		dw	offset DGROUP:enabledriver_20
		dw	offset DGROUP:softreset_21

handler33	proc
		assume	ds:nothing,es:nothing
		cld
		test	ah,ah
	if_ zero
		push	ds
		MOVSEG	ds,cs,,DGROUP
		cmp	al,21h
		ja	language_23
		push	es
		PUSHALL
		mov	si,ax			;!!! AX must be unchanged
		mov	bp,sp
		shl	si,1
		call	handler33table[si]	; call by calculated offset
@rethandler:	POPALL
		pop	es ds
	end_
		iret
		assume	ds:DGROUP

;
; 23 - Get language for messages
; Out:	[BX]			(language code: 0 - English)
;
language_23:	cmp	al,23h
		je	@iretBX0

;
; 24 - Get software version, mouse type and IRQ
; Out:	[AX] = 24h/FFFFh	(installed/error)
;	[BX]			(version)
;	[CL]			(IRQ #/0=PS/2)
;	[CH] = 1=bus/2=serial/3=InPort/4=PS2/5=HP (mouse type)
; Use:	driverversion
;
version_24:	cmp	al,24h
	if_ eq
		mov	bx,driverversion
		OPCODE_MOV_CX
mouseinfo	db	?,4
	end_

;
; 26 - Get maximum virtual screen coordinates
; Out:	[BX]			(mouse disabled flag)
;	[CX]			(max virtual screen X)
;	[DX]			(max virtual screen Y)
; Use:	bitmapshift
;
maxscreen_26:	cmp	al,26h
	if_ eq
		mov	cl,[bitmapshift]
		OPCODE_MOV_BX
scanline	dw	?
		OPCODE_MOV_DX
screenheight	dw	?
		shl	bx,cl
		dec	dx
		mov	cx,bx
		dec	cx
		OPCODE_MOV_BX
disabled?	db	1,0			; 1=driver disabled
	end_

;
; 27 - Get screen/cursor masks and mickey counters
; Out:	[AX]			(screen mask/start scanline)
;	[BX]			(cursor mask/end scanline)
;	[CX]			(number of mickeys mouse moved
;	[DX]			 horizontally/vertically since last call)
; Use:	startscan, endscan
; Modf:	mickeys
;
cursor_27:	cmp	al,27h
	if_ eq
		mov	ax,[startscan]
		mov	bx,[endscan]
		xor	cx,cx
		xor	dx,dx
		xchg	cx,[mickeys.X]
		xchg	dx,[mickeys.Y]
		pop	ds
		iret
	end_

;
; 31 - Get current virtual cursor coordinates
; Out:	[AX]			(min virtual cursor X)
;	[BX]			(min virtual cursor Y)
;	[CX]			(max virtual cursor X)
;	[DX]			(max virtual cursor Y)
; Use:	rangemin, rangemax
;
cursrange_31:	cmp	al,31h
	if_ eq
		mov	ax,[rangemin.X]
		mov	bx,[rangemin.Y]
		lds	cx,[rangemax]
		mov	dx,ds
		pop	ds
		iret
	end_

;
; 32 - Get supported advanced functions flag
; Out:	[AX]			(bits 15-0=function 25h-34h supported)
;	[CX] = 0
;	[DX] = 0
;	[BX] = 0
;
active_32:	cmp	al,32h
	if_ eq
		mov	ax,0111010000001100b	; active: 26 27 28 2A 31 32
		xor	cx,cx
		xor	dx,dx
@iretBX0:	xor	bx,bx
		pop	ds
		iret
	end_

;
; 4D - Get pointer to copyright string
; Out:	[ES:DI]			(copyright string)
; Use:	IDstring
;
copyright_4D:	cmp	al,4Dh
	if_ eq
		MOVSEG	es,ds,,DGROUP
		mov	di,offset DGROUP:IDstring
	end_

;
; 6D - Get pointer to version
; Out:	[ES:DI]			(version string)
; Use:	msversion
;
version_6D:	cmp	al,6Dh
	if_ eq
		MOVSEG	es,ds,,DGROUP
		mov	di,offset DGROUP:msversion
	end_

;
; 28 - Set video mode
; In:	CX			(video mode)
;	DX			(font size, ignored)
; Out:	[CX]			(=0 if successful)
; Modf:	none
; Call:	none
;
setvidmode_28:	cmp	al,28h
	if_ eq
		push	ax bx bp		;!!! some BIOSes trash BP
		test	ch,ch			; VESA mode >= 100h
	 if_ zero
		mov	ax,cx
		;mov	ah,0
		int	10h			; set the video mode in AL
		mov	ah,0Fh
		int	10h			; get current video mode
		cmp	al,cl			; CL=requested video mode
		je	@@setmoderet0		; return if successful
	 end_
		mov	bx,cx
		mov	ax,4F02h
		int	10h			; set VESA video mode in BX
		cmp	ax,004Fh		; CX=requested video mode
		jne	@@setmoderet

@@setmoderet0:	xor	cx,cx			; CX=0 if successful
@@setmoderet:	pop	bp bx ax		; CX=requested mode or 0
	end_

;
; 2A - Get cursor hot spot
; Out:	[AX]			(cursor visibility counter)
;	[BX]			(hot spot X)
;	[CX]			(hot spot Y)
;	[DX] = 1=bus/2=serial/3=InPort/4=PS2/5=HP (mouse type)
; Use:	nocursorcnt, hotspot
;
hotspot_2A:	cmp	al,2Ah
	if_ eq
		;mov	ah,0
		mov	al,[nocursorcnt]
		lds	bx,[hotspot]
		mov	cx,ds
		OPCODE_MOV_DX
mouseinfo1	db	4,0
	end_

		pop	ds
		iret
handler33	endp

; END OF INT 33 SERVICES 


RILversion	label
msversion	db	driverversion / 100h,driverversion mod 100h
IDstring	db	'CuteMouse ',CTMVER,0
IDstringlen = $ - IDstring

TSRend		label


; INITIALIZATION PART DATA 

.const

messages segment virtual ; place at the end of current segment
INCLUDE ctmouse.msg
messages ends

S_mousetype	dw	DGROUP:S_atPS2
		dw	DGROUP:S_inMSYS
		dw	DGROUP:S_inLT
		dw	DGROUP:S_inMS

.data

options		dw	0
OPT_PS2		equ		00000001b
OPT_serial	equ		00000010b
OPT_COMforced	equ		00000100b
OPT_PS2after	equ		00001000b
OPT_3button	equ		00010000b
OPT_noMSYS	equ		00100000b
OPT_lefthand	equ		01000000b
OPT_noUMB	equ		10000000b
OPT_newTSR	equ	0000000100000000b


; Real Start 

.code

say		macro	stroff
		MOVOFF_	di,<stroff>
		call	sayASCIIZ
endm

real_start:	cld
		DOSGetIntr 33h
		saveFAR [oldint33],es,bx	; save old INT 33h handler

;---------- parse command line and find mouse
		say	DGROUP:Copyright	; 'Cute Mouse Driver'
		mov	si,offset PSP:cmdline_len
		lodsb
		cbw				; OPTIMIZE: instead MOV AH,0
		mov	bx,ax
		mov	[si+bx],ah		; OPTIMIZE: AH instead 0
		call	commandline		; examine command line

		mov	cx,word ptr oldint33[2]
	if_ ncxz
		mov	ax,1Fh			; disable old driver
		pushf				;!!! Logitech MouseWare
		call	[oldint33]		;  Windows driver workaround
	end_
;----------
		mov	ax,[options]
		testflag ax,OPT_PS2+OPT_serial
	if_ zero				; if no /S and /P then
		setflag	ax,OPT_PS2+OPT_serial	;  both PS2 and serial assumed
	end_
;---
		testflag ax,OPT_PS2after
	if_ nz
		call	searchCOM		; call if /V
		jnc	@@serialfound
	end_
;---
		testflag ax,OPT_PS2+OPT_PS2after
	if_ nz
		push	ax
		call	checkPS2		; call if /V or PS2
		pop	ax
	andif_ nc
		mov	mouseinfo[0],bh
		j	@@mousefound
	end_
;---
		testflag ax,OPT_PS2after
	if_ zero
		testflag ax,OPT_serial+OPT_noMSYS
	andif_ nz
		call	searchCOM		; call if no /V and serial
		jnc	@@serialfound
	end_
		mov	di,offset DGROUP:E_notfound ; 'Error: device not found'
		jmp	EXITENABLE
;----------
@@serialfound:	;push	ax			; preserve OPT_newTSR value
		mov	al,2
		mov	mouseinfo[1],al
		mov	[mouseinfo1],al
		;pop	ax
@@mousefound:	mov	[mousetype],bl

;---------- check if CuteMouse driver already installed
		testflag ax,OPT_newTSR
		jnz	@@newTSR
		call	getCuteMouse
		mov	di,offset DGROUP:S_reset ; 'Resident part reset to'
		mov	cx,4C02h		; terminate, al=return code

	if_ ne
;---------- allocate UMB memory, if possible, and set INT 33 handler
@@newTSR:	mov	bx,(TSRend-TSRstart+15)/16
		push	bx
		call	prepareTSR		; new memory segment in ES
		memcopy	<size oldint33>,es,,DGROUP:oldint33,ds,,DGROUP:oldint33
		push	ds
		MOVSEG	ds,es,,DGROUP
		mov	[disabled?],1		; copied back in setupdriver
		DOSSetIntr 33h,,,DGROUP:handler33
		pop	ds ax
		mov	di,offset DGROUP:S_installed ; 'Installed at'
		mov	cl,0			; errorlevel
	end_ if

;----------
		push	ax			; size of TSR for INT 21/31
		say	di
		mov	al,[mousetype]

		mov	bx,offset DGROUP:S_CRLF
		shl	al,1
	if_ carry				; if wheel (=8xh)
		mov	bx,offset DGROUP:S_wheel
	end_

		cbw				; OPTIMIZE: instead MOV AH,0
		cmp	al,1 shl 1
		xchg	si,ax			; OPTIMIZE: instead MOV SI,AX
	if_ ae					; if not PS/2 mode (=0)
	 if_ eq					; if Mouse Systems (=1)
		inc	cx			; OPTIMIZE: CX instead CL
	 end_
		say	DGROUP:S_atCOM
	end_
		push	cx			; exit function and errorlevel
		say	S_mousetype[si]
		say	bx
		call	setupdriver

;---------- close all handles (20 pieces) to prevent decreasing system
;	    pool of handles if INT 21/31 used
		mov	bx,19
	loop_
		DOSCloseFile
		dec	bx
	until_ sign
;----------
		pop	ax dx			; AH=31h (TSR) or 4Ch (EXIT)
		int	21h

;
;		Setup resident driver code and parameters
;

setupdriver	proc
;---------- detect VGA card (VGA ATC have one more register)
		mov	ax,1A00h
		int	10h			; get display type in BX
		cmp	al,1Ah
	if_ eq
		xchg	ax,bx			; OPTIMIZE: instead MOV AL,BL
		sub	al,7
		cmp	al,8-7
	andif_ be					; if monochrome or color VGA
		inc	videoregs@[(size RGROUPDEF)*3].regscnt
	end_

;---------- setup left hand mode handling
		OPCODE_MOV_CX
mousetype	db	?,0			; 0=PS/2,1=MSys,2=LT,3=MS,
						; 80h=PS/2+wheel,83h=MS+wheel
		test	cl,7Fh
		mov	al,00000000b	; =0
	if_ nz					; if not PS/2 mode (=x0h)
		mov	al,00000011b	; =3
	end_
		testflag [options],OPT_lefthand
	if_ nz
		flipflag al,00000011b	; =3
	end_
		mov	[SWAPMASK],al

;---------- setup buttons count, mask and wheel flags
		mov	al,3
		testflag [options],OPT_3button
	if_ zero
		jcxz	@@setbuttons		; jump if PS/2 mode (=0)
		cmp	cl,al			; OPTIMIZE: AL instead 3
	andif_ eq				; if MS mode (=3)
@@setbuttons:	mov	[buttonsmask],al	; OPTIMIZE: AL instead 0011b
		dec	ax
		mov	[buttonscnt],al		; OPTIMIZE: AL instead 2
	end_

		cmp	cl,80h
	if_ ae
		inc	wheelflags[0]
		mov	[wheelmask],00001111b	; =0Fh
	end_

;---------- setup mouse handlers code
	block_
		test	cl,7Fh
	 breakif_ zero				; break if PS/2 mode (=x0h)

		mov	word ptr [IRQproc],((OCW2<OCW2_EOI>) shl 8)+0B0h
						; MOV AL,OCW2<OCW2_EOI>
		mov	[BUFFADDR],offset DGROUP:bufferSERIAL
		mov	[enableproc],enableUART-enableproc-2
		mov	[disableproc],disableUART-disableproc-2
		dec	cx
	 breakif_ zero				; break if Mouse Systems mode (=1)

		mov	[mouseproc],MSLTproc-mouseproc-2
		dec	cx
	 breakif_ zero				; break if Logitech mode (=2)

		mov	MSLTCODE3[1],2
		loop	@@setother		; break if wheel mode (=83h)

		cmp	al,2			; OPTIMIZE: AL instead [buttonscnt]
	 if_ ne					; if not MS2
		mov	[MSLTCODE2],075h	; JNZ
	 end_
		mov	al,0C3h			; RET
		mov	[MSLTCODE1],al
		mov	[MSLTCODE3],al
	end_ block

;---------- setup, if required, other parameters
@@setother:	push	es ds es ds
		pop	es ds			; get back [oldint10]...
		memcopy	<size oldint10>,es,,DGROUP:oldint10,ds,,DGROUP:oldint10
		mov	al,[disabled?]
		pop	ds
		mov	[disabled?],al		; ...and [disabled?]

		DOSGetIntr <byte ptr [IRQintnum]>
		mov	ax,es
		pop	es
		mov	di,offset DGROUP:oldIRQaddr
		xchg	ax,bx
		stosw				; save old IRQ handler
		xchg	ax,bx			; OPTIMIZE: instead MOV AX,BX
		stosw

;---------- copy TSR image (even if ES=DS - this is admissible)
		memcopy	((TSRend-TSRdata+1)/2)*2,es,,DGROUP:TSRdata,ds,,DGROUP:TSRdata

;---------- call INT 33/0000 (Reset driver)
		pop	ax
		pushf				;!!! Logitech MouseWare
		push	cs ax			;  Windows driver workaround
		mov	ax,offset DGROUP:handler33
		push	es ax
		xor	ax,ax			; reset driver
		retf
setupdriver	endp

;
;		Check given or all COM-ports for mouse connection
;

searchCOM	proc
		;mov	[LCRset],LCR<0,,LCR_noparity,0,2>
		mov	di,offset DGROUP:detectmouse
		call	COMloop
		jnc	@searchret

		testflag [options],OPT_noMSYS
		stc
		jnz	@searchret

		mov	[LCRset],LCR<0,,LCR_noparity,0,3>
		mov	bl,1			; =Mouse Systems mode
		mov	di,offset DGROUP:checkUART
		;j	COMloop
searchCOM	endp

;

COMloop		proc
		push	ax
		xor	ax,ax			; scan only current COM port
		testflag [options],OPT_COMforced
		jnz	@@checkCOM
		mov	ah,3			; scan all COM ports

	loop_
		inc	ax			; OPTIMIZE: AX instead AL
		push	ax
		call	setCOMport
		pop	ax
@@checkCOM:	push	ax
		mov	si,[IO_address]
		call	di
		pop	ax
		jnc	@@searchbreak
		dec	ah
	until_ sign
		;stc				; preserved from prev call

@@searchbreak:	pop	ax
@searchret:	ret
COMloop		endp

;
;			Check if UART available
;
;
; In:	SI = [IO_address]
; Out:	Carry flag		(no UART detected)
; Use:	none
; Modf:	AX, DX
; Call:	none
;
checkUART	proc
		test	si,si
		 jz	@@noUART		; no UART if base=0

;---------- check UART registers for reserved bits
		movidx	dx,MCR_index,si		; {3FCh} MCR (modem ctrl reg)
		 in	ax,dx			; {3FDh} LSR (line status reg)
		testflag al,mask MCR_reserved+mask MCR_AFE
		 jnz	@@noUART
		movidx	dx,LSR_index,si,MCR_index
		 in	al,dx			; {3FDh} LSR (line status reg)
		inc	ax
		 jz	@@noUART		; no UART if AX was 0FFFFh

;---------- check LCR function
		cli
		movidx	dx,LCR_index,si,LSR_index
		 in	al,dx			; {3FBh} LCR (line ctrl reg)
		 push	ax
		out_	dx,%LCR<1,0,-1,-1,3>	; {3FBh} LCR: DLAB on, 8S2
		 inb	ah,dx
		out_	dx,%LCR<0,0,0,0,2>	; {3FBh} LCR: DLAB off, 7N1
		 in	al,dx
		sti
		sub	ax,(LCR<1,0,-1,-1,3> shl 8)+LCR<0,0,0,0,2>

	if_ zero				; zero if LCR conforms
;---------- check IER for reserved bits
		movidx	dx,IER_index,si,LCR_index
		 in	al,dx			; {3F9h} IER (int enable reg)
		movidx	dx,LCR_index,si,IER_index
		;mov	ah,0
		and	al,mask IER_reserved	; reserved bits should be clear
	end_ if

		neg	ax			; nonzero makes carry flag
		pop	ax
		 out	dx,al			; {3FBh} LCR: restore contents
		ret

@@noUART:	stc
		ret
checkUART	endp

;
;			Detect mouse type if present
;
;
; In:	SI = [IO_address]
; Out:	Carry flag		(no UART or mouse found)
;	BX			(mouse type: 2=Logitech,3=MS,83h=MS+wheel)
; Use:	0:46Ch, LCRset
; Modf:	AX, CX, DX, ES
; Call:	checkUART
;
detectmouse	proc
		call	checkUART
		jc	@@detmret

;---------- save current LCR/MCR
		movidx	dx,LCR_index,si		; {3FBh} LCR (line ctrl reg)
		 in	ax,dx			; {3FCh} MCR (modem ctrl reg)
		 push	ax			; keep old LCR and MCR values

;---------- reset UART: drop RTS line, interrupts and disable FIFO
		;movidx	dx,LCR_index,si		; {3FBh} LCR: DLAB off
		 out_	dx,%LCR<>,%MCR<>	; {3FCh} MCR: DTR/RTS/OUT2 off
		movidx	dx,IER_index,si,LCR_index
		 ;mov	ax,(FCR<> shl 8)+IER<>	; {3F9h} IER: interrupts off
		 out	dx,ax			; {3FAh} FCR: disable FIFO

;---------- set communication parameters and flush receive buffer
		movidx	dx,LCR_index,si,IER_index
		 out_	dx,%LCR{LCR_DLAB=1}	; {3FBh} LCR: DLAB on
		xchg	dx,si
		 ;mov	ah,0			; 1200 baud rate
		 out_	dx,96,ah		; {3F8h},{3F9h} divisor latch
		xchg	dx,si
		 out_	dx,[LCRset]		; {3FBh} LCR: DLAB off, 7/8N1
		movidx	dx,RBR_index,si,LCR_index
		 in	al,dx			; {3F8h} flush receive buffer

;---------- wait current+next timer tick and then raise RTS line
		MOVSEG	es,0,ax,BIOS
	loop_
		mov	ah,byte ptr [BIOS_timer]
	 loop_
		cmp	ah,byte ptr [BIOS_timer]
	 until_ ne				; loop until next timer tick
		xor	al,1
	until_ zero				; loop until end of 2nd tick

		movidx	dx,MCR_index,si,RBR_index
		 out_	dx,%MCR<,,,0,,1,1>	; {3FCh} MCR: DTR/RTS on, OUT2 off

;---------- detect if Microsoft or Logitech mouse present
		mov	bx,0103h		; bl=mouse type, bh=no `M'
	countloop_ 4,cl				; scan 4 first bytes
	 countloop_ 2+1,ch			; length of silence in ticks
						; (include rest of curr tick)
		mov	ah,byte ptr [BIOS_timer]
	  loop_
		movidx	dx,LSR_index,si
		 in	al,dx			; {3FDh} LSR (line status reg)
		testflag al,mask LSR_RBF
		 jnz	@@parse			; jump if data ready
		cmp	ah,byte ptr [BIOS_timer]
	  until_ ne				; loop until next timer tick
	 end_ countloop				; loop until end of 2nd tick
	 break_					; break if no more data

@@parse:	movidx	dx,RBR_index,si
		 in	al,dx			; {3F8h} receive byte
		cmp	al,'('-20h
	 breakif_ eq				; break if PnP data starts
		cmp	al,'M'
	 if_ eq
		mov	bh,0			; MS compatible mouse found...
	 end_
		cmp	al,'Z'
	 if_ eq
		mov	bl,83h			; ...MS mouse+wheel found
	 end_
		cmp	al,'3'
	 if_ eq
		mov	bl,2			; ...Logitech mouse found
	 end_
	end_ countloop

		movidx	dx,LCR_index,si
		 pop	ax			; {3FBh} LCR: restore contents
		 out	dx,ax			; {3FCh} MCR: restore contents

		shr	bh,1			; 1 makes carry flag
@@detmret:	ret
detectmouse	endp

;
;				Check for PS/2
;
;
; In:	none
; Out:	Carry flag		(no PS/2 device found)
;	BL			(mouse type: 0=PS/2)
;	BH			(interrupt #/0=PS/2)
; Use:	none
; Modf:	AX, CX, BX
; Call:	INT 11, INT 15/C2xx, setRate, outKBD, flushKBD
;
checkPS2	proc
		int	11h			; get equipment list
		testflag al,mask HW_PS2
		jz	@@noPS2			; jump if PS/2 not indicated
		mov	bh,3
		PS2serv 0C205h,@@noPS2		; initialize mouse, bh=datasize
		mov	bh,3
		PS2serv 0C203h,@@noPS2		; set mouse resolution bh
		MOVSEG	es,cs,,DGROUP
		mov	bx,offset DGROUP:PS2dummy
		PS2serv	0C207h,@@noPS2		; mouse, es:bx=ptr to handler
		MOVSEG	es,0,bx,nothing
		PS2serv	0C207h			; mouse, es:bx=ptr to handler

;---------- select IntelliMouse Z wheel + 3 button mode
		mov	ah,200
		call	setRate
		mov	ah,100
		call	setRate
		mov	ah,80
		call	setRate			; 200->100->80 rate does this

;---------- check if successful
		mov	dx,0D464h		; enable mouse functions
		call	outKBD
		mov	dx,0F260h		; get ID
		call	flushKBD
		xor	bx,bx			; =PS/2 mouse
		cmp	ah,3			; ID=3 -> 3 button+wheel mode
	if_ eq
		mov	PS2WHEELCODE[1],@PS2_3-PS2WHEELCODE-2
		mov	bl,80h			; =PS/2+wheel mouse
	end_
;----------
		mov	byte ptr [IRQintnum],68h+12
		clc
		ret
@@noPS2:	stc
		ret
PS2dummy:	retf
checkPS2	endp

;

SetRate		proc
		push	ax
		mov	dx,0D464h		; enable mouse functions
		call	outKBD
		mov	dx,0F360h		; set rate...
		call	flushKBD		; should be 0FAh (ACK) or 0FEh (resend)

		mov	dx,0D464h		; enable mouse functions
		call	outKBD
		pop	dx
		mov	dl,60h			; ...value
		;j	flushKBD		; should be 0FAh (ACK) or 0FEh (resend)
SetRate		endp

;
; In:	DL			(port)
;	DH			(value)
; Out:	AH
; Use:	0:46Ch
; Modf:	AL, DX, ES
; Call:	outKBD
;
flushKBD	proc
		call	outKBD

		mov	ah,0
@@inKBDstart:	MOVSEG	es,0,dx,BIOS
	loop_
		mov	dl,byte ptr [BIOS_timer]
	 loop_
		in	al,64h			; keyboard status register
		testflag al,00000001b	; =1
	  if_ nz				; if read buffer full
		 inb	ah,60h
		 j	@@inKBDstart		; wait next byte
	  end_
		cmp	dl,byte ptr [BIOS_timer]
	 until_ ne				; loop until next timer tick
		xor	dh,1
	until_ zero				; loop until end of 2nd tick
		ret
flushKBD	endp

;
;			Send command to auxiliary device
;
;
; In:	DL			(port)
;	DH			(value)
; Out:	none
; Use:	0:46Ch
; Modf:	AL, DH, ES
; Call:	none
;
outKBD		proc
		push	dx
		MOVSEG	es,0,dx,BIOS
	loop_
		mov	dl,byte ptr [BIOS_timer]
	 loop_
		in	al,64h			; keyboard status register
		testflag al,00000010b	; =2	; check if we can send data
		 jz	@@outKBD		; jump if write buffer empty
		cmp	dl,byte ptr [BIOS_timer]
	 until_ ne				; loop until next timer tick
		xor	dh,1
	until_ zero				; loop until end of 2nd tick

@@outKBD:	pop	dx
		mov	al,dh
		mov	dh,0
		out	dx,al
		ret
outKBD		endp

;
;				Set COM port
;
;
; In:	AL			(COM port, 1-4)
; Out:	none
; Use:	0:400h
; Modf:	AL, CL, ES, com_port, IO_address, S_atIO
; Call:	setIRQ
;
setCOMport	proc
		push	ax di
		add	al,'0'
		mov	[com_port],al

		cbw				; OPTIMIZE: instead MOV AH,0
		xchg	di,ax			; OPTIMIZE: instead MOV DI,AX
		MOVSEG	es,0,ax,BIOS
		shl	di,1
		mov	ax,COM_base[di-'1'-'1']
		mov	[IO_address],ax

		MOVSEG	es,ds,,DGROUP
		mov	di,offset DGROUP:S_atIO	; string for 4 digits
		mov	cl,12
	loop_
		push	ax
		shr	ax,cl
		and	al,0Fh
		digit2x
		stosb
		pop	ax
		sub	cl,4
	until_ below

		pop	di ax
		and	al,1			; 1=COM1/3, 0=COM2/4
		add	al,3			; IRQ4 for COM1/3
		;j	setIRQ			; IRQ3 for COM2/4
setCOMport	endp

;
;				Set IRQ number
;
;
; In:	AL			(IRQ#, 1-7)
; Out:	none
; Use:	none
; Modf:	AL, CL, IRQno, mouseinfo, IRQintnum, PIC1state, notPIC1state
; Call:	none
;
setIRQ		proc
		add	al,'0'
		mov	[IRQno],al
		sub	al,'0'
		mov	mouseinfo[0],al
		mov	cl,al
		add	al,8			; INT=IRQ+8
		mov	byte ptr [IRQintnum],al
		mov	al,1
		shl	al,cl			; convert IRQ into bit mask
		mov	[PIC1state],al		; PIC interrupt disabler
		not	al
		mov	[notPIC1state],al	; PIC interrupt enabler
		ret
setIRQ		endp

;
;		Check if CuteMouse driver is installed
;
;
; In:	none
; Out:	Zero flag		(ZF=1 if installed)
;	ES			(driver segment)
; Use:	oldint33, IDstring
; Modf:	AX, CX, SI, DI
; Call:	INT 33/004D
;
getCuteMouse	proc
		cmp	word ptr oldint33[2],1
	if_ ae					; if handler seg nonzero
		mov	ax,4Dh
		;xor	di,di
		pushf				;!!! Logitech MouseWare
		call	[oldint33]		;  Windows driver workaround
		mov	si,offset DGROUP:IDstring
		cmp	di,si
	andif_ eq
		mov	cx,IDstringlen
		repe	cmpsb
	end_
		ret
getCuteMouse	endp


; COMMAND LINE PARSING 

;
;			Parse Serial option
;

_serialopt	proc
		mov	bx,(4 shl 8)+1
		call	parsedigit
	if_ nc					; '/Sc' -> set COM port
		setflag	[options],OPT_COMforced
		call	setCOMport

		;mov	bl,1
		mov	bh,7
		call	parsedigit
		jnc	setIRQ			; '/Sci' -> set IRQ line
	end_
		ret
_serialopt	endp

;
;			Parse Resolution option
;

_resolution	proc
		;mov	ah,0
		mov	bx,(9 shl 8)+0
		call	parsedigit		; first argument
	if_ nc
		mov	ah,al
		;mov	bx,(9 shl 8)+0
		call	parsedigit		; second argument
		jnc	@@setres		; jump if digit present
	end_
		mov	al,ah			; replicate missing argument

@@setres:	add	ax,0101h
		push	ax			; AL=RY+1, AH=RX+1
		mov	al,10
		mul	ah			; AX=10*(RX+1)
		mov	bx,offset X+(StartDefArea-StartSaveArea)
		call	senscalc
		pop	ax
		mov	ah,10
		mul	ah			; AX=10*(RY+1)
		mov	bx,offset Y+(StartDefArea-StartSaveArea)
		jmp	senscalc
_resolution	endp

;
; In:	DS:SI			(string pointer)
;	BL			(low bound)
;	BH			(upper bound)
; Out:	DS:SI			(pointer after digit)
;	Carry flag		(no digit)
;	AL			(digit if no carry)
; Use:	none
; Modf:	CX
; Call:	BADOPTION
;
parsedigit	proc
		lodsb
		;c2digit
		sub	al,'0'
		cmp	al,bh
	if_ be
		cmp	al,bl
		jae	@ret			; JAE mean CF=0
	end_
		cmp	al,10
		mov	cx,offset DGROUP:E_argument ; 'Error: Invalid argument'
		jb	BADOPTION		; error if decimal digit
		dec	si
		stc
@ret:		ret
parsedigit	endp

;
;		Check if mouse services already present
;

_checkdriver	proc
		mov	cx,word ptr oldint33[2]
		jcxz	@ret
		;mov	ah,0
		mov	al,21h			; OPTIMIZE: AL instead AX
		int	33h
		inc	ax
		jnz	@ret
		mov	di,offset DGROUP:E_mousepresent ; 'Mouse service already...'
		j	EXITMSG
_checkdriver	endp

;
.const

OPTION		struc
  optchar	db	?
  optmask	dw	0
  optproc@	dw	?
ends

OPTABLE		OPTION	<'P',OPT_PS2,			@ret>
		OPTION	<'S',OPT_serial,		_serialopt>
		OPTION	<'Y',OPT_noMSYS,		@ret>
		OPTION	<'V',OPT_PS2after,		@ret>
		OPTION	<'3' and not 20h,OPT_3button,	@ret>
		OPTION	<'R',,				_resolution>
		OPTION	<'L',OPT_lefthand,		@ret>
		OPTION	<'B',,				_checkdriver>
		OPTION	<'N',OPT_newTSR,		@ret>
		OPTION	<'W',OPT_noUMB,			@ret>
		OPTION	<'U',,				unloadTSR>
		OPTION	<'?' and not 20h,,		EXITMSG>
OPTABLEend	label

.code

;
; In:	DS:SI			(null terminated command line)
;
commandline	proc
	loop_
		lodsb
		test	al,al
		jz	@ret			; exit if end of command line
		cmp	al,' '
	until_ above				; skips spaces and controls

		cmp	al,'/'			; option character?
	if_ eq
		lodsb
		and	al,not 20h		; uppercase
		mov	di,offset DGROUP:Syntax	; 'Options:'
		mov	bx,offset DGROUP:OPTABLE
	 loop_
		cmp	al,[bx].optchar
	  if_ eq
		mov	ax,[bx].optmask
		or	[options],ax
		call	[bx].optproc@
		j	commandline
	  end_
		add	bx,size OPTION
		cmp	bx,offset DGROUP:OPTABLEend
	 until_ ae
	end_ if

		mov	cx,offset DGROUP:E_option ; 'Error: Invalid option'
BADOPTION:	say	DGROUP:E_error		; 'Error: Invalid '
		say	cx			; 'option'/'argument'
		mov	di,offset DGROUP:E_help	; 'Enter /? on command line'

EXITMSG:	mov	bl,[di]
		inc	di
		say	di
		say	DGROUP:S_CRLF
		xchg	ax,bx			; OPTIMIZE: instead MOV AL,BL
		mov	ah,4Ch			; terminate, al=return code
		int	21h
commandline	endp

;
; In:	DS:DI			(null terminated string)
; Out:	none
; Use:	none
; Modf:	AH, DL, DI
; Call:	none
;
sayASCIIZ_	proc
	loop_
		DOSWriteC dl
		inc	di
sayASCIIZ:	mov	dl,[di]
		or	dl,dl
	until_ zero
		ret
sayASCIIZ_	endp


; TSR MANAGEMENT 

;
;			Unload driver and quit
;

unloadTSR	proc
		call	getCuteMouse		; check if CTMOUSE installed
		mov	di,offset DGROUP:E_nocute ; 'CuteMouse driver is not installed!'
		jne	EXITMSG

		push	es
		mov	ax,1Fh			; disable CuteMouse driver
		pushf				;!!! Logitech MouseWare
		call	[oldint33]		;  Windows driver workaround
		mov	cx,es
		pop	es
		cmp	al,1Fh
		mov	di,offset DGROUP:E_notunload ; 'Driver unload failed...'
		jne	@@exitenable

		saveFAR	[oldint33],cx,bx
		push	ds
		DOSSetIntr 33h,cx,,bx		; restore old int33 handler
		pop	ds
		call	FreeMem
		mov	di,offset DGROUP:S_unloaded ; 'Driver successfully unloaded...'
EXITENABLE:	mov	cx,word ptr oldint33[2]
		jcxz	EXITMSG			; exit if no old driver
@@exitenable:	mov	ax,20h			; enable old driver
		pushf				;!!! Logitech MouseWare
		call	[oldint33]		;  Windows driver workaround
		j	EXITMSG
unloadTSR	endp

;
; Prepare memory for TSR
;
; In:	BX			(TSR size)
;	DS			(PSP segment)
; Out:	ES			(memory segment to be TSR)
;	CH			(exit code for INT 21)
; Use:	PSP:2Ch, MCB:8
; Modf:	AX, CL, DX, SI, DI
; Call:	INT 21/49, AllocUMB
;
prepareTSR	proc
		assume	ds:PSP
		mov	cx,[env_seg]
	if_ ncxz				; suggested by Matthias Paul
		DOSFreeMem cx			; release environment block
		mov	[env_seg],0		; zero-pad for MEM-like pgms
		assume	ds:DGROUP
	end_

		call	AllocUMB
		mov	ax,ds
		mov	es,dx
		mov	ch,31h			; TSR exit, al=return code
		cmp	dx,ax
	if_ ne					; if TSR not "in place"
		push	ds
		dec	ax			; current MCB
		dec	dx			; target MCB...
						; ...copy process name
		memcopy	8,dx,MCB,MCB:ownername,ax,MCB,MCB:ownername
		POPSEG	ds,DGROUP

		inc	dx
		mov	[MCB:ownerID],dx	; ...and set owner to itself

		mov	ah,26h
		int	21h			; create PSP at segment in DX
		movadd	ax,[MCB_size],dx	; =target (PSP seg+MCB size)
		MOVSEG	es,dx,,PSP		; target PSP...
		mov	es:[PSP:next_seg],ax	; ...fix field

		mov	ch,4Ch			; terminate, al=return code
	end_ if
		ret
prepareTSR	endp


; MEMORY HANDLING 

;
; Get XMS handler address
;
; In:	none
; Out:	Carry flag		(set if no XMS support)
; Use:	none
; Modf:	AX, CX, BX, XMSentry
; Call:	INT 2F/4300, INT 2F/4310
;
getXMSaddr	proc	C uses es
		DOSGetIntr 2Fh			; suggested by Matthias Paul
		mov	cx,es
		stc
	if_ ncxz				; if INT 2F initialized
		mov	ax,4300h
		int	2Fh			; XMS: installation check
		cmp	al,80h
		stc
	andif_ eq				; if XMS service present
		mov	ax,4310h		; XMS: Get Driver Address
		int	2Fh
		saveFAR [XMSentry],es,bx
		clc
	end_
		ret
getXMSaddr	endp

;
; Save allocation strategy
;
; In:	none
; Out:	Carry flag		(no UMB link supported)
; Use:	none
; Modf:	AX, SaveMemStrat, SaveUMBLink
; Call:	INT 21/5800, INT 21/5802
;
SaveStrategy	proc
		mov	ax,5800h
		int	21h			; get DOS memalloc strategy
		mov	[SaveMemStrat],ax
		mov	ax,5802h
		int	21h			; get UMB link state
		mov	[SaveUMBLink],al
		ret
SaveStrategy	endp

;
; Restore allocation strategy
;
; In:	none
; Out:	none
; Use:	SaveMemStrat, SaveUMBLink
; Modf:	AX, BX
; Call:	INT 21/5801, INT 21/5803
;
RestoreStrategy	proc
		OPCODE_MOV_BX
SaveMemStrat	dw	?
		mov	ax,5801h
		int	21h			; set DOS memalloc strategy
		OPCODE_MOV_BX
SaveUMBLink	db	?,0
		mov	ax,5803h
		int	21h			; set UMB link state
		ret
RestoreStrategy	endp

;
; Allocate high memory
;
; In:	BX			(required memory size in para)
;	DS			(current memory segment)
; Out:	DX			(seg of new memory or DS)
; Use:	XMSentry
; Modf:	AX, ES
; Call:	INT 21/48, INT 21/49, INT 21/58,
;	SaveStrategy, RestoreStrategy, getXMSaddr
;
AllocUMB	proc
		push	bx
		testflag [options],OPT_noUMB
		jnz	@@allocasis		; jump if UMB prohibited
		mov	ax,ds
		cmp	ah,0A0h
		jae	@@allocasis		; jump if already loaded hi

;---------- check if UMB is a DOS type
		call	SaveStrategy
		mov	bx,1			; add UMB to MCB chain
		mov	ax,5803h
		int	21h			; set UMB link state

; try to set a best strategy to allocate DOS supported UMBs
		mov	bl,41h			; OPTIMIZE: BL instead BX
		mov	ax,5801h		; hi mem, best fit
		int	21h			; set alloc strategy

	if_ carry
; try a worse one then
		mov	bl,81h			; OPTIMIZE: BL instead BX
		mov	ax,5801h		; hi mem then low mem, best fit
		int	21h			; set alloc strategy
	end_

		pop	bx
		push	bx
		mov	ah,48h
		int	21h			; allocate UMB memory (size=BX)
		pushf
		xchg	dx,ax			; OPTIMIZE: instead MOV DX,AX
		call	RestoreStrategy		; restore allocation strategy
		popf
	if_ nc
		cmp	dh,0A0h			; exit if allocated mem is
		jae	@@allocret		;  is above 640k (segment
		DOSFreeMem dx			;  0A000h) else free it
	end_

;---------- try a XMS manager to allocate UMB
		call	getXMSaddr
	if_ nc
		pop	dx
		push	dx
		mov	ah,10h			; XMS: Request UMB (size=DX)
		call	[XMSentry]		; ...AX=1 -> BX=seg, DX=size
		dec	ax
	andif_ zero
		pop	ax
		push	ax
		cmp	bx,ax
		mov	dx,bx
		jae	@@allocret
		mov	ah,11h			; XMS: Release UMB (seg=DX)
		call	[XMSentry]
	end_

;---------- use current memory segment
@@allocasis:	mov	dx,ds

@@allocret:	pop	bx
		ret
AllocUMB	endp

;
; In:	ES			(segment to free)
; Out:	none
; Use:	XMSentry
; Modf:	AH, DX
; Call:	INT 21/49, getXMSaddr
;
FreeMem		proc
		assume	es:nothing
		call	getXMSaddr
	if_ nc
		mov	dx,es
		mov	ah,11h			; XMS: Release UMB
		db	09Ah			; CALL FAR dword
XMSentry	dd	?
	end_
		DOSFreeMem			; free allocated memory
		ret
FreeMem		endp

;

		end	start
