/*
   Direct.c - directory manipulation code.
   Copyright (C) 2000 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at: imre.leber@worldonline.be
*/

#include <time.h>
#include <alloc.h>
#include <string.h>

#include "..\header\rdwrsect.h"
#include "..\header\boot.h"
#include "..\header\bool.h"
#include "..\header\direct.h"
#include "..\header\fatconst.h"

#define DSY 1980                      /* DOS standard year. */

int GetDirectoryStart(RDWRHandle handle)
{
    int reserved, fats, SectorsPerFat;

    reserved = GetReservedSectors(handle);
    if (reserved == 0) return FALSE;

    fats = GetNumberOfFats(handle);
    if (fats == 0) return FALSE;

    SectorsPerFat = GetSectorsPerFat(handle);
    if (SectorsPerFat == 0) return FALSE;

    return reserved + (fats * SectorsPerFat);
}

int ReadDirEntry(RDWRHandle handle, int index,
		 struct DirectoryEntry* entry)
{
   struct DirectoryEntry* buf;

   if (handle->dirbegin == 0)
   {
      handle->dirbegin = GetDirectoryStart(handle);
      if (!handle->dirbegin) return FALSE;
   }

   buf = (struct DirectoryEntry*) malloc(BYTESPERSECTOR);
   if (buf == NULL) return FALSE;

   if (ReadSectors(handle, 1, handle->dirbegin + (index / ENTRIESPERSECTOR),
		   buf) == -1)
   {
      free(buf);
      return FALSE;
   }

   memcpy(entry, buf + (index % ENTRIESPERSECTOR),
	  sizeof(struct DirectoryEntry));

   free(buf);
   return TRUE;
}

int WriteDirEntry(RDWRHandle handle, int index,
		  struct DirectoryEntry* entry)
{
   int    result;
   struct DirectoryEntry* buf;

   if (handle->dirbegin == 0)
   {
      handle->dirbegin = GetDirectoryStart(handle);
      if (!handle->dirbegin) return FALSE;
   }

   buf = (struct DirectoryEntry*) malloc(BYTESPERSECTOR);
   if (buf == NULL) return FALSE;

   if (ReadSectors(handle, 1, handle->dirbegin + (index / 16), buf) == -1)
   {
      free(buf);
      return FALSE;
   }

   memcpy(buf + (index % 16), entry, 32);

   result = (WriteSectors(handle, 1, handle->dirbegin + (index / 16), buf)
			  == -1);

   free(buf);
   return result;
}

int GetRootDirPosition(RDWRHandle handle, int index,
		       struct DirectoryPosition* pos)
{
     int dirstart = GetDirectoryStart(handle);

     if (!dirstart) return FALSE;

     pos->sector = dirstart + (index / 16);
     pos->offset = index % 16;

     return TRUE;
}

int IsRootDirPosition(RDWRHandle handle, struct DirectoryPosition* pos)
{
    struct DirectoryPosition pos1;

    GetRootDirPosition(handle, GetNumberOfRootEntries(handle), &pos1);

    if ((pos->sector <= pos1.sector) && (pos->offset <= pos1.offset))
       return TRUE;
    else
       return FALSE;
}

void UnPackTimeDateStamp(struct tm* time, short timestamp, short datestamp)
{
     time->tm_sec  = timestamp &   0x1f;
     time->tm_min  = timestamp &  0x3e0;
     time->tm_hour = timestamp & 0x7c00;

     time->tm_mday = datestamp &   0x1f;
     time->tm_mon  = datestamp &  0x1e0;
     time->tm_year = datestamp & 0xfe00+DSY;         /* Incorect! */
}

void PackTimeDateStamp(struct tm* time, short* timestamp, short* datestamp)
{
     *timestamp  = time->tm_sec;
     *timestamp += time->tm_min  << 5;
     *timestamp += time->tm_hour << 11;

     *datestamp  = time->tm_mday;
     *datestamp += time->tm_mon          <<  5;
     *datestamp += ((time->tm_year)-DSY) << 10;
}

