#!/usr/bin/env python

import locale
import subprocess
from collections import defaultdict
locale.setlocale(locale.LC_ALL, 'en_US.UTF-8')

header = """
#
#   #####   #######           #     #     #     #     #     #
# ##     #  #     #          # #    #  #  #    # #     #   #
# ##        #     #         #   #   #  #  #   #   #     # #
# ##  ####  #     #        #     #  #  #  #  #     #     #
# ##     #  #     #        #######  #  #  #  #######     #      ###
# ##     #  #     #        #     #  #  #  #  #     #     #      ###
# # #####   #######        #     #   ## ##   #     #     #      ###
#
# Seriously. You shouldn't even be looking at this file unless you're
# debugging generate_mozbuild.py.
#
# DO NOT MODIFY THIS FILE IT IS AUTOGENERATED.
#

skia_opt_flags = []

if CONFIG['MOZ_OPTIMIZE']:
    if CONFIG['CC_TYPE'] in ('msvc', 'clang-cl'):
        skia_opt_flags += ['-O2']
    elif CONFIG['CC_TYPE'] in ('clang', 'gcc'):
        skia_opt_flags += ['-O3']

"""

footer = """

# We allow warnings for third-party code that can be updated from upstream.
AllowCompilerWarnings()

FINAL_LIBRARY = 'gkmedias'
LOCAL_INCLUDES += [
    'skia/include/c',
    'skia/include/codec',
    'skia/include/config',
    'skia/include/core',
    'skia/include/effects',
    'skia/include/encode',
    'skia/include/gpu',
    'skia/include/pathops',
    'skia/include/ports',
    'skia/include/private',
    'skia/include/utils',
    'skia/include/utils/mac',
    'skia/include/views',
    'skia/src/core',
    'skia/src/gpu',
    'skia/src/gpu/effects',
    'skia/src/gpu/gl',
    'skia/src/gpu/glsl',
    'skia/src/image',
    'skia/src/lazy',
    'skia/src/opts',
    'skia/src/sfnt',
    'skia/src/shaders',
    'skia/src/sksl',
    'skia/src/utils',
    'skia/src/utils/mac',
    'skia/src/utils/win',
]

if CONFIG['MOZ_WIDGET_TOOLKIT'] == 'windows':
    if CONFIG['CC_TYPE'] in ('gcc', 'clang'):
        DEFINES['SK_JUMPER_USE_ASSEMBLY'] = 0
    elif CONFIG['CPU_ARCH'] == 'x86':
        SOURCES['skia/src/jumper/SkJumper_generated_win.S'].flags += ['-safeseh']
    DEFINES['UNICODE'] = True
    DEFINES['_UNICODE'] = True
    UNIFIED_SOURCES += [
        'skia/src/fonts/SkFontMgr_indirect.cpp',
        'skia/src/fonts/SkRemotableFontMgr.cpp',
    ]

# Work around a crash when jumping into assembly on platforms where
# Clang has 4-byte stack alignment.
if CONFIG['CPU_ARCH'] == 'x86' and CONFIG['CC_TYPE'] == 'clang':
    SOURCES['skia/src/jumper/SkJumper.cpp'].flags += [
        '-mstack-alignment=16',
        '-mstackrealign'
    ]

# We should autogenerate these SSE related flags.

if CONFIG['INTEL_ARCHITECTURE'] and (CONFIG['CC_TYPE'] in ('clang', 'clang-cl', 'gcc')):
    SOURCES['skia/src/opts/SkBitmapProcState_opts_SSE2.cpp'].flags += CONFIG['SSE2_FLAGS']
    SOURCES['skia/src/opts/SkBitmapProcState_opts_SSSE3.cpp'].flags += ['-mssse3']
    SOURCES['skia/src/opts/SkBlitRow_opts_SSE2.cpp'].flags += CONFIG['SSE2_FLAGS']
    SOURCES['skia/src/opts/SkOpts_ssse3.cpp'].flags += ['-mssse3']
    SOURCES['skia/src/opts/SkOpts_sse41.cpp'].flags += ['-msse4.1']
    SOURCES['skia/src/opts/SkOpts_sse42.cpp'].flags += ['-msse4.2']
    SOURCES['skia/src/opts/SkOpts_avx.cpp'].flags += ['-mavx']
    SOURCES['skia/src/opts/SkOpts_hsw.cpp'].flags += ['-mavx2']
elif CONFIG['CC_TYPE'] in ('msvc', 'clang-cl'):
    # MSVC doesn't need special compiler flags, but Skia needs to be told that these files should
    # be built with the required SSE level or it will simply compile in stubs and cause runtime crashes
    SOURCES['skia/src/opts/SkBitmapProcState_opts_SSE2.cpp'].flags += ['-DSK_CPU_SSE_LEVEL=20']
    SOURCES['skia/src/opts/SkBitmapProcState_opts_SSSE3.cpp'].flags += ['-DSK_CPU_SSE_LEVEL=31']
    SOURCES['skia/src/opts/SkBlitRow_opts_SSE2.cpp'].flags += ['-DSK_CPU_SSE_LEVEL=20']
    SOURCES['skia/src/opts/SkOpts_ssse3.cpp'].flags += ['-DSK_CPU_SSE_LEVEL=31']
    SOURCES['skia/src/opts/SkOpts_sse41.cpp'].flags += ['-DSK_CPU_SSE_LEVEL=41']
    SOURCES['skia/src/opts/SkOpts_sse42.cpp'].flags += ['-DSK_CPU_SSE_LEVEL=42']
    SOURCES['skia/src/opts/SkOpts_avx.cpp'].flags += ['-DSK_CPU_SSE_LEVEL=51']
    SOURCES['skia/src/opts/SkOpts_hsw.cpp'].flags += ['-DSK_CPU_SSE_LEVEL=52']
elif CONFIG['CPU_ARCH'] == 'arm' and CONFIG['CC_TYPE'] in ('clang', 'gcc'):
    CXXFLAGS += CONFIG['NEON_FLAGS']
elif CONFIG['CPU_ARCH'] == 'aarch64' and CONFIG['CC_TYPE'] in ('clang', 'gcc'):
    SOURCES['skia/src/opts/SkOpts_crc32.cpp'].flags += ['-march=armv8-a+crc']

DEFINES['SKIA_IMPLEMENTATION'] = 1

if CONFIG['MOZ_ENABLE_SKIA_PDF_SFNTLY']:
    DEFINES['SK_PDF_USE_SFNTLY'] = 1

if not CONFIG['MOZ_ENABLE_SKIA_GPU']:
    DEFINES['SK_SUPPORT_GPU'] = 0

if CONFIG['MOZ_TREE_FREETYPE']:
    DEFINES['SK_CAN_USE_DLOPEN'] = 0

# Suppress warnings in third-party code.
if CONFIG['CC_TYPE'] in ('clang', 'clang-cl', 'gcc'):
    CXXFLAGS += [
        '-Wno-deprecated-declarations',
        '-Wno-overloaded-virtual',
        '-Wno-shadow',
        '-Wno-sign-compare',
        '-Wno-unreachable-code',
        '-Wno-unused-function',
    ]
if CONFIG['CC_TYPE'] == 'gcc':
    CXXFLAGS += [
        '-Wno-logical-op',
        '-Wno-maybe-uninitialized',
    ]
if CONFIG['CC_TYPE'] in ('clang', 'clang-cl'):
    CXXFLAGS += [
        '-Wno-implicit-fallthrough',
        '-Wno-inconsistent-missing-override',
        '-Wno-macro-redefined',
        '-Wno-unused-private-field',
    ]

if CONFIG['MOZ_WIDGET_TOOLKIT'] in ('gtk3', 'android'):
    CXXFLAGS += CONFIG['MOZ_CAIRO_CFLAGS']
    CXXFLAGS += CONFIG['CAIRO_FT_CFLAGS']

if CONFIG['MOZ_WIDGET_TOOLKIT'] == 'gtk3':
    CXXFLAGS += CONFIG['MOZ_PANGO_CFLAGS']

if CONFIG['MOZ_ENABLE_SKIA_PDF_SFNTLY']:
    LOCAL_INCLUDES += CONFIG['SFNTLY_INCLUDES']
"""

import json

platforms = ['linux', 'mac', 'android', 'win']

def parse_sources(output):
  return set(v.replace('//', 'skia/') for v in output.split() if v.endswith('.cpp') or v.endswith('.S'))

def generate_opt_sources():
  cpus = [('intel', 'x86', [':sse2', ':ssse3', ':sse41', ':sse42', ':avx', ':hsw']),
          ('arm', 'arm', [':armv7']),
          ('arm64', 'arm64', [':arm64', ':crc32']),
          ('none', 'none', [':none'])]

  opt_sources = {}
  for key, cpu, deps in cpus:
    subprocess.check_output('cd skia && bin/gn gen out/{0} --args=\'target_cpu="{1}"\''.format(key, cpu), shell=True)
    opt_sources[key] = set()
    for dep in deps:
        output = subprocess.check_output('cd skia && bin/gn desc out/{0} {1} sources'.format(key, dep), shell=True)
        if output:
            opt_sources[key].update(parse_sources(output))

  return opt_sources

def generate_platform_sources():
  sources = {}
  platform_args = {
    'win' : 'win_vc="C:/" win_sdk_version="00.0.00000.0"'
  }
  for plat in platforms:
    args = platform_args.get(plat, '')
    output = subprocess.check_output('cd skia && bin/gn gen out/{0} --args=\'target_os="{0}" {1}\' > /dev/null && bin/gn desc out/{0} :skia sources'.format(plat, args), shell=True)
    if output:
      sources[plat] = parse_sources(output)

  deps = {':effects' : 'common', ':gpu' : 'gpu', ':pdf' : 'pdf'}
  for dep, key in deps.items():
    output = subprocess.check_output('cd skia && bin/gn desc out/linux {} sources'.format(dep), shell=True)
    if output:
      sources[key] = parse_sources(output)

  return dict(sources.items() + generate_opt_sources().items())


def generate_separated_sources(platform_sources):
  blacklist = [
    'GrGLMakeNativeInterface',
    'GrGLCreateNullInterface',
    'GrGLAssembleInterface',
    'GrGLTestInterface',
    'skia/src/android/',
    'skia/src/atlastext/',
    'skia/src/c/',
    'skia/src/effects/Gr',
    'skia/src/effects/Sk',
    'skia/src/fonts/',
    'skia/src/jumper/SkJumper_generated_win.S',
    'skia/src/ports/SkImageEncoder',
    'skia/src/ports/SkImageGenerator',
    'skia/src/gpu/vk/',
    'SkBitmapRegion',
    'SkLite',
    'codec',
    'SkWGL',
    'SkMemory_malloc',
    'third_party',
    'SkCamera',
    'SkCanvasStack',
    'SkCanvasStateUtils',
    'SkCurveMeasure',
    'SkDeferredCanvas',
    'SkDumpCanvas',
    'SkFrontBufferedStream',
    'SkInterpolator',
    'SkMultiPictureDocument',
    'SkNullCanvas',
    'SkNWayCanvas',
    'SkOverdrawCanvas',
    'SkPaintFilterCanvas',
    'SkParseColor',
    'SkTextBox',
    'SkWhitelistTypefaces',
    'SkXPS',
    'SkCreateCGImageRef',
  ]

  def isblacklisted(value):
    for item in blacklist:
      if value.find(item) >= 0:
        return True

    return False

  separated = defaultdict(set, {
    'common': {
      'skia/src/core/SkBlurImageFilter.cpp',
      'skia/src/core/SkGpuBlurUtils.cpp',
      'skia/src/effects/GrCircleBlurFragmentProcessor.cpp',
      'skia/src/effects/SkBlurMask.cpp',
      'skia/src/effects/SkBlurMaskFilter.cpp',
      'skia/src/effects/SkDashPathEffect.cpp',
      'skia/src/effects/SkImageSource.cpp',
      'skia/src/gpu/gl/GrGLMakeNativeInterface_none.cpp',
      'skia/src/ports/SkDiscardableMemory_none.cpp',
      'skia/src/ports/SkMemory_mozalloc.cpp',
      'skia/src/ports/SkImageGenerator_none.cpp',
    },
    'android': {
      # 'skia/src/ports/SkDebug_android.cpp',
      'skia/src/ports/SkFontHost_cairo.cpp',
      # 'skia/src/ports/SkFontHost_FreeType.cpp',
      'skia/src/ports/SkFontHost_FreeType_common.cpp',
      # 'skia/src/ports/SkTime_Unix.cpp',
      # 'skia/src/utils/SkThreadUtils_pthread.cpp',
    },
    'linux': {
      'skia/src/ports/SkFontHost_cairo.cpp',
      'skia/src/ports/SkFontHost_FreeType_common.cpp',
    },
    'no-mingw': {
      'skia/src/jumper/SkJumper_generated_win.S',
    },
    'intel': set(),
    'arm': set(),
    'arm64': set(),
    'none': set(),
    'pdf': set(),
    'gpu': set()
  })

  for plat in platform_sources.keys():
    for value in platform_sources[plat]:
      if isblacklisted(value):
        continue

      if value in separated['common']:
        continue

      key = plat

      if all(value in platform_sources.get(p, {})
             for p in platforms if p != plat):
        key = 'common'

      separated[key].add(value)

  return separated

def uniq(seq):
  seen = set()
  seen_add = seen.add
  return [ x for x in seq if x not in seen and not seen_add(x)]

def write_cflags(f, values, subsearch, cflag, indent):
  def write_indent(indent):
    for _ in range(indent):
        f.write(' ')

  if isinstance(subsearch, str):
    subsearch = [ subsearch ]

  def iswhitelisted(value):
    for item in subsearch:
      if value.find(item) >= 0:
        return True

    return False

  val_list = uniq(sorted(values, key=lambda x: x.lower()))

  if len(val_list) == 0:
    return

  for val in val_list:
    if iswhitelisted(val):
      write_indent(indent)
      f.write("SOURCES[\'" + val + "\'].flags += " + cflag + "\n")

opt_whitelist = [
  'skia/src/opts/Sk',
  'SkOpts',
  'SkBitmapProcState',
  'SkBlitMask',
  'SkBlitRow',
  'SkBlitter',
  'SkJumper',
  'SkSpriteBlitter',
  'SkMatrix.cpp',
]

# Unfortunately for now the gpu and pathops directories are
# non-unifiable. Keep track of this and fix it.
unified_blacklist = [
  'FontHost',
  'SkAdvancedTypefaceMetrics',
  'SkBitmapProcState_matrixProcs.cpp',
  'SkBlitter_A8.cpp',
  'SkBlitter_ARGB32.cpp',
  'SkBlitter_RGB16.cpp',
  'SkBlitter_Sprite.cpp',
  'SkScan_Antihair.cpp',
  'SkScan_AntiPath.cpp',
  'SkScan_DAAPath.cpp',
  'SkParse.cpp',
  'SkPDFFont.cpp',
  'SkPictureData.cpp',
  'skia/src/gpu/effects/',
  'GrResourceCache',
  'GrResourceProvider',
  'GrAA',
  'GrGL',
  'GrCCPathProcessor',
  'GrMSAAPathRenderer.cpp',
  'GrNonAAFillRect',
  'GrPathUtils',
  'GrShadowRRectOp',
  'SkColorSpace',
  'SkImage_Gpu.cpp',
  'SkPathOpsDebug.cpp',
  'SkParsePath.cpp',
  'SkRecorder.cpp',
  'SkMiniRecorder.cpp',
  'SkXfermode',
  'SkMatrix44.cpp',
  'SkRTree.cpp',
  'SkVertices.cpp',
  'SkJumper',
  'SkSLLexer.cpp',
  'SkSLLayoutLexer.cpp',
  'SkThreadedBMPDevice.cpp',
] + opt_whitelist

def write_sources(f, values, indent):
  def isblacklisted(value):
    for item in unified_blacklist:
      if value.find(item) >= 0:
        return True

    return False

  sources = {}
  sources['nonunified'] = set()
  sources['unified'] = set()

  for item in values:
    if isblacklisted(item):
      sources['nonunified'].add(item)
    else:
      sources['unified'].add(item)

  write_list(f, "UNIFIED_SOURCES", sources['unified'], indent)
  write_list(f, "SOURCES", sources['nonunified'], indent)
  
def write_list(f, name, values, indent):
  def write_indent(indent):
    for _ in range(indent):
        f.write(' ')

  val_list = uniq(sorted(values, key=lambda x: x.lower()))

  if len(val_list) == 0:
    return

  write_indent(indent)
  f.write(name + ' += [\n')
  for val in val_list:
    write_indent(indent + 4)
    f.write('\'' + val + '\',\n')

  write_indent(indent)
  f.write(']\n')

def write_mozbuild(sources):
  filename = 'moz.build'
  f = open(filename, 'w')

  f.write(header)

  write_sources(f, sources['common'], 0)
  write_cflags(f, sources['common'], opt_whitelist, 'skia_opt_flags', 0)

  f.write("if CONFIG['MOZ_ENABLE_SKIA_PDF']:\n")
  write_sources(f, sources['pdf'], 4)

  f.write("if CONFIG['MOZ_ENABLE_SKIA_GPU']:\n")
  write_sources(f, sources['gpu'], 4)

  f.write("if CONFIG['MOZ_WIDGET_TOOLKIT'] == 'android':\n")
  write_sources(f, sources['android'], 4)

  f.write("if CONFIG['MOZ_WIDGET_TOOLKIT'] in ('cocoa', 'uikit'):\n")
  write_sources(f, sources['mac'], 4)

  f.write("if CONFIG['MOZ_WIDGET_TOOLKIT'] == 'gtk3':\n")
  write_sources(f, sources['linux'], 4)

  f.write("if CONFIG['MOZ_WIDGET_TOOLKIT'] == 'windows':\n")
  f.write("    if CONFIG['CC_TYPE'] not in ('gcc', 'clang'):\n")
  write_list(f, "SOURCES", sources['no-mingw'], 8)
  # Windows-specific files don't get unification because of nasty headers.
  # Luckily there are not many files in this.
  write_list(f, "SOURCES", sources['win'], 4)

  f.write("if CONFIG['INTEL_ARCHITECTURE']:\n")
  write_sources(f, sources['intel'], 4)
  write_cflags(f, sources['intel'], opt_whitelist, 'skia_opt_flags', 4)

  f.write("elif CONFIG['CPU_ARCH'] == 'arm' and CONFIG['CC_TYPE'] in ('clang', 'gcc'):\n")
  write_sources(f, sources['arm'], 4)
  write_cflags(f, sources['arm'], opt_whitelist, 'skia_opt_flags', 4)

  f.write("elif CONFIG['CPU_ARCH'] == 'aarch64' and CONFIG['CC_TYPE'] in ('clang', 'gcc'):\n")
  write_sources(f, sources['arm64'], 4)
  write_cflags(f, sources['arm64'], opt_whitelist, 'skia_opt_flags', 4)

  f.write("else:\n")
  write_sources(f, sources['none'], 4)

  f.write(footer)

  f.close()

  print 'Wrote ' + filename

def main():
  platform_sources = generate_platform_sources()
  separated_sources = generate_separated_sources(platform_sources)
  write_mozbuild(separated_sources)


if __name__ == '__main__':
  main()
